#!/bin/bash

# name		:	x2sh
# version	:	0.8a1 (FEB1206) - prototype
# author	:	George Makrydakis <gmakmail@gmail.com>
# license	:	GPL v 2.x or up
#
# purpose	:	x2sh (formerly posted as bparser) is an "xml - to - shell" script converter.
#			It includes a series of pure bash functions eliminating any dependency to
#			a third party binary or a subshell call within bash. bparser was originally
#			posted to the aLFS mailing list for reviewing.
#
# technote	:	x2sh is a script using O(n) and O(n^2) algorithms at times. The reason for this
#			has to do with the fact that more complicated and more efficient algorithms would
#			exceed bash shell abilities requiring increasingly complex structures.  A series of
#			"hacks" will be used in future editions to make this O(n) footprint less apparent.
#			For the	number of XML files to be processed, knowing the structure of the LFS xml
#			sources and the level of simplicity we wish to achieve, a pure - bash solution with no
#			external dependencies may prove a targeted compromise. x2sh should be launched in
#			the LFS book xml sources root, there are hardcoded portions that require that.
#
#			Hardcoded portions will be eventually removed. This is a PROTOTYPE!

# usage		:	take patches.ent, general.ent and glibc.xml and put them in the script's working directory, then run it!

	#shopt  -s -o nounset
	
	# PART 1: declaration of globally available variables.
	
	declare -a arrayDB
	declare -a preloader
	declare -a fifostack=(0);
	declare -a resultsDB=(0);
	declare -a entityDB
	declare  entityDBindex=8

	declare aDBtotal_FIELDS
	declare aDBtotal_RECORDS
	declare aDBtotal_POINTER
	
	declare -r startTAG='<'
	declare -r closeTAG='>'
	declare -r slashTAG='/'
	declare -r equalTAG='='
	declare -r quoteTAG='"'
	declare -r whiteTAG=' '
	
	declare elmentATTN=""
	declare elmentATTV=""
	declare elmentNAME=""
	declare elmentDUMP=""
	declare parserLINE=""
	declare parserFILE=""
	declare parserBUFF=""
	declare parserFLAG="ENABLED"
	
	declare -i checkpoint=0
	declare -r myfile="general.ent"
	declare entityTOKEN=""
	declare commentFLAG="OFF"
	
	entityDB[0]="lt"
	entityDB[1]='<'
	entityDB[2]="gt"
	entityDB[3]=">"
	entityDB[4]="amp"
	entityDB[5]="&"
	entityDB[6]="quot"
	entityDB[7]='"'
	# PREAMBLE" helper functions
	
	function pushARRAY ()
	{
		fifostack=($1 "${fifostack[@]}");
		return
	}

	function shiftARRAY ()
	{
		fifostack=(${fifostack[@]:1});
		return
	}
	
	# PART 2: arrayDB function declarations.
	
	function x2sh_arrayDBdeclare ()
	{
		# description	: declares the headers of the DB
		# usage			: x2sh_arrayDBdeclare FIELD1 FIELD2 FIELD3 FIELD4...
		
		declare addfieldname; declare aindex
		for addfieldname in $@
		do
			arrayDB[aindex]=$addfieldname
			let "aindex+=1"
		done
		aDBtotal_FIELDS=${#arrayDB[@]}
		aDBtotal_RECORDS=0
		aDBtotal_POINTER=$aDBtotal_FIELDS
		return
	}

	function x2sh_arrayDBaddrecord ()
	{
		# description	: adds a record to arrayDB
		# usage			: x2sh_arrayDBaddrecord FIELD1 FIELD2 FIELD3 FIELD4...
		
		declare add_record_entry
		for add_record_entry in $@
		do
			arrayDB[$aDBtotal_POINTER]="$add_record_entry"
			let "aDBtotal_POINTER+=1"
		done
			let "aDBtotal_RECORDS+=1"
		return	
	}

	function x2sh_arrayDBaddDATA ()
	{
		# description	: adds element "data" to arrayDB
		# usage			: x2sh_arrayDBaddDATA
		
		case $commentFLAG in
			ON)
				arrayDB[$aDBtotal_POINTER]="COMMENT"; let "aDBtotal_POINTER+=1"
				;;
			OFF)
				arrayDB[$aDBtotal_POINTER]="DATA"; let "aDBtotal_POINTER+=1"
				;;
		esac
			arrayDB[$aDBtotal_POINTER]="${fifostack[0]}"; let "aDBtotal_POINTER+=1"
			arrayDB[$aDBtotal_POINTER]="$parserBUFF"; let "aDBtotal_POINTER+=1"
		return
	}

	function x2sh_arrayDBaddATTRIBUTE ()
	{
		# description	: adds attribute "data" to arrayDB
		# usage			: x2sh_arrayDBaddATTRIBUTE
		
		arrayDB[$aDBtotal_POINTER]="ATTRIBUTE"; let "aDBtotal_POINTER+=1"
		arrayDB[$aDBtotal_POINTER]="$elmentATTN"; let "aDBtotal_POINTER+=1"
		arrayDB[$aDBtotal_POINTER]="$elmentATTV"; let "aDBtotal_POINTER+=1"
		return
	}
	
	function x2sh_arrayDBprintRECORD ()
	{
		# description	: prints out results from a x2sh_arrayDB
		# usage			: x2sh_arrayDBprintRECORD
		declare rowstuff
		for rowstuff in ${resultsDB[@]}
		do
		echo ${arrayDB[(($rowstuff * $aDBtotal_FIELDS))]} \
			 ${arrayDB[(($rowstuff * $aDBtotal_FIELDS + 1))]} \
			 ${arrayDB[(($rowstuff * $aDBtotal_FIELDS + 2))]}
		done
		
	}
	
	function x2sh_arrayDBprintRECORDvalues ()
	{
		# description	: prints out results from a x2sh_arrayDB
		# usage			: x2sh_arrayDBprintRECORDvalues
		
		declare rowstuff
		for rowstuff in ${resultsDB[@]}
		do
		echo ${arrayDB[(($rowstuff * $aDBtotal_FIELDS))]} \
			 ${arrayDB[(($rowstuff * $aDBtotal_FIELDS + 1))]} \
			 ${arrayDB[(($rowstuff * $aDBtotal_FIELDS + 2))]}
		done
		
	}
	
	function x2sh_arrayDBsearchDATA ()
	{
		# description	: searches entries in the arrayDB that in FIELD1 and FIELD2 have specific values
		# usage			: x2sh_arrayDBsearchDATA FIELD1 VALUE1  FIELD2 VALUE2
		
		declare queryinfield1="$1"
		declare queryinvalue1="$2"
		declare queryinfield2="$3"
		declare queryinvalue2="$4"
		declare rowindex=0
		declare colindex1=0
		declare colindex2=0
		declare results=0
		resultsDB=();

		for ((colindex1=0; colindex1 < $aDBtotal_FIELDS; colindex1++))
		do
			if [ "$queryinfield1" = "${arrayDB["$colindex1"]}" ] ; then
				break
			fi	
		done
		for ((colindex2=0; colindex2 < $aDBtotal_FIELDS; colindex2++))
		do
			if [ "$queryinfield2" = "${arrayDB["$colindex2"]}" ] ; then
				break
			fi	
		done

		# at this point we have both colindex1 and colindex2

		for ((rowindex=1; rowindex < $aDBtotal_RECORDS + 1; rowindex++))
		do
			if [ "$queryinvalue1" = "${arrayDB[(($rowindex * $aDBtotal_FIELDS + $colindex1))]}" ] && \
			   [ "$queryinvalue2" = "${arrayDB[(($rowindex * $aDBtotal_FIELDS + $colindex2))]}" ]	; then
				resultsDB[results]=$rowindex
				let "results+=1"
			fi
	       	done
	}
	
	# PART 3: entityDB function declarations.

	function x2sh_entityDBhousekeep ()
	{
		# with all entries being in entityDB, we have to dereference crossed references!
		# warning: O(n^2) algo...

		for ((jumpstart=0 ; jumpstart < ${#entityDB[@]}; jumpstart+=2));
		do
			entityMATCH="&"${entityDB[jumpstart]}";"
			for ((refitem=1 ; refitem < ${#entityDB[@]}; refitem+=2));
			do
				temporary="${entityDB[refitem]}"
				entityDB[refitem]="${temporary//$entityMATCH/${entityDB[((jumpstart + 1))]}}"
			done
		done

	}
	
	function x2sh_entityDBlookup ()
	{
		# description: dereferences a single entity

		declare lookupENTITY="$1"
		declare lookupCOUNTR=0
		for ((lookupCOUNTR=0; lookupCOUNTR < ${#entityDB[@]}; lookupCOUNTR+=2));
		do
			case ${entityDB[lookupCOUNTR]} in
			
			$lookupENTITY)
			echo ${entityDB[((lookupCOUNTR + 1))]}
			;;
			*);;
			esac
		done
	}
	
	function x2sh_entityDBaddDATA ()
	{
		# description	: adds entity "data" to entityDB
		# usage			: x2sh_entityDBaddDATA
		
		until [ "${parserLINE:$checkpoint:1}" = '>' ] ;
		do
			case ${parserLINE:$checkpoint:1} in
			$whiteTAG)
				let "checkpoint++"
				entityTOKEN=""
				until [ "${parserLINE:$checkpoint:1}" = '"' ] ;	
				do
					entityTOKEN=$entityTOKEN${parserLINE:$checkpoint:1}
					let "checkpoint++"
					case ${parserLINE:$checkpoint:1} in
					$whiteTAG)
						case $entityTOKEN in
						'%')
							echo a PARAMETER is defined >> /dev/null
							entityTOKEN=""
							return
							;;
						*)
							if [ $entityTOKEN = "SYSTEM" ] ; then
							echo parameter has SYSTEM keyword >> /dev/null
							return
							elif [ $entityTOKEN = "PUBLIC" ] ; then
							echo parameter has PUBLIC keyword >> /dev/null
							return
							else
							entityDB[entityDBindex]="$entityTOKEN"
							let "entityDBindex++"
							fi
							entityTOKEN=""
								;;
							esac
								;;
						esac
				done
				let "checkpoint-=1"
				;;
			$quoteTAG)
				let "checkpoint++"
				entityVAL=""
				until [ "${parserLINE:$checkpoint:1}" = '"' ] ;
				do
					entityVAL=$entityVAL${parserLINE:$checkpoint:1}
					let "checkpoint++"
				done
					entityDB[entityDBindex]="$entityVAL"
					let "entityDBindex++"
				;;
			esac
			let "checkpoint++"
		done
		return
	}
	
	# PART 4: the XML "parsing" module
	
	function x2shPARSER_loadfile ()
	{
		# status:	operational
		# info:		This is the main XML "parsing module" for the x2sh script.
		declare x2shFILE="$1"
		while read parserLINE
		do
		parserBUFF=""
		for ((checkpoint=0; checkpoint < ${#parserLINE}; checkpoint++)) ;
		do
			case ${parserLINE:$checkpoint:1} in
			$startTAG)
				let "checkpoint++"; elmentNAME=""; parserFLAG="ENABLED"
				if [ "$parserBUFF" != '' ] ; then 
						x2sh_arrayDBaddDATA
						parserBUFF=""
				fi
				until [ "${parserLINE:$checkpoint:1}" = ' ' ] || \
					  [ "${parserLINE:$checkpoint:1}" = '>' ] || \
					  [ $checkpoint = ${#parserLINE} ];
				do
					elmentNAME=$elmentNAME${parserLINE:$checkpoint:1}
					let "checkpoint++"
				done
				case $elmentNAME in
					!ENTITY) x2sh_entityDBaddDATA ;;
					!ATTLIST);;
					!DOCTYPE);;
					!--)
						commentFLAG="ON"
						x2sh_arrayDBaddrecord "COMMENT"  "${elmentNAME#*/}" "STARTING"
					;;
					*)
						if [ "${elmentNAME:0:1}" = "$slashTAG" ] ; then
							x2sh_arrayDBaddrecord "ELEMENT" "CLOSING" "${elmentNAME#*/}"
							shiftARRAY
							parserBUFF=""
						else
							pushARRAY $elmentNAME; parserBUFF=""
							x2sh_arrayDBaddrecord "ELEMENT" "STARTING" "$elmentNAME"
							case ${parserLINE:$checkpoint:1} in
								$whiteTAG)	elmentDUMP="$elmentNAME" ;;
								'')		elmentDUMP="$elmentNAME" ;;
							esac
						fi
					;;
				esac
			;;
			$closeTAG)
				if [ "$commentFLAG" = "ON" ] ; then
					x2sh_arrayDBaddrecord "COMMENT"  "${elmentNAME#*/}" "CLOSED"
					commentFLAG="OFF"
				fi
			;;
			$slashTAG)
				case $parserFLAG in
				ENABLED)
					if [ "${parserLINE:$((checkpoint + 1)):1}" = "$closeTAG" ] ; then
						x2sh_arrayDBaddrecord "ELEMENT" "CLOSING" "$elmentDUMP"
						shiftARRAY
						elmentDUMP=""; parserBUFF=""; let "checkpoint+=2"
					fi
				;;
				DISABLD)
				;;
				esac
			;;
			$quoteTAG)
				case $parserFLAG in
					ENABLED)
						let "checkpoint++"; elmentATTV=""; parserBUFF=""
						until [ "${parserLINE:$checkpoint:1}" = '"' ] ;
							do
								elmentATTV=$elmentATTV${parserLINE:$checkpoint:1}
								let "checkpoint++"
							done
						let "checkpoint++";
						x2sh_arrayDBaddATTRIBUTE
						case ${parserLINE:$checkpoint:1} in
							$closeTAG) if [ "${parserLINE:$((checkpoint + 1)):1}" = '<' ] ; then let "checkpoint-=1"; fi ;;
							$slashTAG) let "checkpoint-=1" ;;
						esac
					;;
					DISABLD)
					;;
				esac		
			;;
			$equalTAG)
				case $parserFLAG in
					ENABLED)
						elmentATTN=""
						if [ ${parserLINE:$((checkpoint + 1)):1} = '"' ] ; then
							elmentATTN=${parserBUFF##$equalTAG}
							elmentATTN=${elmentATTN##*$whiteTAG}
						fi
					;;
					DISABLD)
					;;
				esac
			;;
			esac
			case $((checkpoint + 1)) in
				${#parserLINE})
					parserBUFF=$parserBUFF${parserLINE:$checkpoint:1}
					if [ "${parserLINE:$checkpoint:1}" != "$closeTAG" ] ; then
						x2sh_arrayDBaddDATA
					fi
					;;
					*)
						case ${parserLINE:$checkpoint:1} in
							$closeTAG)	parserBUFF=""; parserFLAG="DISABLD";;
							*)		parserBUFF=$parserBUFF${parserLINE:$checkpoint:1};;
						esac
					;;
			esac
		done
	done < "$x2shFILE"
	return
	}
	
	function x2sh_cleanscript ()
	{
		# with all entries being in entityDB, we have to dereference crossed references!
		# warning: O(n^2) algo...
		
			thisgoes="$1"
			entityMATCH="&"$thisgoes";"

			for ((refitem=0 ; refitem < ${#preloader[@]}; refitem++));
			do
				temporary="${preloader[refitem]}"
				preloader[refitem]="${temporary//$entityMATCH/$2}"
			done
	}
	
	function x2shPARSER_makescript ()
	{
		# description	: dumps xml to a script
		# usage			: x2shPARSER_makescript
		
		declare queryinfield1="DATAMODE"
		declare queryinvalue1="DATA"
		declare queryinfield2="DATATYPE"
		declare queryinvalue2="userinput"
		declare queryinvalue3="literal"
		declare rowindex=0
		declare colindex1=0
		declare colindex2=0

		for ((colindex1=0; colindex1 < $aDBtotal_FIELDS; colindex1++))
		do
			if [ "$queryinfield1" = "${arrayDB["$colindex1"]}" ] ; then
				break
			fi	
		done
		for ((colindex2=0; colindex2 < $aDBtotal_FIELDS; colindex2++))
		do
			if [ "$queryinfield2" = "${arrayDB["$colindex2"]}" ] ; then
				break
			fi	
		done

		# at this point we have both colindex1 and colindex2

		for ((rowindex=1; rowindex < $aDBtotal_RECORDS + 1; rowindex++))
		do
			if [ "$queryinvalue1" = "${arrayDB[(($rowindex * $aDBtotal_FIELDS + $colindex1))]}" ] && \
			   ( [ "$queryinvalue2" = "${arrayDB[(($rowindex * $aDBtotal_FIELDS + $colindex2))]}" ]	|| \
			   	 [ "$queryinvalue3" = "${arrayDB[(($rowindex * $aDBtotal_FIELDS + $colindex2))]}" ] ) ; then
				preloader[counterwork]="${arrayDB[(($rowindex * $aDBtotal_FIELDS + 2))]}"
				
				let "counterwork++"
			fi
		done

	x2sh_cleanscript gt `x2sh_entityDBlookup gt`
	x2sh_cleanscript lt `x2sh_entityDBlookup lt`
	x2sh_cleanscript quot `x2sh_entityDBlookup quot`
	x2sh_cleanscript amp `x2sh_entityDBlookup amp`
	   
	}
	
	##### HARDCODED PART TESTING FOR glibc.xml
	
	#################################### preparatory phase
	x2sh_arrayDBdeclare "DATAMODE" "DATATYPE" "DATAWORD"
	x2shPARSER_loadfile "general.ent"	# load ENTITY declarations from general.ent
	x2shPARSER_loadfile "patches.ent"	# load ENTITY declarations from patches.ent
	x2sh_entityDBhousekeep			# dereference cross-referencing in the entityDB table
	#################################### the above part has to be executed once
	
	x2shPARSER_loadfile "glibc.xml"
	x2shPARSER_makescript

	x2sh_cleanscript glibc-version `x2sh_entityDBlookup glibc-version`
	x2sh_cleanscript glibc-rtld-patch `x2sh_entityDBlookup glibc-rtld-patch`
	x2sh_cleanscript glibc-testfix-patch `x2sh_entityDBlookup glibc-testfix-patch`
	x2sh_cleanscript glibc-tls_assert-patch `x2sh_entityDBlookup glibc-tls_assert-patch`

	for ((displayvariable=0; displayvariable < ${#preloader[@]}; displayvariable++));
	do
		echo ${preloader[displayvariable]}
	done
	exit	
