/*
 *  Copyright (c) 2002 Martin Soto
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *                   Creative Labs, Inc.
 *
 *  Use and distribute under the terms of the GNU GPL Version 2
 */


/* Be careful before changing the includes.  The order is very
   important. */

/* Standard C library */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Kernel includes in the standard C library.  Any recent version of
   the kernel include files will do. */
#include <asm/bitops.h>

/* The local definitions for libasound, taken from the source tarball.
   They are necessary if we want to include emu10k1.h (see below).  */
#include <local.h>

/* Standard libasound header file. */
#include <alsa/asoundlib.h>

/* Emu10k1 definitions from ALSA. */
#include <sound/emu10k1.h>


#define FXGPREGBASE		0x100		/* FX general purpose
                                                 * registers base  */


static void snd_emu10k1_write_op( emu10k1_fx8010_code_t *icode,
                                  unsigned int *ptr,
                                  unsigned int op,
                                  unsigned int r,
                                  unsigned int a,
                                  unsigned int x,
                                  unsigned int y )
{
    set_bit( *ptr, &icode->code_valid );
    icode->code[*ptr    ][0] = ((x & 0x3ff) << 10) | (y & 0x3ff);
    icode->code[(*ptr)++][1] =
        ((op & 0x0f) << 20) | ((r & 0x3ff) << 10) | (a & 0x3ff);
}

#define OP( icode, ptr, op, r, a, x, y ) \
	snd_emu10k1_write_op( icode, ptr, op, r, a, x, y )


static void init_stereo_control( emu10k1_fx8010_control_gpr_t *ctl,
                                 const char *name,
                                 int gpr,
                                 int defval )
{
    ctl->id.iface = SNDRV_CTL_ELEM_IFACE_MIXER;
    strcpy( ctl->id.name, name );

    ctl->vcount = ctl->count = 2;

    ctl->gpr[0] = gpr + 0;
    ctl->value[0] = defval;

    ctl->gpr[1] = gpr + 1;
    ctl->value[1] = defval;

    ctl->min = 0;
    ctl->max = 100;
    ctl->translation = EMU10K1_GPR_TRANSLATION_TABLE100;
}


int main( int argc, char *argv[] )
{
    char *dev_name;
    snd_hwdep_t *hwdep;
    int ret;
    unsigned i;
    unsigned int ptr, reg, ctrl;

    emu10k1_fx8010_code_t old_code, new_code;
    emu10k1_fx8010_control_gpr_t controls[256];
    snd_ctl_elem_id_t del_controls[255];
    emu10k1_fx8010_control_gpr_t add_controls[64];

    if ( argc < 2 ) {
        fprintf( stderr, "Insufficient arguments\n" );
        return -1;
    }
    dev_name = argv[1];

    ret = snd_hwdep_open( &hwdep, dev_name, 0 );
    if ( ret < 0 ) {
        fprintf( stderr, "%s: %s\n", dev_name, snd_strerror( ret ) );
        return -1;
    }

    old_code.gpr_list_controls = controls;
    old_code.gpr_list_control_count = 256;
    ret = snd_hwdep_ioctl( hwdep, SNDRV_EMU10K1_IOCTL_CODE_PEEK, &old_code );
    if ( ret < 0 ) {
        fprintf( stderr, "SNDRV_EMU10K1_IOCTL_CODE_PEEK: %s\n",
                 snd_strerror( ret ) );
        return -1;
    }

    memset( &new_code, 0, sizeof new_code );

    strcpy( new_code.name, "Dummy EMU10k1 code - Martin Soto" );

    for ( i = 0; i < old_code.gpr_list_control_total; i++ ) {
        del_controls[i] = old_code.gpr_list_controls[i].id;
    }
    new_code.gpr_del_control_count = old_code.gpr_list_control_total;
    new_code.gpr_del_controls = del_controls;

    ptr = 0;
    reg = 64;
    ctrl = 0;

    /* Put yout program here (see emu10k1.h in ALSA for the definition
       of the OP macro. */

    OP( &new_code, &ptr, iMACINT0, GPR(0), C_00000000,
        FXBUS(FXBUS_PCM_LEFT), C_00000004 );
    OP( &new_code, &ptr, iMACINT0, GPR(1), C_00000000,
        FXBUS(FXBUS_PCM_RIGHT), C_00000004 );

    OP( &new_code, &ptr, iMAC0,
        EXTOUT(EXTOUT_TOSLINK_L), C_00000000,
        GPR(0), GPR(reg) );
    OP( &new_code, &ptr, iMAC0,
        EXTOUT(EXTOUT_TOSLINK_R), C_00000000,
        GPR(1), GPR(reg + 1) );
    init_stereo_control( add_controls + ctrl, "SP/DIF Master", reg, 0 );
    ctrl++;

    new_code.gpr_add_control_count = ctrl;
    new_code.gpr_add_controls = add_controls;

    /* Fill the rest of the instruction memory with no-ops. */
    while ( ptr < 0x200 ) {
        OP( &new_code, &ptr, iACC3, C_00000000, C_00000000, C_00000000,
            C_00000000 );
    }

    ret = snd_hwdep_ioctl( hwdep, SNDRV_EMU10K1_IOCTL_CODE_POKE, &new_code );
    if ( ret < 0 ) {
        fprintf( stderr, "SNDRV_EMU10K1_IOCTL_CODE_POKE: %s\n",
                 snd_strerror( ret ) );
        return -1;
    }    

    return 0;
}
