/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.taskdefs.optional;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;

import org.apache.tools.ant.taskdefs.MatchingTask;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.types.TextElement;

import org.apache.oro.text.regex.PatternCompiler;
import org.apache.oro.text.regex.Perl5Compiler;
import org.apache.oro.text.regex.PatternMatcher;
import org.apache.oro.text.regex.Perl5Matcher;
import org.apache.oro.text.regex.Substitution;
import org.apache.oro.text.regex.Perl5Substitution;
import org.apache.oro.text.regex.Pattern;
import org.apache.oro.text.regex.Perl5Pattern;
import org.apache.oro.text.regex.Util;


/**
 * A {@link MatchingTask} that peforms a regular expression 
 * substitution on a file or set of files.
 *
 * @author Rodney Waldhoff (<a href="mailto:rwaldhof@us.britannica.com">rwaldhof@us.britannica.com</a>)
 */
public class REReplace extends MatchingTask {
   /** A {@link PatternCompiler} for building my regular expressions. */
   protected final PatternCompiler _patcompiler = new Perl5Compiler(); // could/should be static?

   /** A {@link PatternMatcher} for finding regular expressions in strings. */
   protected final PatternMatcher _patmatcher = new Perl5Matcher();  // could/should be static

   /**
    * A regular expression that matches on newline
    * characters (<code>\n</code>).
    * @see #processFile
    */
   protected final Pattern _newlinePattern;
   // static block initializing _newlinePattern
   {
      Pattern x = null;
      try {
         x = _patcompiler.compile("\n");
      } catch(Exception e) {
         System.err.println("Panic!  Unable to compile findNewline pattern in REReplace. This should never happen");
      } finally {
         _newlinePattern = x;
      }
   }

   /**
    * Flag indicating whether to quietly ignore or throw a 
    * {@link BuildException} when a file that is not 
    * writeable is encountered.
    *
    * @see #execute
    * @see #setSkipLocked
    */
   protected boolean _skipWhenLocked = true;

   /** 
    * The source file to do the replacing in.
    * <p>
    * At least one of {@link #src} and {@link #dir}
    * should be non-<code>null</code> when {@link #execute}
    * is invoked.
    * </p>
    * @see #dir
    * @see #setFile
    */
   protected File _src = null;
   
   /** 
    * The source directory to do the replacing in.
    * <p>
    * At least one of {@link #_src} and {@link #dir}
    * should be non-<code>null</code> when {@link #execute}
    * is invoked.
    * </p>
    * @see #_src
    * @see #setDir
    */
   protected File _dir = null;

   /**
    * The expression to find.
    *
    * @see #createFind
    * @see #setFind
    * @see #processFile
    */
   protected TextElement _find = null;
   
   /**
    * The expression to replace the {@link #_find found expression}.    
    *
    * @see #createReplaceWith
    * @see #setReplaceWith
    * @see #processFile
    */
   protected TextElement _replacewith = new TextElement();

   /**
    * My Ant action method.
    */
   public void execute() throws BuildException {
      // validate inputs
      if(null == _find) {
         throw new BuildException("REReplace: token must not be null", location);
      }
      if(null == _replacewith) {
         throw new BuildException("REReplace: value must not be null", location);
      }
      if(null == _src && null == _dir) {
         throw new BuildException("REReplace: Either the file or the dir attribute must be specified", location);
      }

      // log it
      log("Replacing the pattern \"" + _find + "\" with \"" + _replacewith + "\".");

      // process a single file, if any
      if(null != _src) {
         if(_src.canRead() && _src.canWrite()) {
            processFile(_src);
         } else {
            if(!_skipWhenLocked) {
               throw new BuildException("REReplace: unable either read or write to the source file \"" + _src.getAbsolutePath() + ".");
            } else {
               log("Skipping \"" + _src.getAbsolutePath() + "\" because it is not writeable.");
            }
         }
      }
      // process a set of files, if any
      if(null != _dir) {
         DirectoryScanner ds = super.getDirectoryScanner(_dir);
         String[] srcs = ds.getIncludedFiles();
         for(int i=0; i<srcs.length; i++) {
            File file = new File(_dir,srcs[i]); 
            if(file.canRead() && file.canWrite()) {
               processFile(file);
            } else {
               if(!_skipWhenLocked) {
                  throw new BuildException("REReplace: unable either read or write to the source file \"" + file.getAbsolutePath() + ".");
               } else {
                  log("Skipping \"" + file.getAbsolutePath() + "\" because it is not writeable.");
               }
            }
         }
      }
   }

   /**
    * Perform the REReplace action on the given file.
    */
   protected void processFile(File src) throws BuildException {
      if(!src.exists()) {
         throw new BuildException("REReplace: source file " + src.getAbsolutePath() + " doesn't exist", location);
      }

      File temp = null;
      boolean deletetemp = true;
      BufferedReader br = null;
      BufferedWriter bw = null;

      try {         
         // create the temp file to write to
         try {
            temp = File.createTempFile("rerep",".tmp");
         } catch(Exception e) {
            throw new BuildException("REReplace: Unable to create temp file.",e,location);
         }
         
         // create the steams to read from/write to
         br = new BufferedReader(new FileReader(src));
         bw = new BufferedWriter(new FileWriter(temp));

         // read the entire file into a char[]
         int fileLength = (int)(src.length());
         char[] tmpBuf = new char[fileLength];
         int numread = 0;
         int totread = 0;
         while(numread != -1 && totread < fileLength) {
            numread = br.read(tmpBuf,totread,fileLength);
            totread += numread;
         }

         // convert the char[] into a string
         String buf = new String(tmpBuf);

         // line separators in values and tokens are "\n"
         // in order to compare with the file contents, replace them
         // as needed
         Substitution subst = new Perl5Substitution(System.getProperty("line.separator"));
         String tok = Util.substitute(_patmatcher,_newlinePattern,subst,_find.getText(),Util.SUBSTITUTE_ALL);
         String val = Util.substitute(_patmatcher,_newlinePattern,subst,_replacewith.getText(),Util.SUBSTITUTE_ALL);
         
         // create the REs
         Pattern tokpat = null;
         try {
            tokpat = _patcompiler.compile(tok);
         } catch(Exception e) {
            throw new BuildException("REReplace: Error compiling the pattern \"" + tok + "\".",e,location);
         }
         
         Substitution valsubst = null;
         try {
            valsubst = new Perl5Substitution(val);
         } catch(Exception e) {
            throw new BuildException("REReplace: Error compiling the pattern \"" + val + "\".",e,location);
         }
         
         // do the subst
         String newString = Util.substitute(_patmatcher,tokpat,valsubst,buf,Util.SUBSTITUTE_ALL);

         // only write it out if it's changed
         boolean changed = !(newString.equals(buf));
         if(changed) {
            bw.write(newString,0,newString.length());
            bw.flush();
         }

         // cleanup
         bw.close();
         bw = null;
         br.close();
         br = null;

         // If there were changes, move the new one to the old one
         if(changed) {
            deletetemp = false;
            try {
               src.delete();
               temp.renameTo(src);
            } catch(Exception e) {
               throw new BuildException("REReplace: Error overwriting the source file.  The file \"" + temp.getAbsolutePath() + "\" may contain the new file.");
            }
         }
      } catch(Exception e) {
         throw new BuildException(e, location);
      } finally {
         try { bw.close(); } catch(Exception e) { }
         try { br.close(); } catch(Exception e) { }
         if(deletetemp) {
            try { temp.delete(); } catch(Exception e) { }
         }
      }
   }

   /**
    * Sets the file in which to perform the substitution.
    */
   public void setFile(File file) {
      _src = file;
   }

   /**
    * Sets the directory in which to perform the substitution.
    */
   public void setDir(File dir) {
      _dir = dir;
   }

   /**
    * Indicates whether to quietly skip or
    * throw a {@link BuildException} when
    * a locked (not-writeable) file is encountered.
    */
   public void setSkipLocked(boolean b) {
      _skipWhenLocked = b;
   }

   /**
    * Sets the string token to replace.
    */
   public void setFind(String find) {
      createFind().addText(find);
   }

   /**
    * Set the string value to use as token replacement.
    */
   public void setReplaceWith(String value) {
      createReplaceWith().addText(value);
   }

   /**
    * Nested &lt;find&gt; element.
    */
   public TextElement createFind() {
      if(null == _find) {
         _find = new TextElement();
      }
      return _find;
   }

   /**
    * Nested &lt;replacewith&gt; element.
    */
   public TextElement createReplaceWith() {
      return _replacewith;
   }
}
