
package org.apache.tools.ant.gui.xml.dtd;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.IOException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import org.apache.tools.ant.gui.xml.DOMNode;
import org.apache.tools.ant.gui.xml.NamedDOMNodeMap;
import org.apache.tools.ant.gui.xml.TreeWalker;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.DTDHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.xml.sax.ext.DeclHandler;
import org.xml.sax.ext.LexicalHandler;

import org.xml.sax.helpers.XMLReaderFactory;

public class ANTDocumentType {
    public final static int CORE_ELEMENT = 0;
    public final static int OPTIONAL_ELEMENT = 1;
    private boolean isInit = false;
    private HashMap coreElementMap = new HashMap();
    private HashMap optionalElementMap = new HashMap();
    private HashMap elementMap;
    private final static String XMLDOC_1 = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" +
        "<!DOCTYPE project SYSTEM \"file:/";
    private final static String XMLDOC_2 = 
        "\"><project name=\"sample-project\">" + 
        "</project>";
    private final static String DTD_1 = "project.dtd";
    private final static String DTD_2 = "project-ext.dtd";
    private final static String DTD_SHARE = "share.dtd";

    public ANTDocumentType() {
    }
    
    public NamedDOMNodeMap getNotations() {
        return null;
    }
    
    public NamedDOMNodeMap getEntities() {
        return null;
    }
    
    public String getName() {
        return null;
    }
    
    /**
     * Loads the DTD if not already loaded.
     */
    public void init() {
        // Return if already inited. 
        if (isInit) {
            return;
        }
        
        try {
            setupDefaultParser();
            
            XMLReader reader = XMLReaderFactory.createXMLReader();
            
            DtdHandler handler = new DtdHandler();
            reader.setProperty(
                "http://xml.org/sax/properties/lexical-handler",
                handler);
            reader.setProperty(
                "http://xml.org/sax/properties/declaration-handler",
                handler);
            
            reader.setDTDHandler(handler);
            reader.setContentHandler(handler);
            reader.setEntityResolver(new ACSResolver());
            
            String coreDoc = XMLDOC_1 + DTD_1 + XMLDOC_2;
            String optionalDoc = XMLDOC_1 + DTD_2 + XMLDOC_2;
            
            // Parse the core task DTD
            elementMap = coreElementMap;
            InputStream xmldocCore = 
                new ByteArrayInputStream(coreDoc.getBytes());
            reader.parse(new InputSource(xmldocCore));
            
            // Parse the optional task DTD
            elementMap = optionalElementMap;
            InputStream xmldocOptional = 
                new ByteArrayInputStream(optionalDoc.getBytes());
            reader.parse(new InputSource(xmldocOptional));
            
            isInit = true;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    /**
     * If a parser has not been configured, setup a default.
     */
    protected void setupDefaultParser() {
        
        String prop = "org.xml.sax.driver";
        String className = System.getProperty(prop);
        
        // If the default parser is set, do nothing.
        if (className != null) {
            return;
        }
        
        // Look for the crimson parser
        try {
            className = "org.apache.crimson.parser.XMLReaderImpl";
            Class.forName(className);
            System.setProperty(prop, className);
            return;
        } catch (Exception e) {}
        
        // Look for the xerces parser
        try {
            className = "org.apache.xerces.parsers.SAXParser";
            Class.forName(className);
            System.setProperty(prop, className);
            return;
        } catch (Exception e) {}
    }
    
    public DtdElement findElement(int elementType, String name) {
        if (elementType == OPTIONAL_ELEMENT) {
            return (DtdElement) optionalElementMap.get(name);
        }
        return (DtdElement) coreElementMap.get(name);
    }
    
    public static class DtdElement {
        private String _name;
        private String[] _contentModel;
        private DtdAttributes _map = new DtdAttributes();
        
        public String getName() {
            return _name;
        }
        public void setName(String name) {
            _name = name;
        }
        public String[] getContentModel() {
            return _contentModel;
        }
        public void setContentModel(String[] model) {
            _contentModel = model;
        }
        public DtdAttributes getAttributes() {
            return _map;
        }
    }
    
    public static class DtdAttribute {
        private String _name;
        private String _type;
        private String _defaultValue;
        private boolean _isFixed;
        private boolean _isRequired;
        
        public String getName() {
            return _name;
        }
        public void setName(String name) {
            _name = name;
        }
        public String getType() {
            return _type;
        }
        public void setType(String type) {
            _type = type;
        }
        public String getDefaultValue() {
            return _defaultValue;
        }
        public void setDefaultValue(String value) {
            _defaultValue = value;
        }
        public boolean isFixed() {
            return _isFixed;
        }
        public void setFixed(boolean value) {
            _isFixed = value;
        }
        public boolean isRequired() {
            return _isRequired;
        }
        public void setRequired(boolean value) {
            _isRequired = value;
        }
    }

    public static class DtdAttributes extends HashMap {
        /**
         * Default constructor
         */
        public DtdAttributes() {
        }

        /**
         * Adds the Attribute
         *
         * @param attribute new attribute
         */
        public void addAttribute(DtdAttribute attribute) {
            put(attribute.getName(), attribute);
        }

        /**
         * Return the requested attribute
         *
         * @param name attribute name
         * @returns the requested attribute
         */
        public DtdAttribute getAttribute(String name) {
            return (DtdAttribute) get(name);
        }

        /**
         * @returns an array of the optional attribute names
         */
        public String[] getOptionalAttributes() {
            ArrayList list = new ArrayList();
            Iterator i = values().iterator();
            while(i.hasNext()) {
                DtdAttribute a = (DtdAttribute)i.next();
                if (!a.isRequired()) {
                    list.add(a.getName());
                }
            }
            String[] result = new String[list.size()];
            list.toArray(result);
            return result;
        }

        /**
         * @returns an array of the required attribute names
         */
        public String[] getRequiredAttributes() {
            ArrayList list = new ArrayList();
            Iterator i = values().iterator();
            while(i.hasNext()) {
                DtdAttribute a = (DtdAttribute)i.next();
                if (a.isRequired()) {
                    list.add(a.getName());
                }
            }
            String[] result = new String[list.size()];
            list.toArray(result);
            return result;
        }
        /**
         * @returns an array of the all attribute names
         */
        public String[] getAttributes() {
            ArrayList list = new ArrayList();
            Iterator i = values().iterator();
            while(i.hasNext()) {
                DtdAttribute a = (DtdAttribute)i.next();
                list.add(a.getName());
            }
            String[] result = new String[list.size()];
            list.toArray(result);
            return result;
        }
    }
    
    /**
     * When parsing XML documents, DTD related events are signaled through
     * this interface. 
     */
    class DtdHandler implements DTDHandler, ContentHandler, DeclHandler,
            LexicalHandler {
        
        public void attributeDecl (
            String		elementName,
            String		attributeName,
            String		attributeType,
            String		valueDefault,
            String              value
        ) throws SAXException
        {
            // Try to find the element.
            DtdElement e = (DtdElement) elementMap.get(elementName);
            if (e == null) {
                throw new SAXException("element " + elementName +
                " not declared before attributes");
            }
            
            // Update the element's attribute.
            DtdAttribute attrib = new DtdAttribute();
            attrib.setName(attributeName);
            attrib.setType(attributeType);
            boolean isRequired = ("#REQUIRED".equals(valueDefault));
            boolean isFixed = ("#FIXED".equals(valueDefault));
            attrib.setFixed(isFixed);
            attrib.setRequired(isRequired);
            attrib.setDefaultValue(value);
            e.getAttributes().addAttribute(attrib);
        }
        
        public void elementDecl (
            String elementName,
            String contentModel
        ) throws SAXException
        {
            DtdElement e = new DtdElement();
            e.setName(elementName);

            // Break the contentModel string into pieces.
            ArrayList list = new ArrayList();
            StringTokenizer st = new StringTokenizer(contentModel, "|()*");
            while (st.hasMoreTokens()) {
                String s = st.nextToken();
                if ( s.length() > 0 && !"EMPTY".equals(s) ) {
                    list.add(s);
                }
            }            
            String[] array = new String[list.size()];
            list.toArray(array);
            e.setContentModel(array);
            
            // Update the map
            elementMap.put(e.getName(), e);
        }
        
        public void unparsedEntityDecl(String p1, String p2, String p3,
                String p4) throws SAXException {}

        public void notationDecl(String p1, String p2, String p3)
                throws SAXException {}
   
        public void endElement(String p1,String p2,String p3)
                throws SAXException {}
        
        public void setDocumentLocator(Locator p1) {}
        
        public void startPrefixMapping(String p1,String p2)
                throws SAXException {}
        
        public void endDocument() throws SAXException {}
        
        public void startDocument() throws SAXException {}
        
        public void endPrefixMapping(String p1) throws SAXException {}
        
        public void startElement(String p1,String p2,String p3,
                Attributes p4) throws SAXException {}
        
        public void ignorableWhitespace(char[] p1,int p2,int p3)
                throws SAXException {}
        
        public void skippedEntity(String p1) throws SAXException {}
        
        public void processingInstruction(String p1,String p2)
                throws SAXException {}
        
        public void characters(char[] p1,int p2,int p3)
                throws SAXException {}
        
        public void externalEntityDecl(String p1,String p2,String p3)
                throws SAXException {}
        
        public void internalEntityDecl(String p1,String p2)
                throws SAXException {}
        
        public void endDTD()
                throws SAXException {}
        
        public void startCDATA()
                throws SAXException {}
        
        public void startEntity(String p1)
                throws SAXException {}
        
        public void endCDATA()
                throws SAXException {}
        
        public void endEntity(String p1)
                throws SAXException {}
        
        public void startDTD(String p1,String p2,String p3)
                throws SAXException {}
        
        public void comment(char[] p1,int p2,int p3)
                throws SAXException {}
    }
    
    /**
     * We provide the location for the ant dtds.
     */
    class ACSResolver implements org.xml.sax.EntityResolver {
        
        public InputSource resolveEntity (
            String publicId,
            String systemId)
            throws SAXException, IOException {
                
            InputStream result = null;
            
            // Is it the project.dtd?
            if (systemId.indexOf(DTD_1) != -1) {
                try {
                    // Look for it as a resource
                    result = getClass().getResourceAsStream(DTD_1);
                } catch (Exception e) {}
            }
            // Is it the project-ext.dtd?
            if (systemId.indexOf(DTD_2) != -1) {
                try {
                    // Look for it as a resource
                    result = getClass().getResourceAsStream(DTD_2);
                } catch (Exception e) {}
            }
            if (result != null) {
                return new InputSource(result);
            }
            // Is it the share.dtd?
            if (systemId.indexOf(DTD_SHARE) != -1) {
                try {
                    // Look for it as a resource
                    result = getClass().getResourceAsStream(DTD_SHARE);
                } catch (Exception e) {}
            }
            if (result != null) {
                return new InputSource(result);
            }

            // Otherwise, use the default impl.
            com.sun.xml.parser.Resolver r = new com.sun.xml.parser.Resolver();
            return r.resolveEntity(publicId, systemId);
        }
    }
}
