/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.taskdefs.optional.sos;


import java.io.File;
import org.apache.tools.ant.*;
import org.apache.tools.ant.taskdefs.Execute;
import org.apache.tools.ant.taskdefs.LogStreamHandler;
import org.apache.tools.ant.types.Commandline;
import org.apache.tools.ant.types.Path;

/**
 * A base class for creating tasks for executing commands on SourceOffSite
 * SOS is a remote access add-on to Microsoft's Source Safe
 * It also provides Linux / Solaris connectivity.
 * <p>
 * The class extends the 'exec' task as it operates by executing the soscmd(.exe) program
 * supplied with SOS. By default the task expects soscmd(.exe) to be in the path,
 * you can override this be specifying the sosdir attribute.
 * </p>
 * <p>
 * This class provides set and get methods for the following attributes
 * <br> 'login'
 * <br> 'password'
 * <br> 'projectpath'
 * <br> 'soshome'
 * <br> 'vssserverpath'
 * <br> 'sosserverpath'
 * <br> 'soscmddir'
 * <br> 'vsspath'
 * <br> 'file'
 * <br> 'comment'
 * <br> 'localpath"
 * <br> 'label'
 * <br> 'version'
 * <br> 'recursive'
 * <br> 'verbose'
 * <br> 'nocache'
 * <br> 'nocompression'
 * <br>
 *  It also contains constants for the flags that can be passed to SOS.
 * <p>
 *  These tasks are loosly based on the VSS tasks
 *
 * @author Jesse Stockall - jesse@cryptocard.com
 */

public abstract class SOS extends Task {

    private String m_sosCmd = "";
    private String m_vssLogin = null;
    private String m_vssPass = null;
    private String m_ProjectPath = null;
    private String m_vssServerPath = null;
    private String m_sosServerPath = null;
    private String m_sosHome = null;
    private String m_LocalPath = null;
    private String m_File = null;
    private String m_Version = null;
    private String m_Label = null;
    private String m_Comment = null;

    private boolean m_NoCompress = false;
    private boolean m_NoCache = false;
    private boolean m_Recursive = false;
    private boolean m_Verbose = false;


    /**
     * Set the directory where soscmd(.exe) is located
     *
     * @param dir the directory containing soscmd(.exe)
     */
    public final void setSosCmd(String dir) {
        m_sosCmd = project.translatePath(dir);
    }

    /**
     * Builds and returns the command string to execute soscmd(.exe)
     */
    public final String getSosCmd() {
        String toReturn = m_sosCmd;
        if ( !toReturn.equals("") && !toReturn.endsWith("\\") ) {
            toReturn += "\\";
        }
        toReturn += SOSCMD_EXE;
        return toReturn;
    }

    /**
     * Set the login to use when accessing VSS via SOS.
     * <p>
     * Should be just the username
     *
     * @param login the user name to to be used for SOS login
     */
    public final void setLogin(String login) {
        m_vssLogin = login;
    }

    /**
     * @return the appropriate login
     */
    public String getLogin() {
        if ( m_vssLogin == null ) {
            return null;
        } else {
            return m_vssLogin;
        }
    }


    /**
    * Set the password to be used for login
    *
    * @param password to be used for SOS login
    */
    public final void setPass(String password) {
        if (password == null) {
            m_vssPass = " ";
            //set it to " " so that you can login with no pass
	    // SOS requires the -password parameter even if no password is used
	} else {
	    m_vssPass = password;
        }
    }

    /**
     * @return the appropriate password if it was specified, otherwise an empty string
     */
    public String getPass() {
        if ( m_vssPass == null ) {
            return "";
        } else {
            return m_vssPass;
        }
    }


    /**
     * Set the path to the item in SOS to operate on
     * <p>
     * Ant can't cope with a '$' sign in an attribute so we have to add it here.
     * Also we strip off any 'vss://' prefix which is an XMS special and should probably be removed!
     *
     * @param projectpath specified in the buildfile
     */
    public final void setProjectPath(String projectpath) {
        if ( projectpath.startsWith("vss://") ) {
            m_ProjectPath= PROJECT_PREFIX + projectpath.substring(5);
        } else {
            m_ProjectPath = PROJECT_PREFIX + projectpath;
        }
    }

    /**
     * @return the VSS project to act upon
     */
    public String getProjectPath() {
        return m_ProjectPath;
    }

    /**
     * Set the path to the location of the ss.ini
     *
     * @param vssServerPath specified in the buildfile
     */
    public final void setVssServerPath(String vssServerPath) {
        m_vssServerPath = vssServerPath;
    }

    /**
     * @return the path to srcsafe.ini
     */
    public String getVssServerPath() {
        return m_vssServerPath;
    }


   /**
    * Set the path of soshome
    * @param sosHome specified in the buildfile
    */
    public final void setSosHome(String sosHome) {
        m_sosHome = sosHome;
    }

   /**
    * @return the path of the SourceOffSite home directory
    */
    public String getSosHome() {
        return m_sosHome;
    }

    /**
     * Set the address & port of SourceOffSite Server
     *
     * @param sosServerPath the Address & port of the SOS server specified in the buildfile
     */
    public final void setSosServerPath(String sosServerPath) {
        m_sosServerPath = sosServerPath;
    }

    /**
     * @return the Address & port of the SOS server
     */
    public String getSosServerPath() {
        return m_sosServerPath;
    }

    /**
     * Set the local path.
     *
     * @param localPath specified in the buildfile
     */
    public void setLocalPath(Path localPath) {
        m_LocalPath = localPath.toString();
    }

    /**
     * Builds and returns the working directory if required
     * <p>
     * The localpath is created if it didn't exist
     *
     * @return the absolute path of the working directory
     */
    public String getLocalPath() {
        if (m_LocalPath == null) {
            return null;
        } else {
            // make sure m_LocalDir exists, create it if it doesn't
            File dir = project.resolveFile(m_LocalPath);
            if (!dir.exists()) {
                boolean done = dir.mkdirs();
                if (done == false) {
                    String msg = "Directory " + m_LocalPath + " creation was not " +
                        "successful for an unknown reason";
                    throw new BuildException(msg, location);
                }
                project.log("Created dir: " + dir.getAbsolutePath());
            }
		return dir.getAbsolutePath();
        }
    }

    /**
     * Set the file to get
     * @param file specified in the buildfile
     */
    public final void setFile(String file) {
        m_File = file;
    }

    /**
     * Get the filename
     * @return the file to act upon or null if there is none
     */
    public String getFile() {
    	return m_File;
    }

   /**
    * Disable compression
    *
    * @param nocompress boolean specified in the buildfile
    */
    public void setNoCompress(boolean nocompress) {
        m_NoCompress = nocompress;
    }

   /**
    * @return the 'nocompression' flag if the attribute was 'true', otherwise an empty string
    */
    public String getNoCompress() {
        if ( !m_NoCompress ) {
            return "";
        } else {
            return FLAG_NO_COMPRESSION;
        }
    }

   /**
    * Disable the cache
    *
    * @param nocache boolean specified in the buildfile
    */
    public void setNoCache(boolean nocache) {
        m_NoCache = nocache;
    }

   /**
    * @return the 'nocache' command if the attribute was 'true', otherwise an empty string
    */
    public String getNoCache() {
        if ( !m_NoCache ) {
            return "";
        } else {
            return FLAG_NO_CACHE;
        }
    }

   /**
    * Set behaviour verbose or quiet
    *
    * @param verbose boolean specified in the buildfile
    */
    public void setVerbose(boolean verbose) {
        m_Verbose = verbose;
    }

   /**
    * @return the 'verbose' command if the attribute was 'true', otherwise an empty string
    */
    public String getVerboseCommand() {
        if ( !m_Verbose ) {
            return "";
        } else {
            return FLAG_VERBOSE;
        }
    }

    /**
     * Set behaviour recursive or non-recursive
     *
     * @param recursive boolean specified in the buildfile
     */
    public void setRecursive(boolean recursive) {
        m_Recursive = recursive;
    }

    /**
     * @return the 'recursive' command if the attribute was 'true', otherwise an empty string
     */
    public String getRecursiveCommand() {
        if ( !m_Recursive ) {
            return "";
        } else {
            return FLAG_RECURSION;
        }
    }

   /**
    * Set the stored version string
    * <p>
    * Note we assume that if the supplied string has the value "null" that something
    * went wrong and that the string value got populated from a null object. This
    * happens if a ant variable is used e.g. version="${ver_server}" when ver_server
    * has not been defined to ant!
    *
    * @param version specified in the buildfile
    */
    public void setVersion(String version) {
        if (version.equals("") || version.equals("null") ) {
            m_Version = null;
        } else {
            m_Version = version;
        }
    }

    /**
     * @return the revision to get or null if there is none
     */
    public String getVersion() {
        return m_Version;
    }

    /**
     * Set the labeled version to operate on in SourceSafe
     * <p>
     * Note we assume that if the supplied string has the value "null" that something
     * went wrong and that the string value got populated from a null object. This
     * happens if a ant variable is used e.g. label="${label_server}" when label_server
     * has not been defined to ant!
     *
     * @param version specified in the buildfile
     */
    public void setLabel(String label) {
        if ( label.equals("") || label.equals("null") ) {
            m_Label = null;
        } else {
            m_Label = label;
        }
    }

    /**
     * @return the label to act upon or null if there is none
     */
    public String getLabel() {
        return m_Label;
    }

    /**
     * Set the comment to get
     * @param commnent specified in the buildfile
     */
    public void setComment(String comment) {
        if ( comment.equals("") || comment.equals("null") ) {
	    m_Comment = null;
        } else {
            m_Comment = comment;
        }
    }

    /**
     * Get the comment
     * @return the comment to use or null if there is none
     */
    public String getComment() {
        return m_Comment;
    }

    protected int run(Commandline cmd) {
        try {
            Execute exe = new Execute(new LogStreamHandler(this,
                                                           Project.MSG_INFO,
                                                           Project.MSG_WARN));

            exe.setAntRun(project);
            exe.setWorkingDirectory(project.getBaseDir());
            exe.setCommandline(cmd.getCommandline());
            return exe.execute();
        } catch (java.io.IOException e) {
            throw new BuildException(e, location);
        }
    }

    /**
     * Constant for the thing to execute
     */
    private static final String SOSCMD_EXE = "soscmd";
    /** */
    public static final String PROJECT_PREFIX = "$";

    /**
     * The 'GetFile' command
     */
    public static final String COMMAND_GET_FILE = "GetFile";
    /**
     * The 'GetProject' command
     */
    public static final String COMMAND_GET_PROJECT = "GetProject";
    /**
     * The 'Checkout File' command
     */
    public static final String COMMAND_CHECKOUT_FILE = "CheckOutFile";
    /**
     * The 'Checkout Project' command
     */
    public static final String COMMAND_CHECKOUT_PROJECT = "CheckOutProject";
    /**
     * The 'Checkin File' command
     */
    public static final String COMMAND_CHECKIN_FILE = "CheckInFile";
    /**
     * The 'Checkin Project' command
     */
    public static final String COMMAND_CHECKIN_PROJECT = "CheckInProject";
    /**
     * The 'History' command
     */
    public static final String COMMAND_HISTORY = "GetFileHistory";
    /**
     * The 'Label' command
     */
    public static final String COMMAND_LABEL = "AddLabel";

    /** */
    public static final String FLAG_COMMAND = "-command";
    /** */
    public static final String FLAG_COMMAND_FILE = "File";
    /** */
    public static final String FLAG_COMMAND_PROJECT = "Project";
    /** */
    public static final String FLAG_DATABASE = "-database";
    /** */
    public static final String FLAG_LOGIN = "-name";
    /** */
    public static final String FLAG_PASS = "-password";
    /** */
    public static final String FLAG_COMMENT = "-log";
    /** */
    public static final String FLAG_OVERRIDE_WORKING_DIR = "-workdir";
    /** */
    public static final String FLAG_RECURSION = "-recursive";
    /** */
    public static final String FLAG_VERSION = "-revision";
    /** */
    public static final String FLAG_LABEL = "-label";
    /** */
    public static final String FLAG_NO_COMPRESSION = "-nocompress";
    /** */
    public static final String FLAG_NO_CACHE = "-nocache";
    /** */
    public static final String FLAG_SOS_SERVER = "-server";
    /** */
    public static final String FLAG_PROJECT = "-project";
    /** */
    public static final String FLAG_VERBOSE = "-verbose";
    /** */
    public static final String FLAG_FILE = "-file";
    /** */
}