/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.taskdefs.optional.sos;

import org.apache.tools.ant.*;
import org.apache.tools.ant.types.Commandline;


import java.io.File;

/**
 * Task to perform GET commands from Microsoft Visual Source Safe via a SourceOffSite server.
 * <p>
 * The following attributes are interpretted:
 * <table border="1">
 *   <tr>
 *     <th>Attribute</th>
 *     <th>Values</th>
 *     <th>Required</th>
 *   </tr>
 *   <tr>
 *      <td>soscmddir</td>
 *      <td>path to soscmd(.exe)</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>vssserverpath</td>
 *      <td>path to the srcsafe.ini  - eg. \\server\vss\srcsafe.ini</td>
 *      <td>Yes</td>
 *   <tr>
 *   <tr>
 *      <td>sosserverpath</td>
 *      <td>address & port of the SOS server  - eg. address:port</td>
 *      <td>Yes</td>
 *   <tr>
 *   <tr>
 *      <td>projectpath</td>
 *      <td>SourceSafe project path with out the "$"</td>
 *      <td>Yes</td>
 *   <tr>
 *   <tr>
 *      <td>file</td>
 *      <td>Filename to act upon<br> If no file is specified then act upon the project</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>login</td>
 *      <td>SourceSafe username</td>
 *      <td>Yes</td>
 *   <tr>
 *   <tr>
 *      <td>pass</td>
 *      <td>SourceSafe password</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>localpath</td>
 *      <td>Override the working directory and get to the specified path</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>soshome</td>
 *      <td>The path to the SourceOffSite home directory</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>nocompression</td>
 *      <td>true or false - disable compression</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>recursive</td>
 *      <td>true or false - Only works with the GetProject command</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>version</td>
 *      <td>a version number to get - Only works with the GetFile command</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>label</td>
 *      <td>a label version to get - Only works with the GetProject command</td>
 *      <td>No</td>
 *   <tr>
 *      <td>nocache</td>
 *      <td>true or false - Only needed if SOSHOME is set as an enviroment variable</td>
 *      <td>No</td>
 *   <tr>
 *   <tr>
 *      <td>verbose</td>
 *      <td>true or false - Status messages are displayed</td>
 *      <td>No</td>
 *   <tr>
 * </table>
 *
 *
 * @author <a href="mailto:jesse@cryptocard.com">Jesse Stockall</a>
 */

public class SOSGET extends SOS {

    //Prefix for this task
    private static final String FLAG_CMD = "Get";

   /**
    * Executes the task.
    * <p>
    * Builds a command line to execute soscmd and then calls Exec's run method
    * to execute the command line.
    */
    public void execute() throws BuildException {
        Commandline commandLine = new Commandline();
        int result = 0;

       /**
        * build the command line from what we got the format is:
        *
        * the GetFile command required valid -server -name -password
        *   -database -project -file parameters
        *  optional ones are -workdir -revision
        *
        * the GetProject command required valid -server -name -password
        * -database -project parameters
        * optional ones are -label -workdir -recursive
        */

        // Get the path to the soscmd(.exe)
        commandLine.setExecutable(getSosCmd());


       /*
        * If we find a "file" attribute then act on a file
        * otherwise act on a project
        */
        if (getFile() != null) {
            // add -command GetFile to the commandline
            commandLine.createArgument().setValue(FLAG_COMMAND);
            commandLine.createArgument().setValue(FLAG_CMD+FLAG_COMMAND_FILE);
            // add -file xxxxx to the commandline
            commandLine.createArgument().setValue(FLAG_FILE);
            commandLine.createArgument().setValue(getFile());
            // look for a version attribute
            if (getVersion() != null) {
                //add -revision xxxxx to the commandline
                commandLine.createArgument().setValue(FLAG_VERSION);
                commandLine.createArgument().setValue(getVersion());
            }
        } else {
            // add -command GetProject to the commandline
            commandLine.createArgument().setValue(FLAG_COMMAND);
            commandLine.createArgument().setValue(FLAG_CMD+FLAG_COMMAND_PROJECT);
            // look for a recursive option
            commandLine.createArgument().setValue(getRecursiveCommand());
            // look for a label option
            if (getLabel() != null) {
                commandLine.createArgument().setValue(FLAG_LABEL);
                commandLine.createArgument().setValue(getLabel());
            }
        }

        // SOS server address is required
        if (getSosServerPath() == null) {
            String msg = "sosServerPath attribute must be set!";
            throw new BuildException(msg, location);
        }
        commandLine.createArgument().setValue(FLAG_SOS_SERVER);
        commandLine.createArgument().setValue(getSosServerPath());

        // Login info is required
        if (getLogin() == null) {
            String msg = "login attribute must be set!";
            throw new BuildException(msg, location);
        }
        commandLine.createArgument().setValue(FLAG_LOGIN);
        commandLine.createArgument().setValue(getLogin());

        // The SOS class knows that the SOS server needs a password,
        // so even if the password attribute is blank we send a " "
        commandLine.createArgument().setValue(FLAG_PASS);
        commandLine.createArgument().setValue(getPass());

        // VSS Info is required
        if (getVssServerPath() == null) {
            String msg = "vssServerPath attribute must be set!";
            throw new BuildException(msg, location);
        }
        commandLine.createArgument().setValue(FLAG_DATABASE);
        commandLine.createArgument().setValue(getVssServerPath());

        // SS project path is required
        if (getProjectPath() == null) {
            String msg = "ProjectPath attribute must be set!";
            throw new BuildException(msg, location);
        }
        commandLine.createArgument().setValue(FLAG_PROJECT);
        commandLine.createArgument().setValue(getProjectPath());

        // The following options are optional.

        // -verbose
        commandLine.createArgument().setValue(getVerboseCommand());

        // Disable Compression
        commandLine.createArgument().setValue(getNoCompress());

        // Path to the SourceOffSite home directory
        if (getSosHome() == null) {
            // If -soshome was not specified then we can look for nocache
            commandLine.createArgument().setValue(getNoCache());
        } else {
            commandLine.createArgument().setValue(getSosHome());
        }


       /**
        * If a working directory was specified then add it to the command line
        */
        if (getLocalPath() != null) {
            commandLine.createArgument().setValue(FLAG_OVERRIDE_WORKING_DIR);
            commandLine.createArgument().setValue(getLocalPath());
        }

        result = run(commandLine);

        if ( result == 255 ) {
            // This is the exit status
            String msg = "Failed executing: " + commandLine.toString();
            throw new BuildException(msg, location);
        }
    }
}

