/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.types.cullers;

import java.io.File;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Locale;

import org.apache.tools.ant.types.EnumeratedAttribute;

/**
 * Culler that filters files based on their last modified date.
 *
 * @author Bruce Atherton <a href="mailto:bruce@callenish.com">bruce@callenish.com</a>
 */
public class DateCuller extends Culler {

    private long millis = -1;
    private String dateTime = null;
    private boolean includeDirs = false;
    private int cmp = 0;
    private String errmsg = null;

    public DateCuller() {
    }

    /**
     * For users that prefer to express time in milliseconds since 1970
     *
     * @param millis the time to compare file's last modified date to,
     *        expressed in milliseconds
     */
    public void setMillis(long millis) {
        this.millis = millis;
    }

    /**
     * Sets the date. The user must supply it in MM/DD/YYYY HH:MM AM_PM
     * format
     *
     * @param dateTime a string in MM/DD/YYYY HH:MM AM_PM format
     */
    public void setDatetime(String dateTime) {
        this.dateTime = dateTime;
        if (dateTime != null) {
            DateFormat df = DateFormat.getDateTimeInstance(
                                                    DateFormat.SHORT,
                                                    DateFormat.SHORT,
                                                    Locale.US);
            try {
                setMillis(df.parse(dateTime).getTime());
                if (millis < 0) {
                    errmsg = "Date of " + dateTime
                        + " results in negative milliseconds value relative"
                        + " to epoch (January 1, 1970, 00:00:00 GMT).";
                }
            } catch (ParseException pe) {
                    errmsg = "Date of " + dateTime
                        + " Cannot be parsed correctly. It should be in"
                        + " MM/DD/YYYY HH:MM AM_PM format.";
            }
        }
    }

    /**
     * Sets the type of comparison to be done on the file's last modified
     * date.
     *
     * @param cmp The comparison to perform, an EnumeratedAttribute
     */
    public void setWhen(Comparisons cmp) {
        this.cmp = cmp.getIndex();
    }


    /**
     * This abstract method should implement an internal consistency check
     * on each culler, to ensure it's values are reasonable.
     *
     * @return an error message, or null if no error
     */
    public String checkForError() {
        if (millis < 0) {
            return "Date of " + dateTime + " results in negative milliseconds"
                + " value relative to epoch (January 1, 1970, 00:00:00 GMT).";
        }
        return errmsg;
    }

    /**
     * The heart of the matter. This is where the culler gets to veto
     * the inclusion of a file in a particular fileset.
     *
     * @param filename is the name of the file to check
     * @param file is a java.io.File object the culler can use
     * @return whether the file should be culled or not
     */
    public boolean isCulled(String filename, File file) {
        if (file.isDirectory() && (includeDirs == false)) {
            return false;
        }
        if (cmp == 0) {
            return (file.lastModified() < millis);
        }
        else if (cmp == 1) {
            return (file.lastModified() > millis);
        }
        else {
            return (file.lastModified() == millis);
        }
    }

    /**
     * Enumerated attribute with the values for comparison.
     * <p>
     * Note that there is no need for <code>beforeorequal</code> or
     * <code>afterorequal</> because you can get the same effect
     * by inverting the opposite. So setting <code>mustbe="before"</code>
     * and <code>inverted="true"</code> culls the files that were last
     * modified on a date after or equal to the date you specify.
     */
    public static class Comparisons extends EnumeratedAttribute {
        public String[] getValues() {
            return new String[] {"before", "after", "equal"};
        }
    }

}


