/*
 * libss7: An implementation of Signalling System 7
 *
 * Written by Paul Bagyenda <bagyenda@dsmagic.com>
 *
 * SIGTRAN M3UA over SCTP implementation 
 *
 * Copyright (C) 2009-, Digital Solutions
 * All Rights Reserved.
 */

/*
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2 as published by the
 * Free Software Foundation. See the LICENSE file included with
 * this program for more details.
 *
 * In addition, when this program is distributed with Asterisk in
 * any form that would qualify as a 'combined work' or as a
 * 'derivative work' (but not mere aggregation), you can redistribute
 * and/or modify the combination under the terms of the license
 * provided with that copy of Asterisk, instead of the license
 * terms granted here.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <errno.h>
#include "libss7.h"
#include "ss7_internal.h"
#include "m3ua.h"
#include "mtp3.h"
#include "isup.h"


#ifdef WITH_SCTP
#include "netinet/sctp.h"
#endif

struct m3ua_header {
	unsigned char ver;
	unsigned char _reserved;
	unsigned char msg_class;
	unsigned char msg_type;
	unsigned int len;
};

struct m3ua_data_header {
	u_int32_t opc;
	u_int32_t dpc;
	u_int8_t si;
	u_int8_t ni;
	u_int8_t mp;
	u_int8_t sls;	
};

#define M3UA_NET_MSG 0xFF /* impossible user part, used to signal that this is a net message */
#define M3UA_PPID    0x03 /* M3UA SCTP PID val */
#define ROUNDUP(x,n) (((x)+(n)-1) & ~((n)-1)) /* assumes n is power of two */

#define DEFAULT_RETRY_TIMEOUT 2000
struct m3ua *m3ua_new(char *name, unsigned int adjpc)
{
	struct m3ua *m = calloc(1, sizeof *m);
	
	if (!m)
		return NULL;
	strncpy(m->name, name, sizeof name);
	m->adjpc = adjpc;
	m->state = M3UA_ASP_DOWN;
	m->t1 = -1;
	m->t2 = -1;
	m->t3 = -1;
	return m;
}

int m3ua_set_routinglabel(unsigned char *sif, struct routing_label *rl)
{
	struct m3ua_data_header *x = (void *)sif;
	
	x->opc = htonl(rl->opc);
	x->dpc = htonl(rl->dpc);
	x->sls = rl->sls;
	return 12;
}

#define COMP_PC(pc1,pc2,mask) (((pc1) ^ (pc2)) & (mask)) /* compare PCs with bits masked off */

static inline int dest_congestion(struct m3ua *link, int dpc)
{
	struct m3ua_congestion *c;
	for (c = link->con; c; c = c->next)
		if (COMP_PC(dpc, c->pc, c->mask) == 0) 
			return c->con;	
	return 0;
}

/* adapted from mtp3.c */
static inline int link_available(struct ss7 *ss7, int linkid, struct routing_label rl)
{
	struct m3ua *m = ss7->m3ua_links[linkid];
	if (m && m->state == M3UA_ASP_ACTIVE) {
		struct m3ua_route *r;

		if (m->adjpc == rl.dpc) /* Directly linked, report as such */
			return 1;
		
		/* Check list, if our destination is in there, stop. */
		for (r = m->routes; r; r = r->next) 
			if (COMP_PC(r->pc,rl.dpc,r->mask) == 0) /* same in all bits but masked ones */
				return r->avail;		
	}
	
	return 0;
}

static inline struct m3ua * rl_to_link(struct ss7 *ss7, struct routing_label rl, int has_sls)
{
	int linkid;

	linkid = (rl.sls >> ss7->sls_shift) % ss7->num_m3ua_links;

	if (has_sls && link_available(ss7, linkid, rl))
		return ss7->m3ua_links[linkid];
	else {
		struct m3ua *winner = NULL;
		int i, best;

		if (has_sls)
			for (i = 0; i < ss7->num_m3ua_links; i++) {
				if (link_available(ss7, i, rl)) {
					winner = ss7->m3ua_links[i];
					break;
				}
			}
		else  /* Find the one with the least congestion on this destination */			
			for (i = 0, best = 0; i < ss7->num_m3ua_links; i++) 
				if (link_available(ss7, i, rl)) {
					int c = dest_congestion(ss7->m3ua_links[i], rl.dpc);
					if (!winner  || c < best)
						winner = ss7->m3ua_links[i];
				}		
		return winner;
	}
}

static struct m3ua * rl_to_m3ua_no_check(struct ss7 *ss7, struct routing_label rl)
{
	int i = 0;
	struct m3ua *link = NULL;

	for (i = 0; i < ss7->num_m3ua_links; i++) {
		if (ss7->m3ua_links[i]->slc == rl.sls && (ss7->m3ua_links[i]->adjpc == rl.dpc)) {
			link = ss7->m3ua_links[i];
			break;
		}
	}

	return link;
}

int m3ua_rl_to_link(struct ss7 *ss7, struct routing_label rl, int userpart, int has_sls, struct ss7_route *rt)
{
	
	if (ss7->num_m3ua_links == 0)
		return -1;
        else if (userpart == SIG_ISUP || userpart == SIG_SCCP)
		rt->link = rl_to_link(ss7,  rl,has_sls);
	else 
		rt->link = rl_to_m3ua_no_check(ss7, rl);

	if (rt->link == NULL)
		return -1;
	rt->type = M3UA_ROUTE;
	rt->data = NULL;
	
	return 0;
}

int m3ua_set_sockopts(int fd)
{
	/* Set sctp events flags */
#ifdef WITH_SCTP
	struct sctp_event_subscribe s = {
		.sctp_send_failure_event = 1,
		.sctp_shutdown_event = 1,
		.sctp_data_io_event  = 1,
	};
	return setsockopt(fd, IPPROTO_SCTP, SCTP_EVENTS, &s, sizeof s);
#else
	return -1;     
#endif

}
static void down_link(struct m3ua *link);
static int m3ua_transmit_int(struct ss7 *ss7, struct m3ua *link, 
			     u_int16_t msg_type,
			     u_int8_t sls,
			     struct ss7_msg *m)
{
	struct m3ua_header *hdr = (void *)m->buf;
	int res;
	hdr->msg_class = M3UA_MSG_CLASS(msg_type);
	hdr->msg_type  = M3UA_MSG_CODE(msg_type);
     
	hdr->len = htonl(m->size + M3UA_COMMON_HLEN);
	hdr->ver = 1;

#ifdef WITH_SCTP
	res = sctp_sendmsg(link->fd, m->buf, m->size + M3UA_COMMON_HLEN, NULL, 0, M3UA_PPID, 0, sls, 0, 0);

	if (res < 0) { /* Close connection? If no associations  */
		uint32_t n = 0;
		socklen_t x = sizeof n;
		if (getsockopt(link->fd, IPPROTO_SCTP, 
			       SCTP_GET_ASSOC_NUMBER, &n, &x) == 0 && 
		    n<=0) 
			link->state = M3UA_ASP_DOWN; /* But don't down_link() since caller might do that. */
		
		
	}
#else
	ss7_error(ss7, "No M3UA support (SCTP implementation missing!\n");
	res = -1;
#endif
	free(m);
	return res;
}

#define add_tlv(p,tag,val,len)  do {					\
		u_int16_t *_t = (void *)(p);				\
		u_int32_t  _len = (len)+2*sizeof _t[0];			\
		_t[0] = htons(tag);					\
		_t[1] = htons(_len); /*incl. len of tl part */		\
		memcpy(((char *)(p)) + 2*sizeof _t[0], (val), (len));	\
		_len = ROUNDUP(_len,4);					\
		_t += _len/sizeof _t[0]; /* Skip to end */		\
		(p) = (void *)_t;					\
	} while (0)


int m3ua_transmit_ex(struct ss7 *ss7, struct ss7_route rt, unsigned int userpart, 
		     int priority, int corrid, struct ss7_msg *m)
{
	struct m3ua *link = rt.link;
	if (link) {
		/* Fix up message: headers and so forth */
		struct m3ua_data_header *dhr = (void *)(m->buf + M3UA_COMMON_HLEN + M3UA_DATA_HLEN);
		u_int16_t *s = (u_int16_t *)(m->buf + M3UA_COMMON_HLEN);
		u_int32_t cid = (corrid >= 0) ? corrid : link->fd;
		unsigned char *p;
	     
		s[0] = htons(0x0210); /* Tag  for data param */
		s[1] = htons(m->size); /* already includes length of tlv (before padding) */
	     
		dhr->mp = priority;
		dhr->ni = ss7->ni;			
		dhr->si = userpart & 0xFF; /* indicate user part. */
	     
		m->size = ROUNDUP(m->size, 4); /* Add padding. XXX assumes allocator cleared the mem. */
		p =  (void *)(m->buf + M3UA_COMMON_HLEN + m->size);
		
		/* Now add correlation ID */
		add_tlv(p, 0x0013, &cid, sizeof cid);
		m->size += 8; /* added correlation TLV */
		return m3ua_transmit_int(ss7, link, M3UA_DATA_MSG, rt.sls, m);
	} else {
                ss7_error(ss7, "No signalling link available sending message!\n");
                free(m);
                return -1;
	}
}

void m3ua_free(struct m3ua *m)
{
	struct m3ua_route *x;
	if (!m)
		return;
     
	while (m->routes) {
		x = m->routes;
		m->routes = m->routes->next;
		free(x);
	}


	while (m->keys) {
		struct m3ua_routing_key *x = m->keys;
		m->keys = m->keys->next;
		free(x);
	}

	free(m);
}

/* Remove a link completely, assume we are locked */
static void remove_link(struct m3ua *link)
{
	struct ss7 *ss7 = link->master;
	int i;
     
	ss7_schedule_del(ss7, &link->t1);
	ss7_schedule_del(ss7, &link->t2);
	ss7_schedule_del(ss7, &link->t3);

	for (i = 0; i<ss7->num_m3ua_links; i++) 
		if (ss7->m3ua_links[i] == link) {
			int nobjs = ss7->num_m3ua_links-i-1;
			ss7_event *e;

			if (nobjs > 0) /* remove it */
				memmove(&ss7->m3ua_links[i], &ss7->m3ua_links[i + 1], nobjs*sizeof ss7->m3ua_links[0]);
			ss7->num_m3ua_links--; 
			
			/* Inform upper layer of shutdown and delete this puppy */
			e = ss7_next_empty_event(ss7);
			if (e) {
				e->e = M3UA_LINK_DOWN;
				e->gen.data = link->fd;
			}
			m3ua_free(link);
			break;
		}
}

#define DOWN_RETRIES 5

/* Attempt to bring a link down gracefully */
static void down_link(struct m3ua *link)
{
	struct ss7_msg *m;

	if (!link)
		return;
	ss7_schedule_del(link->master, &link->t3); /* Cancel any pending */
	ss7_schedule_del(link->master, &link->t2); 
	ss7_schedule_del(link->master, &link->t1); 
	if (link->state == M3UA_ASP_DOWN)
		goto free_link;
	
	m = ss7_msg_new();
	
	if (m) {
		unsigned char *p = m->buf + M3UA_COMMON_HLEN, *q = p;
		int msg = -1;

		if (link->down_ctr > DOWN_RETRIES)  /* handle retries timeout issue. */
			link->state = (link->state == M3UA_ASP_IA_SENT) ? M3UA_ASP_INACTIVE : M3UA_ASP_DOWN;

		if (link->state != M3UA_ASP_DOWN)  { /* send a notification and wait. */
			char buf[32];
			int len = sprintf(buf, "%d", link->fd);
			add_tlv(p, 0x0004, buf, len);
			msg = (link->state == M3UA_ASP_ACTIVE) ? M3UA_ASPIA_MSG : M3UA_ASPDN_MSG;
			link->state = (link->state == M3UA_ASP_ACTIVE) ? M3UA_ASP_IA_SENT : M3UA_ASP_DOWN_SENT;
			m->size =  p - q;
		}  else 
			link->state = M3UA_ASP_DOWN;
		if (msg >= 0)  {
			int res = m3ua_transmit_int(link->master, link, msg, link->master->sls, m);

			if (res > 0) 
				link->down_ctr++; /* Keep count of how many times we've sent this */
									
			if (res < 0)
				link->state = M3UA_ASP_DOWN;
			else if (link->state != M3UA_ASP_DOWN)
				link->t3 = ss7_schedule_event(link->master, DEFAULT_RETRY_TIMEOUT*2, 
							      (void *)down_link, link);
		} else if (m)
			free(m);
	}

free_link:
	if (link->state == M3UA_ASP_DOWN) /* bring it down */
		remove_link(link); 
}

static void transmit_aspup(void *x)
{
	struct m3ua *link = x;
	struct ss7_msg *m = ss7_msg_new();
	u_int32_t aspid = link->fd;
	unsigned char *p, *q;
	int res;
     
	if (!m) {
		ss7_error(link->master, "Allocation failed!\n");
		return;
	}
	p = q = m->buf + M3UA_COMMON_HLEN;
	add_tlv(p, 0x0011, &aspid, sizeof(aspid));

	m->size = p - q;     
	res = m3ua_transmit_int(link->master, link, M3UA_ASPUP_MSG, 1, m);
     
	if (res > 0) {
		link->state = M3UA_ASP_UP_SENT;
		/* Set retry timer. */
		link->t1 = ss7_schedule_event(link->master, DEFAULT_RETRY_TIMEOUT, (void *)transmit_aspup, link);

	} else if (res < 0 && errno != EAGAIN && errno != EINTR) { /* close the connection */
		link->t1 = -1;
		down_link(link);
	}	  
}


static void transmit_aspac(void *x)
{
	struct m3ua *link = x;
	struct ss7_msg *m;
	char aspid[32];
	unsigned char *p, *q;
	int res, len;
     
	if (!link || link->state != M3UA_ASP_UP)
		return;
	m = ss7_msg_new();
	if (!m) {
		ss7_error(link->master, "Allocation failed!\n");
		return;
	}
	len = sprintf(aspid, "%d", link->fd);
	
	p = q = m->buf + M3UA_COMMON_HLEN;
	add_tlv(p, 0x0004, aspid, len);

	m->size = p - q;     
	res = m3ua_transmit_int(link->master, link, M3UA_ASPAC_MSG, 1, m);
     
	if (res > 0) {
		link->state = M3UA_ASP_AC_SENT;
		/* Set retry timer. */
		link->t2 = ss7_schedule_event(link->master, DEFAULT_RETRY_TIMEOUT, (void *)transmit_aspac, link);

	} else if (res < 0 && errno != EAGAIN && errno != EINTR) { /* close the connection */
		link->t2 = -1;
		down_link(link); /* try and bring it down gracefully */
	}	  
}


void m3ua_start(struct ss7 *ss7)
{
	int i;
	for (i = 0; i<ss7->num_m3ua_links; i++)
		if (ss7->m3ua_links[i]->state == M3UA_ASP_DOWN) 
			transmit_aspup(ss7->m3ua_links[i]);
}


#define NEXT_TAG(tptr) ROUNDUP(ntohs((tptr)[1]), 4)/2 /* round up length, then jump over tlv. */
static inline void *find_param(void *v, int tag, int lim, int *dlen)
{
	unsigned char *p = v;
	u_int16_t *u = (void *)p;

	while ((void *)u < (void *)(p + lim))
		if (ntohs(u[0]) == tag) { /* Data element */
			*dlen = ntohs(u[1]) - 2*sizeof u[0]; /* Remove length of TL part */
			return u + 2;				
		} else 
			u += NEXT_TAG(u); /* Skip TLV */
	return NULL;
}

static int transmit_rkm_reg(struct m3ua *link);
static int net_msg_recv(struct ss7*, struct m3ua *, struct m3ua_header  *, int len, char errbuf[]);
int m3ua_receive(struct m3ua *link)
{
	struct ss7 *ss7 = link->master;	
	unsigned char buf[1024];	
	int flags = MSG_WAITALL, len, ppid, streamid;
	
#ifdef WITH_SCTP
	struct sockaddr addr;
	socklen_t alen = sizeof addr;
	struct sctp_sndrcvinfo sinfo;
	len = sctp_recvmsg(link->fd, buf, sizeof buf, &addr,
			   &alen, 
			   &sinfo, &flags);
	ppid = ntohl(sinfo.sinfo_ppid);
	streamid = ntohl(sinfo.sinfo_stream);

	if (len <= 0) { /* Close connection? If no associations  */
		uint32_t n = 0;
		socklen_t x = sizeof n;
		if (getsockopt(link->fd, IPPROTO_SCTP, 
			       SCTP_GET_ASSOC_NUMBER, &n, &x) == 0 && 
		    n<=0) {
			link->state = M3UA_ASP_DOWN;
			len = -1;
		}
	}
#else 
	len = -1;
	ppid = -1;
	ss7_error(ss7, "M3UA read: No SCTP support!\n");
#endif

	if (len <= 0) /* Nothing read */
		return len;
	
	if (flags & MSG_NOTIFICATION) {
#ifdef WITH_SCTP
		union sctp_notification *sn = (void *)buf;
		switch(sn->sn_header.sn_type) {
		case SCTP_SEND_FAILED: /* Pass correlation ID upwards */
		{
			int dlen;
			u_int32_t *i = find_param(sn->sn_send_failed.ssf_data, 0x0013, sn->sn_send_failed.ssf_length, &dlen);
			/* find correlation ID */
			if (i) {
				ss7_event *e = ss7_next_empty_event(ss7);
				if (e) {
					e->e = M3UA_SEND_FAILED;
					e->gen.data = ntohl(i[0]);
				}
			}
		}
		break;
		case SCTP_SHUTDOWN_EVENT: /* we should do something if this is the last assoc. right? */
			break;
		default: 
			
			break; /* do nothing. */			
		}	
#endif	
		return 0;
	} else if ((flags & MSG_EOR) &&  
		   len >= M3UA_COMMON_HLEN
#if 1
		   && ppid == M3UA_PPID
#endif
		)  { /* Data */
		struct m3ua_header *hdr = (void *)buf;
		int err_code = -1;
		char errbuf[64];
			
		errbuf[0] = 0;
		if (hdr->ver != 0x01)  /* unsupported version. */
			err_code = 0x01;
		else if (hdr->msg_class == M3UA_TRANSFER_CLASS) {

			if (hdr->msg_type != 0x01) 
				err_code = 0x04;
			else { /* Find the protocol data element */				
				int dlen = 0;			
				struct m3ua_data_header *dhr;
				
				dhr = find_param(hdr + 1, 0x0210, len - sizeof hdr[0], &dlen);
				if (dhr) {
					struct routing_label rl = {.opc = ntohl(dhr->opc), 
								   .dpc = ntohl(dhr->dpc), 
								   .sls = dhr->sls};
					int ecode = 0;

					dlen -= sizeof dhr[0];
					switch(dhr->si) {
					case SIG_ISUP:
						ecode = isup_receive(ss7, &rl, (void *)(dhr + 1), dlen);
						break;
					case SIG_SCCP:
					default:
						if (ss7_userpart_receive(ss7, rl, dhr->si, dhr + 1, dlen) < 0)
							ecode = 0x06;
						break;
					}
					if (ecode < 0) { 
						err_code = 0x07; /* Protocol error */
						sprintf(errbuf, "M3UA User 0x%02X returned error: %d", dhr->si, ecode);
					}
				} else {  /* Failed to decode message */
					sprintf(errbuf, "Missing data param in TRANSFER message");	
					err_code = 0x16; /* Missing parameter (data) */							
				}
			}
		} else if (hdr->msg_class == M3UA_MGMT_CLASS ||
			   hdr->msg_class == M3UA_ASPSM_CLASS ||
			   hdr->msg_class == M3UA_ASPTM_CLASS ||
			   hdr->msg_class == M3UA_SSNM_CLASS ||
			   hdr->msg_class == M3UA_RKM_CLASS)
			err_code = net_msg_recv(ss7, link, hdr, len, errbuf);
		else 
			err_code = 0x03;

		if (err_code > 0) {
			struct ss7_msg *m = ss7_msg_new();
			if (m) {
				unsigned char *p = m->buf + M3UA_COMMON_HLEN, *q = p;
				u_int32_t err = htonl(err_code);
				int sl = strlen(errbuf);
				
				add_tlv(p, 0x000c, &err, sizeof err);				
				if (sl > 0) 
					add_tlv(p, 0x0007, errbuf, sl);				
				m->size = p - q;
				return m3ua_transmit_int(ss7, link, M3UA_ERR_MSG, streamid,m);
			} else 
				ss7_error(ss7, "M3UA read: Allocation failed!\n");
			return -1;
		} else 
			return err_code; /* no errors */
	} else {
		ss7_error(ss7, "M3UA read: Unexpected message, skipped!\n");
		return -1;
	}
		
}

static inline struct m3ua *find_m3ua_link(struct ss7 *ss7, int fd)
{
	int i;
	for (i = 0; i<ss7->num_m3ua_links; i++)
		if (ss7->m3ua_links[i]  && ss7->m3ua_links[i]->fd == fd)
			return ss7->m3ua_links[i];
	return NULL;
}

static const char *m3ua_errors[] = {
	[0x01] = "Invalid Version", 
	[0x02] = "Not Used in M3UA", 
	[0x03] = "Unsupported Message Class", 
	[0x04] = "Unsupported Message Type", 
	[0x05] = "Unsupported Traffic Mode Type", 
	[0x06] = "Unexpected Message", 
	[0x07] = "Protocol Error", 
	[0x08] = "Not Used in M3UA", 
	[0x09] = "Invalid Stream Identifier", 
	[0x0a] = "Not Used in M3UA", 
	[0x0b] = "Not Used in M3UA", 
	[0x0c] = "Not Used in M3UA", 
	[0x0d] = "Refused - Management Blocking", 
	[0x0e] = "ASP Identifier Required", 
	[0x0f] = "Invalid ASP Identifier", 
	[0x10] = "Not Used in M3UA", 
	[0x11] = "Invalid Parameter Value", 
	[0x12] = "Parameter Field Error", 
	[0x13] = "Unexpected Parameter", 
	[0x14] = "Destination Status Unknown", 
	[0x15] = "Invalid Network Appearance", 
	[0x16] = "Missing Parameter", 
	[0x17] = "Not Used in M3UA", 
	[0x18] = "Not Used in M3UA", 
	[0x19] = "Invalid Routing Context", 
	[0x1a] = "No Configured AS for ASP", 
};

static int net_msg_recv(struct ss7 *ss7, struct m3ua *link, struct m3ua_header *hdr, int len, char errbuf[])
{
	int *i, dlen = 0;
	unsigned char *s;
	struct ss7_msg *m;
	int type = M3UA_MSG(hdr->msg_class,hdr->msg_type);
	
	switch (type) {
	case M3UA_ASPUP_ACK_MSG:
		if ((i = find_param(hdr + 1, 0x0011, len - sizeof hdr[0], &dlen)) != NULL) { /* Find ASP ID */
			link = find_m3ua_link(ss7, *i);

			if (!link) 
				return 0x07; /* protocol error: no such ASP */
			
		} else if (link->state > M3UA_ASP_ACTIVE)
			return 0x06;
		if (link->state == M3UA_ASP_UP_SENT) {
			ss7_schedule_del(ss7, &link->t1);
			link->state = M3UA_ASP_UP;  /* We are officially UP but not active */
			if (link->keys) 
				transmit_rkm_reg(link); /* Send reg requests, wait for response before we go active*/
			else 
				transmit_aspac(link);
		}  /* else ignore (possible duplicate) */		
		break;
	case M3UA_ASPAC_ACK_MSG:
		if ((s = find_param(hdr + 1, 0x0004, len - sizeof hdr[0], &dlen)) != NULL) { /* Find ASP ID */
			struct m3ua *xlink = link;
			int fd = -1; 
			
			sscanf((void *)s, "%d", &fd);

			xlink = find_m3ua_link(ss7, fd);
			if (xlink) 
			        link = xlink; /* Assume it's broken and use current? XXX */
		}
		if (link->state == M3UA_ASP_AC_SENT) {
			ss7_event *e;
			
			ss7_schedule_del(ss7, &link->t2);
			link->state = M3UA_ASP_ACTIVE; /* Success! */
			
			e = ss7_next_empty_event(ss7);
			if (e) {
				e->e = M3UA_LINK_UP;
				e->gen.data = link->fd;
			}
		} else if (link->state < M3UA_ASP_AC_SENT)
			return 0x06; /* unexpected message */
		break;
	case M3UA_ASPIA_ACK_MSG:
		if ((s = find_param(hdr + 1, 0x0004, len - sizeof hdr[0], &dlen)) != NULL) { /* Find ASP ID */
			struct m3ua *xlink = link;
			int fd = -1; 
			
			sscanf((void *)s, "%d", &fd);
			
			xlink = find_m3ua_link(ss7, fd);
			if (xlink) 
			        link = xlink; /* Assume it's broken and use current? XXX */
		}
		
		if (link->state == M3UA_ASP_IA_SENT) {  /* process of going down, complete it */
			link->state = M3UA_ASP_INACTIVE;
			link->down_ctr = 0; /* reset count */
			down_link(link);		
		} else if (link->state < M3UA_ASP_IA_SENT)
			return 0x06; /* unexpected */
		break;
	case M3UA_BEAT_MSG: /* Shouldn't be ordinarily received when using SCTP, but lets handle it all the same */
		if ((m = ss7_msg_new()) != NULL) {
			int dlen = 0;
			unsigned char *p = m->buf  + M3UA_COMMON_HLEN, *q = p;
			unsigned char *h = find_param(hdr + 1, 0x0009, len - sizeof hdr[0], &dlen);
			
			if (h) 
				add_tlv(p, 0x0009, h, dlen);
			m->size = p - q;
			
			m3ua_transmit_int(ss7, link, M3UA_BEAT_ACK_MSG, 1, m);			
		} else 
			ss7_error(ss7, "M3UA read: Allocation failed!\n");
		break;		
	case M3UA_ERR_MSG:
		if ((i = find_param(hdr + 1, 0x000c, len - sizeof hdr[0], &dlen)) != NULL) { /* Find error ID */
			int id = ntohl(*i);
			char buf[256];
			
			sprintf(buf, "M3UA read: Received  Error packet: %02X => %.100s!\n",
				id, 
				(id >= 0 && id < (sizeof m3ua_errors) / sizeof m3ua_errors[0])  ?
				m3ua_errors[id] : "n/a");
			ss7_error(ss7, buf);
		}  		     
		break;
	case M3UA_SCON_MSG:
		/* Find congestion value */ 
		if ((s = find_param(hdr + 1, 0x0205, len - sizeof hdr[0], &dlen)) != NULL) {
			u_int8_t con = s[3]; 

			
			/* find affected PCs, handle appropriately. */		
			s = (void *)(hdr + 1);
			while ((s = find_param(s, 0x0012, len - (s - (unsigned char *)hdr), &dlen)) != NULL) {
				u_int8_t mask = s[0];
				u_int32_t afpc, inv_mask; 
				struct m3ua_congestion **c, *new;
				
				s[0] = 0; /* clear it before we convert  affected pc */
				afpc = ntohl(*(u_int32_t *)s);
				s[0] = mask; /* really necessary ?? */
				
				if (ss7->switchtype == SS7_ANSI && mask > 24) 
					mask = 24;
				else if (ss7->switchtype == SS7_ITU && mask > 14) 
					mask = 14;
				/* invert mask: That is, top bits a 1, wildcarded ones zero. This will be ANDed... */
				inv_mask = ~((1U<<mask)-1); 
				
				/* Look for previously received congestion levels that are superceded by this one */
				for (c = &link->con; *c; ) 
					if (COMP_PC(afpc,(*c)->pc,inv_mask) == 0 && 
					    inv_mask <= (*c)->mask) { /* i.e. new one is at least as general as old one */
						struct m3ua_congestion *x = (*c);
						(*c) = (*c)->next;
						free(x);              /* remove it */
					} else 
						c = &(*c)->next;
				/* Add a new one */
				new = calloc(1, sizeof new[0]);
				new->pc = afpc;
				new->mask = inv_mask;
				new->con = con;
				new->next = link->con;
				link->con = new;
				
				s += sizeof(u_int32_t); /* Skip over value */
			}
		} else 
			return 0x16; /* missing parameter */
		break;
	case M3UA_DUNA_MSG:
	case M3UA_DRST_MSG:
	case M3UA_DAVA_MSG: /* handled the same, save for availability flag */
		s = find_param(hdr + 1, 0x0012, len - sizeof hdr[0], &dlen);
		if (s)
			while (dlen > 0) {
				u_int8_t mask = s[0];
				u_int32_t afpc, inv_mask; 
				struct m3ua_route **r, *new;
				
				s[0] = 0; /* clear it before we convert  affected pc */
				afpc = ntohl(*(u_int32_t *)s);
				s[0] = mask; /* really necessary ?? */
				
				if (ss7->switchtype == SS7_ANSI && mask > 24) 
					mask = 24;
				else if (ss7->switchtype == SS7_ITU && mask > 14) 
					mask = 14;
				/* invert mask: That is, top bits a 1, wildcarded ones zero. This will be ANDed... */
				inv_mask = ~((1U<<mask)-1); 
				
				/* Look for previously received indications that are superceded by this one, remove them */
				for (r = &link->routes; *r; ) 
					if (COMP_PC(afpc,(*r)->pc,inv_mask) == 0 && 
					    inv_mask <= (*r)->mask) { /* i.e. new one is at least as general as old one */
						struct m3ua_route *x = (*r);
						(*r) = (*r)->next;
						free(x);             
					} else 
						r = &(*r)->next;
				/* Add a new one */
				new = calloc(1, sizeof new[0]);
				new->pc = afpc;
				new->mask = inv_mask;
				new->avail = (type == M3UA_DAVA_MSG) ? 1 : 0;
				new->next = link->routes;
				link->routes = new;
				
				dlen -= sizeof(u_int32_t); /* Skip over value */
				s += sizeof (u_int32_t);
			}
		
		break;
	case M3UA_DUPU_MSG:
		if ((i = find_param(hdr + 1, 0x0012, len - sizeof hdr[0], &dlen)) != NULL) {
			u_int32_t afpc = ntohl(*i);
			u_int16_t *u = find_param(hdr + 1, 0x0204, len - sizeof hdr[0], &dlen);
			ss7_event *e;
			
			if (!u)
				return 0x16;
			
			e = ss7_next_empty_event(ss7);
			if (e) {
				u_int8_t userpart = (u_int8_t)ntohs(u[1]); 
				/* ignore cause */
				e->e = M3UA_DUPU;
				e->gen.arg  =  userpart; 
				e->gen.data = afpc;				
			} 
		} else 
			return 0x16;
		break;
	case M3UA_RKM_REG_RSP_MSG:
		s = (void *)(hdr + 1);
		while ((s = find_param(s, 0x0208, len - (s - (unsigned char *)hdr), &dlen)) != NULL) {
			int x;
			u_int32_t *i = find_param(s, 0x020a, dlen, &x); /* find local identifier */
			u_int32_t status, rc, rki = i ? *i : 0;
			
			
			if ((i = find_param(s, 0x0212, dlen, &x)) != NULL)
				status = ntohl(*i);
			else 
				status = 1;

			if ((i = find_param(s, 0x006, dlen, &x)) != NULL)
				rc = ntohl(*i);
			else 
				rc = 0;
			if (rki) {
				ss7_event *e;
				struct m3ua_routing_key **l = &link->keys, *x;
				int found = 0;
				while (*l) {
					if ((*l)->rk.rki == rki) {
						if (status == 0) {
							(*l)->active = 1;
							(*l)->rc = rc;
							found = 1;
						} else { /* Delete it */
							x = *l;
							*l = (*l)->next; 
							free(x);
							continue;
						}
					} 
					l = &(*l)->next;
				}

				 /* At least one routing context, lets send ACTIVE ind. */
				if (found && status == 0 && link->state == M3UA_ASP_UP)
					transmit_aspac(link);
				
				/* Inform upper level */
				e = ss7_next_empty_event(ss7);
				if (e) {
					e->e = M3UA_ROUTE_REG_RSP;
					e->gen.arg = rki;
					e->gen.data = status;
				}
			}
			s += dlen;
		}
		break;
	case M3UA_RKM_DEREG_RSP_MSG:
		s = (void *)(hdr + 1);
		while ((s = find_param(s, 0x0209, len - (s - (unsigned char *)hdr), &dlen)) != NULL) {
			int x;
			u_int32_t *i = find_param(s, 0x0006, dlen, &x); /* find RC */
			u_int32_t status, rc = i ? ntohl(*i) : 0, rki = 0;
			ss7_event *e;
			struct m3ua_routing_key **l = &link->keys;

			if ((i = find_param(s, 0x0213, dlen, &x)) != NULL)
				status = ntohl(*i);
			else 
				status = 1;
			
			while (*l)  { /* Find the corresponding key, remove it. */
				if ((*l)->rc == rc) {
					struct m3ua_routing_key *x = *l;
					rki = x->rk.rki;
					*l = (*l)->next;

					continue;
				}

				l = &(*l)->next;
			}
			
                        /* Inform upper level */
			e = ss7_next_empty_event(ss7);
			if (e) {
				e->e = M3UA_ROUTE_DEREG_RSP;
				e->gen.arg = rki;
				e->gen.data = status;
			}
		}
		break;

	case M3UA_ASPDN_ACK_MSG:
		if ((s = find_param(hdr + 1, 0x0004, len - sizeof hdr[0], &dlen)) != NULL) { /* Find ASP ID */
			struct m3ua *xlink = link;
			int fd = -1; 
			
			sscanf((void *)s, "%d", &fd);
			
			xlink = find_m3ua_link(ss7, fd);
			if (xlink) 
			        link = xlink; /* Assume it's broken and use current? XXX */
		}
		
		if (link->state == M3UA_ASP_DOWN_SENT) {  /* process of going down, complete it */
			link->state = M3UA_ASP_DOWN;
			link->down_ctr = 0; /* reset count */
			down_link(link);		
		}
		break;

	case M3UA_BEAT_ACK_MSG:
	case M3UA_NTY_MSG:
	case M3UA_ASPUP_MSG:
	case M3UA_ASPDN_MSG:
	case M3UA_ASPAC_MSG:
	case M3UA_ASPIA_MSG:
	case M3UA_RKM_REG_REQ_MSG:	       
	case M3UA_RKM_DEREG_REQ_MSG:	       
		return 0; /* Ignore */		
		break;
	default:
		return 0x04; /* unsupported message. */
		break;
	}
	return 0;
}

/* Computes length of routing key, so we don't overflow message buffer */
static inline int sizeof_rk(struct m3ua_routing_key_req *r)
{
	int len = 8 + 8; /* Local RK-Id value length + dpc length*/

	if (r->traffic_mode >= 1 && r->traffic_mode <= 3) 
		len += 8;
	if (r->net_app >= 0)
		len += 8;
	if (r->si_list.n > 0) 
		len += ROUNDUP(r->si_list.n,4) + 4;
	if (r->opc_list.n > 0)
		len += 4 + r->opc_list.n*4;
	
	if (r->cic_list.n > 0)
		len += r->cic_list.n*8 + 4;
	return len;
}

static int transmit_rkm_reg(struct m3ua *link)
{
	struct m3ua_routing_key *rk;
	struct ss7 *ss7 = link->master;

	if (link->state == M3UA_ASP_DOWN ||
	    link->state == M3UA_ASP_IA_SENT || 
	    link->state == M3UA_ASP_DOWN_SENT)
		return -1;
	if (link->state != M3UA_ASP_UP && link->state != M3UA_ASP_ACTIVE)
		return 0;
	
	
	for (rk = link->keys; rk; rk = rk->next)
		if (rk->sent == 0)  {
			struct m3ua_routing_key_req rk1 = rk->rk;
			u_int16_t *u, rlen;
			unsigned char *s; 
			u_int32_t x;
			struct ss7_msg *m;
			unsigned char *q;
			int res;
			
			m = ss7_msg_new();
			if (!m) {
				ss7_error(ss7, "M3UA send rkm: Allocation failed!\n");
				return -1;
			}
			q = m->buf + M3UA_COMMON_HLEN;
			u = (void *)q;
			s = q + 2*sizeof u[0]; /* leave space for Routing key T & L */

			add_tlv(s, 0x020a, &rk1.rki, sizeof rk1.rki);
			
			if (rk1.traffic_mode > 0  && rk1.traffic_mode <= 3) {
				rk1.traffic_mode = htonl(rk1.traffic_mode);
				add_tlv(s, 0x000b, &rk1.traffic_mode, 4);	
			}		
		
			x = htonl(ss7->pc);  /* add destination as our PC */
			add_tlv(s, 0x020b, &x, sizeof x);  
			if (rk1.net_app>0) {
				rk1.net_app = htonl(rk1.net_app);
				add_tlv(s, 0x0200, &rk1.net_app, 4);	
			}
			if (rk1.si_list.n > 0)
				add_tlv(s, 0x020c, rk1.si_list.si, rk1.si_list.n);

			if (rk1.opc_list.n > 0) {
				int j;
				for (j = 0; j<rk1.opc_list.n; j++) /* re-order */
					rk1.opc_list.pc[j]= htonl(rk1.opc_list.pc[j]);
				add_tlv(s, 0x020e, rk1.opc_list.pc, rk1.opc_list.n*4);
			}
			if (rk1.cic_list.n > 0) {
				int j;
				for (j = 0; j<rk1.cic_list.n; j++) { /* re-order */
					rk1.cic_list.range[j].opc = htonl(rk1.cic_list.range[j].opc);
					rk1.cic_list.range[j].lower = htons(rk1.cic_list.range[j].lower); 
					rk1.cic_list.range[j].upper = htons(rk1.cic_list.range[j].upper); 
				}
				add_tlv(s, 0x020f, rk1.cic_list.range, rk1.cic_list.n*8);
			}
		
			rlen = s - q; /* computing routing key length */
			u[0] = htons(0x0207);
			u[1] = htons(rlen); 
			
			m->size = rlen;

			res = m3ua_transmit_int(ss7, link, M3UA_RKM_REG_REQ_MSG, 1, m);
			if (res > 0)
				rk->sent = 1;
		} 
	return 0;
}

int m3ua_rkm_reg(struct ss7 *ss7,  struct routing_label rl, 
		 struct m3ua_routing_key_req rk[], int num_keys)
{
	struct m3ua *link = rl_to_m3ua_no_check(ss7, rl);
	int i;
	
	if (!link || !rk) 
		return -1;	

	for (i = 0; i < num_keys; i++) 
		if (sizeof_rk(&rk[i]) < M3UA_COMMON_HLEN + 4 + sizeof ((struct ss7_msg *)0)->buf) {	
			struct m3ua_routing_key *r;
			
			r = calloc(1, sizeof r[0]);
			if (!r) {
				ss7_error(ss7, "M3UA rkm reg: Allocation failed!\n");
				return -1;
			}
			r->rk = rk[i];
			r->rk.rki = rk[i].rki = (u_int32_t)r; 
			r->active = 0;
			r->sent = 0;
			
			r->next = link->keys; /* put it in */
			link->keys = r;
		} else 
			rk[i].rki = 0;
	
	return transmit_rkm_reg(link);
}

int m3ua_rkm_dereg(struct ss7 *ss7, struct routing_label rl, u_int32_t rki[], int num_keys)
{
	struct m3ua_routing_key *kl;
	struct m3ua *link = rl_to_m3ua_no_check(ss7, rl);
	struct ss7_msg *m;
	unsigned char *p;
	u_int16_t *s;
	u_int32_t *u;
	int i;
	
	if (!link) 
		return -1;
	
	m = ss7_msg_new();
	if (!m) 
		return -1;
	p = m->buf + M3UA_COMMON_HLEN;
	s = (void *)p;
	for (i = 0, u = (void *)(p+4); 	     
	     i<num_keys && (unsigned char *)u < m->buf + sizeof m->buf; 
	     i++) 
		for (kl = link->keys; kl; kl = kl->next) 
			if (kl->rk.rki == rki[i]) { /* Find the key, use the routin context from it. */
				*u = htonl(kl->rc);
				u++;
			}

	m->size = 4 + (unsigned char *)u - p;

	s[0] = htons(0x0006);
	s[1] = htons(m->size);
	
	return m3ua_transmit_int(ss7, link, M3UA_RKM_DEREG_REQ_MSG, 1, m);
}

void m3ua_down_link(struct ss7 *ss7, int fd)
{
	struct m3ua *link = find_m3ua_link(ss7, fd);

	if (link)
		down_link(link);
}

int m3ua_daud(struct ss7 *ss7, struct routing_label rl, int na, int rc, int dpc, char *info)
{
	struct ss7_msg *m;
	struct m3ua *link = rl_to_m3ua_no_check(ss7, rl);
	unsigned char *p, *s;
	u_int32_t  x;

	
	if (!link) 
		return -1;

	m = ss7_msg_new();
	if (!m) 
		return -1;
	p = s = m->buf + M3UA_COMMON_HLEN;
	
	if (na >= 0) {
		x = htonl(na);
		add_tlv(s, 0x0200, &x, sizeof x);
	}
	if (rc >= 0) {
		x = htonl(rc);
		add_tlv(s, 0x0006, &x, sizeof x);
	}
	x = htonl(dpc);	
	add_tlv(s, 0x0012, &x, sizeof x);
	if (info)
		add_tlv(s, 0x0004, info, strlen(info));
	m->size = s - p;

	return m3ua_transmit_int(ss7, link, M3UA_DAUD_MSG, 1, m);
}
