/*
 * libss7: An implementation of Signalling System 7
 *
 * Written by Paul Bagyenda <bagyenda@dsmagic.com>
 *
 * 
 *
 * Copyright (C) 2009-, Digital Solutions
 * All Rights Reserved.
 */

/*
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2 as published by the
 * Free Software Foundation. See the LICENSE file included with
 * this program for more details.
 *
 * In addition, when this program is distributed with Asterisk in
 * any form that would qualify as a 'combined work' or as a
 * 'derivative work' (but not mere aggregation), you can redistribute
 * and/or modify the combination under the terms of the license
 * provided with that copy of Asterisk, instead of the license
 * terms granted here.
 */

#include <sys/time.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <pthread.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <arpa/inet.h>


#include "libss7.h"
#include "ss7_internal.h"
#include "m3ua.h"


#ifdef WITH_SCTP
#include <netinet/sctp.h>
#endif

#define MAX_ADDRESSES 100
static int  opc, dpc, linknum = 1;
static int callcount = 0;
struct linkset {
	struct ss7 *ss7;
	int linkno;
	int fd;
} linkset[2];


struct m3ua_routing_key_req rk  = {
	.traffic_mode = 2,
	.net_app = -1,
	.si_list = {
		.n = 3,
		.si = {3,5}
	},
	.opc_list = {
		.n = 5,
		.pc = {1,2,3,4,5}
	},
	.cic_list = {
		.n = 2,
		.range = { 
			{.opc = 1,
			 .lower = 1,
			 .upper = 2
			}, 
			{.opc = 2,
			 .lower = 3,
			 .upper = 4
			}}
	}
};

static void myprintf(struct ss7 *ss7, char *fmt)
{
        //int i = 0;
        printf("%s", fmt);
}



static void print_args(void)
{

	printf("Incorrect arguments.  Should be:\n");
        printf("m3uatest dest-ip[:port][,dest-ip2[:port] ...] [OPC - in decimal] [DPC - in decimal]\n");
        printf("Example:\n");
        printf("m3uatest 127.0.0.1:2905 1 2\n");
        printf("This would run the test program, connecting to the SGP on port 2905 at localhost with an OPC of 1 and a DPC of 2\n");
}

static void ss7_call(struct ss7 *ss7)
{
	struct isup_call *c;

	c = isup_new_call(ss7);

	if (c) {
		isup_set_called(c, "12345", SS7_NAI_NATIONAL, ss7);
		isup_set_calling(c, "7654321", SS7_NAI_NATIONAL, SS7_PRESENTATION_ALLOWED, SS7_SCREENING_USER_PROVIDED);
		isup_init_call(ss7, c, (callcount % 12) + 1, dpc);
		isup_iam(ss7, c);
		printf("Callcount = %d\n ", ++callcount);
	}
}


static void *ss7_run(void *data)
{
	struct linkset *linkset = data;
	struct ss7 *ss7 = linkset->ss7;
	int res = 0;
	struct timeval *next = NULL, tv;
	ss7_event *e = NULL;
	struct pollfd poller;
	int nextms = 2000;
	struct isup_call *c;
	unsigned char state[255] =  {0};
	
	struct routing_label rl = {.type = ss7->switchtype, 
				   .dpc = dpc, .opc = opc, .sls=1};
	u_int32_t rkm[] = {rk.rki};
	
	printf("Starting link %d\n", ++linknum);
	ss7_start(ss7);

	while(1) {
		if ((next = ss7_schedule_next(ss7))) {
			gettimeofday(&tv, NULL);
			tv.tv_sec = next->tv_sec - tv.tv_sec;
			tv.tv_usec = next->tv_usec - tv.tv_usec;
			if (tv.tv_usec < 0) {
				tv.tv_usec += 1000000;
				tv.tv_sec -= 1;
			}
			if (tv.tv_sec < 0) {
				tv.tv_sec = 0;
				tv.tv_usec = 0;
			}
			nextms = tv.tv_sec * 1000;
			nextms += tv.tv_usec / 1000;
		}
		poller.fd = linkset->fd;
		poller.events = ss7_pollflags(ss7, linkset->fd);
		poller.revents = 0;

		res = poll(&poller, 1, nextms);
		if (res < 0) {
#if 0
			printf("next->tv_sec = %d\n", next->tv_sec);
			printf("next->tv_usec = %d\n", next->tv_usec);
			printf("tv->tv_sec = %d\n", tv.tv_sec);
			printf("tv->tv_usec = %d\n", tv.tv_usec);
#endif
			perror("select");
		}
		else if (!res) {
			ss7_schedule_run(ss7);
			continue;
		}
#if 0
		if (poller.revents & POLLPRI) {
			if (ioctl(linkset->fd, DAHDI_GETEVENT, &x)) {
				perror("Error in exception retrieval!\n");
			}
			switch (x) {
			case DAHDI_EVENT_OVERRUN:
				printf("Overrun detected!\n");
				break;
			case DAHDI_EVENT_BADFCS:
				printf("Bad FCS!\n");
				break;
			case DAHDI_EVENT_ABORT:
				printf("HDLC Abort!\n");
				break;
			default:
				printf("Got exception %d!\n", x);
				break;
			}
		}
#endif
		if (poller.revents & POLLIN)
			res = ss7_read(ss7, linkset->fd);
		if (poller.revents & POLLOUT) {
			res = ss7_write(ss7, linkset->fd);
			if (res < 0) {
				printf("Error in write %s", strerror(errno));
			}
		}

#if 1
		if (res < 0)
			m3ua_down_link(ss7, linkset->fd);
#endif

		while ((e = ss7_check_event(ss7))) {
			if (e) {
				switch (e->e) {
				case SS7_EVENT_UP:
					printf("[%d] --- SS7 Up ---\n", linkset->linkno);
					//isup_grs(ss7, 1, 24, dpc);
					c = isup_new_call(ss7);
					isup_init_call(ss7, c, 1, dpc);
					isup_grs(ss7, c, 24);
					break;
				case MTP2_LINK_UP:
					printf("[%d] MTP2 link up\n", linkset->linkno);
					break;
				case M3UA_LINK_UP:
					printf("[%d] M3UA link up\n", linkset->linkno);
					// m3ua_daud(ss7,rl, -1, 1, 2, NULL);
					// ss7_call(ss7);
					// m3ua_down_link(ss7, linkset->fd);

					break;
				case M3UA_LINK_DOWN:
					printf("[%d] M3UA link down\n", linkset->linkno);
					break;
				case M3UA_ROUTE_REG_RSP:
					m3ua_rkm_dereg(ss7, rl, rkm, 1);
					break;
				case ISUP_EVENT_GRS:
					printf("Got GRS from cic %d to %d: Acknowledging\n", e->grs.startcic, e->grs.endcic);
					isup_gra(ss7, e->grs.call, e->grs.endcic, state);
					break;
				case ISUP_EVENT_RSC:
					isup_rlc(ss7, e->rsc.call);
					break;
				case ISUP_EVENT_GRA:
					printf("Got GRA from cic %d to %d.\n", e->gra.startcic, e->gra.endcic);
					ss7_call(ss7);
					break;
				case ISUP_EVENT_BLO:
					isup_bla(ss7, e->blo.call);
					break;
				case ISUP_EVENT_CGB:
					isup_cgba(ss7, e->cgb.call, e->cgb.endcic, e->cgb.status);
					break;
				case ISUP_EVENT_CGU:
					isup_cgua(ss7, e->cgu.call, e->cgu.endcic, e->cgu.status);
					break;
				case ISUP_EVENT_IAM:
					printf("Got IAM for cic %d and number %s\n", e->iam.cic, e->iam.called_party_num);
					printf("CallerID is %s\n", e->iam.calling_party_num);
					printf("Sending ACM\n");
					isup_acm(ss7, e->iam.call);
					printf("Sending ANM\n");
					isup_anm(ss7, e->iam.call);
					break;
				case ISUP_EVENT_REL:
					printf("Got REL for cic %d\n", e->rel.cic);
					isup_rlc(ss7, e->rel.call);
					//ss7_call(ss7);
					break;
				case ISUP_EVENT_ACM:
					printf("Got ACM for cic %d\n", e->acm.cic);
					break;
				case ISUP_EVENT_ANM:
					printf("Got ANM for cic %d\n", e->anm.cic);
					isup_rel(ss7, e->anm.call, 16);
					break;
				case ISUP_EVENT_RLC:
					printf("Got RLC for cic %d\n", e->rlc.cic);
					//ss7_call(ss7);
					break;
				default:
					printf("Unknown event %d\n", e->e);
					break;
				}
			}
		}
	}
}
int dummy(void);
int main(int argc, char *argv[])
{
	struct sockaddr_in addrs[MAX_ADDRESSES];
	int ct = 0, sfd;
	pthread_t t;
	char *p;
	struct ss7 *ss7;

	if (argc < 4) {
		print_args();
		exit(-1);
	}
	
 
	opc = atoi(argv[2]);
	dpc = atoi(argv[3]);

	for (p = strtok(argv[1], ", "); 
	     p && ct < MAX_ADDRESSES; 
	     p = strtok(NULL, ", ")) {
		char *q;
		char hst[128] = {0};
		int port = 2905;
		
		strncpy(hst, p, -1 + sizeof hst);
		
		if ((q = strrchr(hst, ':')) != NULL) {
			*q++ = 0;
			port = atoi(q);
		}

		addrs[ct].sin_family = AF_INET;
		addrs[ct].sin_port = htons(port);

		if (inet_aton(hst, &addrs[ct].sin_addr) != 0) 
			ct++;
		else 
			fprintf(stderr, "%s: Invalid IP address %.128s, skipped!\n", argv[0], hst);		
	}
	     
	if (ct <= 0) {
		fprintf(stderr, "%s: No destinations!\n", argv[0]);		
		print_args();
		exit(-1);
	}
		
#ifdef WITH_SCTP
	sfd = socket(PF_INET, SOCK_STREAM, IPPROTO_SCTP);
	if (sfd < 0) {
		fprintf(stderr, "%s: Socket failed: %s!\n", argv[0], strerror(errno));			
		exit(-1);
	}
#if 0
	if (sctp_connectx(sfd, (void *)addrs, ct, NULL) < 0) {
		fprintf(stderr, "%s: Connect failed: %s!\n", argv[0], strerror(errno));			
		exit(-1);
	}
#else
	if (connect(sfd, (void *)addrs, sizeof addrs[0]) < 0) {
		fprintf(stderr, "%s: Connect failed: %s!\n", argv[0], strerror(errno));			
		exit(-1);
	}

#endif
#else
	fprintf(stderr, "%s: No SCTP support. Exiting!\n", argv[0]);			
	exit(-1);
#endif
	
	

	ss7 = ss7_new(SS7_ITU);
	if (!ss7) {
		perror("ss7_new");
		exit(1);
	}
	
	ss7_set_message(myprintf);
        ss7_set_error(myprintf);
        ss7_set_network_ind(ss7, SS7_NI_NAT);

        ss7_set_debug(ss7, 0xfffffff);
        if ((ss7_add_link(ss7,  SS7_TRANSPORT_M3UA, "test", 1, dpc))) {
                perror("ss7_add_link");
                exit(1);
        }

	ss7_set_link_fd(ss7, "test", sfd);
        ss7_set_pc(ss7, opc);
	
	ss7_set_call_null((void *)dummy);
#if 1 /* Key registration */
	{
		struct routing_label rl = {.type = ss7->switchtype, .dpc = dpc, .opc = opc, .sls = 1};
		m3ua_rkm_reg(ss7,rl, &rk, 1);
	}
#endif
	linkset[0].ss7 = ss7;
	linkset[0].fd = sfd;
	linkset[0].linkno = 0;

	if (pthread_create(&t, NULL, ss7_run, &linkset[0]) != 0) {
		perror("thread(0)");
		exit(1);
	}
		

	/* Wait for it to come up and send data. */
	pthread_join(t,NULL);

	return 0;
}

int dummy(void)
{
	return 0;
}
