/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.avalon.excalibur.collections.BucketMap;
import org.apache.avalon.excalibur.collections.FixedSizeBuffer;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.ComponentSelector;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.excalibur.event.Queue;
import org.apache.excalibur.event.command.Command;
import org.apache.excalibur.fortress.handler.ComponentHandler;
import org.apache.excalibur.fortress.util.ContextManager;
import org.apache.excalibur.fortress.util.ExcaliburRoleManager;
import org.apache.excalibur.fortress.util.RoleManager;
import org.apache.excalibur.mpool.PoolManager;

/**
 * The Container is an interface used to mark the Containers in your system.  It
 * exposes a protected getComponentManager() method so that the Container's
 * Manager can expose that to the instantiating class.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:proyal@apache.org">Peter Royal</a>
 * @author <a href="mailto:crafterm@apache.org">Marcus Crafter</a>
 * @version CVS $Revision: 1.34 $ $Date: 2002/05/13 12:17:39 $
 */
public abstract class AbstractContainer
    extends AbstractLogEnabled
    implements Contextualizable, Composable, Configurable, Initializable, Disposable, Container
{
    protected Context m_context;
    private ComponentManager m_manager;
    protected LoggerManager m_logManager;
    protected PoolManager m_poolManager;
    protected Queue m_commandQueue;
    protected ClassLoader m_classLoader;
    protected RoleManager m_roleManager;
    protected Configuration m_configuration;
    protected BucketMap m_configs = new BucketMap();
    protected BucketMap m_mapper = new BucketMap();
    protected List m_components = new ArrayList( 10 );
    protected Configuration m_markerManagerConfiguration;

    /**
     * Wrap this so that ComponentStateValidator functions properly.
     */
    public void enableLogging( Logger logger )
    {
        super.enableLogging( logger );
    }

    /**
     * Pull the manager items from the context so we can use them to set up the
     * system.
     */
    public void contextualize( Context containerContext )
        throws ContextException
    {
        m_context = containerContext;

        m_logManager = (LoggerManager)m_context.get( Container.LOGGER_MANAGER );
        m_poolManager = (PoolManager)m_context.get( Container.POOL_MANAGER );
        m_markerManagerConfiguration = (Configuration)m_context.get( ContextManager.MARKER_MANAGER_CONFIGURATION );

        try
        {
            m_classLoader = (ClassLoader)m_context.get( Container.CONTEXT_CLASSLOADER );
        }
        catch( ContextException ce )
        {
            m_classLoader = Thread.currentThread().getContextClassLoader();
        }

        try
        {
            m_commandQueue = (Queue)m_context.get( Container.COMMAND_QUEUE );
        }
        catch( ContextException ce )
        {
            m_commandQueue = null;
            getLogger().warn( "No Container.COMMAND_QUEUE is given, all management will be performed synchronously" );
        }

        try
        {
            m_roleManager = (RoleManager)m_context.get( Container.ROLE_MANAGER );
        }
        catch( ContextException ce )
        {
            m_roleManager = new ExcaliburRoleManager();
        }
    }

    /**
     * Process the configuration and set up the components and their mappings.
     * At this point, all components are prepared and all mappings are made.
     * However, nothing is initialized.
     *
     * <p>The native Configuration format follows a specific convention.  If you
     * use a RoleManager to map roles and implementations to more helpful names,
     * we will internally rewrite the configuration to match this format.  Please
     * note: If a configuration element does *not* have a unique id, it will not
     * be treated as a Component.  That ID is used as the hint when there is more
     * than one implementation of a role.</p>
     *
     * <pre>
     *   &lt;component role="org.apache.avalon.excalibur.datasource.DataSourceComponent"
     *                 id="default-connection"
     *                 class="org.apache.avalon.excalibur.datasource.JdbcDataSourceComponent"
     *                 handler="org.apache.excalibur.fortress.handler.ThreadSafeComponentHandler"&gt;
     *
     *    &lt;!-- Component specific configuration --&gt;
     *
     *  &lt;/component&gt;
     * </pre>
     */
    public void configure( Configuration configElement )
        throws ConfigurationException
    {
        m_configuration = configElement;

        Configuration elements[] = configElement.getChildren();

        for( int i = 0; i < elements.length; i++ )
        {
            Configuration component = null;
            Object hint = elements[ i ].getAttribute( "id", null );

            if( null != hint )
            {
                if( "component".equals( elements[ i ].getName() ) )
                {
                    component = elements[ i ];
                }
                else
                {
                    component = rewriteConfiguration( elements[ i ] );
                }

                addComponent( component, hint );
            }
        }
    }

    /**
     * Handles when a configuration name is used that is not "component", so it
     * makes it easier to handle ComponentSelector hierarchies.  It is meant to
     * either return a ComponentHandler or a ComponentSelector
     */
    protected Configuration rewriteConfiguration( final Configuration configItem )
        throws ConfigurationException
    {
        DefaultConfiguration temp = new DefaultConfiguration( "component", "AbstractContainer-rewrite" );
        Class klass = m_roleManager.getClassForName( configItem.getName() );
        Class handlerKlass = m_roleManager.getHandlerClassForClass( klass );
        String role = m_roleManager.getRoleForClass( klass );

        if( null == klass )
        {
            throw new ConfigurationException( "No class found matching configuration name " +
                                              "[name: " + configItem.getName() + ", location: " + configItem.getLocation() + "]" );
        }

        temp.setAttribute( "role", role );
        temp.setAttribute( "class", klass.getName() );
        temp.setAttribute( "handler", handlerKlass.getName() );

        final String id = configItem.getAttribute( "id", null );
        if( null != id )
        {
            temp.setAttribute( "id", id );
        }

        final String logger = configItem.getAttribute( "logger", null );
        if( null != logger )
        {
            temp.setAttribute( "logger", logger );
        }

        Configuration[] children = configItem.getChildren();
        for( int i = 0; i < children.length; i++ )
        {
            temp.addChild( children[ i ] );
        }

        temp.makeReadOnly();

        return temp;
    }

    /**
     * Add a Component.
     *
     */
    protected void addComponent( final Configuration component, Object hint )
    {
        if( null == hint )
        {
            throw new IllegalArgumentException( "Hint must not be null" );
        }

        String role = component.getAttribute( "role", null );
        String klass = component.getAttribute( "class", null );
        ComponentHandler handler = getComponentHandler(
            component.getAttribute( "handler", null ), klass, component
        );

        if( null != role && null != klass && null != handler )
        {
            BucketMap hintMap = (BucketMap)m_mapper.get( role );

            if( null == hintMap )
            {
                hintMap = new BucketMap();
            }

            hintMap.put( hint, handler );

            if( hintMap.containsKey( "default" ) )
            {
                if( !hintMap.containsKey( "selector" ) )
                {
                    hintMap.put( "selector", new ContainerComponentSelector( this, role ) );
                }
            }
            else
            {
                hintMap.put( "default", handler );
            }

            m_mapper.put( role, hintMap );
        }
    }

    /**
     * Get a ComponentHandler with the standard <code>HANDLER_CONSTRUCTOR</code>
     * for the component class passed in.
     */
    private ComponentHandler getComponentHandler( String handlerClassName,
                                                  String className,
                                                  Configuration configuration )
    {
        Constructor constructor;
        ComponentHandler handler = null;

        try
        {
            Class klass = m_classLoader.loadClass( className );
            Class handlerKlass = m_classLoader.loadClass( handlerClassName );
            constructor = handlerKlass.getConstructor( ComponentHandler.HANDLER_CONSTRUCTOR );
            handler = (ComponentHandler)constructor.newInstance( new Object[]{
                klass,
                configuration,
                new ContainerComponentManager( this, m_manager ),
                m_context
            } );
            handler.configure( m_markerManagerConfiguration );
        }
        catch( Exception e )
        {
            if( getLogger().isDebugEnabled() )
            {
                getLogger().debug( "Could not create the '" + handlerClassName +
                                   "' handler for the '" + className +
                                   "' component.", e );
            }
        }

        if( getLogger().isDebugEnabled() )
        {
            getLogger().debug( "Component " + className +
                               " uses handler " + handlerClassName );
        }

        m_configs.put( handler, configuration );
        m_components.add( handler );

        return handler;
    }

    /**
     * This is the method that the ContainerComponentManager and Selector use to gain
     * access to the ComponentHandlers and ComponentSelectors.  The actual access of
     * the ComponentHandler is delegated to the Container.
     *
     * @param  role  The role we intend to access a Component for.
     * @param  hint  The hint that we use as a qualifier
     *         (note: if null, the default implementation is returned).
     *
     * @return Object  a reference to the ComponentHandler or ComponentSelector for the
     *                 role/hint combo.
     */
    protected Object get( final String role, final Object hint )
        throws ComponentException
    {
        BucketMap hintMap = (BucketMap)m_mapper.get( role );
        Object value;

        if( null == hintMap )
        {
            throw new ComponentException( "Component does not exist" );
        }

        if( null == hint )
        {
            value = hintMap.get( "selector" );

            if( null == value )
            {
                value = hintMap.get( "default" );
            }

            return value;
        }

        value = hintMap.get( hint );

        if( null == value )
        {
            throw new ComponentException( "Component does not exist" );
        }

        return value;
    }

    /**
     * This is the method that the ContainerComponentManager and Selector use to gain
     * access to the ComponentHandlers and ComponentSelectors.  The actual access of
     * the ComponentHandler is delegated to the Container.
     *
     * @param  role  The role we intend to access a Component for.
     * @param  hint  The hint that we use as a qualifier
     *         (note: if null, the default implementation is returned).
     *
     * @return true  if a reference to the role exists.
     */
    protected boolean has( final String role, final Object hint )
    {
        BucketMap hintMap = (BucketMap)m_mapper.get( role );

        if( null == hintMap )
        {
            return false;
        }

        if( null == hint )
        {
            if( !hintMap.containsKey( "selector" ) )
            {
                return hintMap.containsKey( "default" );
            }

            return false;
        }

        return hintMap.containsKey( hint );
    }

    /**
     * Root ComponentManager.  The Container may choose to have it's ComponentManager
     * delegate to the root manager, or it may choose to be entirely self contained.
     */
    public void compose( ComponentManager manager )
        throws ComponentException
    {
        m_manager = manager;
    }

    /**
     * Initializes all components so that the system is ready to be used.
     */
    public void initialize()
        throws Exception
    {
        Iterator i = m_components.iterator();
        FixedSizeBuffer buffer = new FixedSizeBuffer( m_components.size() );

        while( i.hasNext() )
        {
            try
            {
                if( null != m_commandQueue )
                {
                    m_commandQueue.enqueue(
                        new InitComponentHandlerCommand( (ComponentHandler)i.next(),
                                                         getLogger() )
                    );
                }
                else
                {
			ComponentHandler handler = (ComponentHandler) i.next();
			handler.initialize();
                }
            }
            catch( Exception e )
            {
                getLogger().warn( "Could not initialize component", e );
                buffer.add( e );
            }
        }

        if( buffer.size() > 0 )
        {
            StringBuffer message = new StringBuffer();

            while( !buffer.isEmpty() )
            {
                message.append( ( (Exception)buffer.remove() ).getMessage() );
            }

            throw new Exception( message.toString() );
        }
    }

    /**
     * Disposes of all components and frees resources that they consume.
     */
    public void dispose()
    {
        Iterator i = m_components.iterator();

        while( i.hasNext() )
        {
            if( null != m_commandQueue )
            {
                try
                {
                    m_commandQueue.enqueue(
                        new DisposeComponentHandlerCommand( (ComponentHandler)i.next(),
                                                            getLogger() )
                    );

                    i.remove();
                }
                catch( Exception e )
                {
                    if( getLogger().isWarnEnabled() )
                    {
                        getLogger().warn( "Could not dispose component", e );
                    }
                }
            }
            else
            {
                ( (ComponentHandler)i.next() ).dispose();
            }
        }
    }

    /**
     * Exposes to subclasses the component manager which this container
     * uses to manage its child components.
     *
     * @return the child component manager
     */
    protected final ComponentManager getComponentManager()
    {
        return new ContainerComponentManager( this, m_manager );
    }

    /**
     * This is the Default ComponentManager for the Container.  It provides
     * a very simple abstraction, and makes it easy for the Container to manage
     * the references.
     */
    protected static final class ContainerComponentManager
        implements ComponentManager
    {
        private final AbstractContainer m_components;
        private final BucketMap m_used;
        private final ComponentManager m_parent;

        /**
         * This constructor is for a ContainerComponentManager with no parent
         * ComponentManager
         */
        public ContainerComponentManager( final AbstractContainer container )
        {
            this( container, null );
        }

        /**
         * This constructor is for a ContainerComponentManager with a parent
         * ComponentManager
         */
        public ContainerComponentManager( final AbstractContainer container, final ComponentManager parent )
        {
            m_parent = parent;
            m_components = container;
            m_used = new BucketMap();
        }

        public Component lookup( String role )
            throws ComponentException
        {
            Object temp = null;

            try
            {
                temp = m_components.get( role, null );
            }
            catch( ComponentException ce )
            {
                //                ce.printStackTrace( System.err );
                if( null != m_parent )
                {
                    return m_parent.lookup( role );
                }
                else
                {
                    throw ce;
                }
            }

            if( temp instanceof ComponentSelector )
            {
                return (Component)temp;
            }

            if( !( temp instanceof ComponentHandler ) )
            {
                throw new ComponentException( "Invalid entry in component manager" );
            }

            ComponentHandler handler = (ComponentHandler)temp;

            final Component component;

            try
            {
                if( !handler.isInitialized() )
                {
                    handler.initialize();
                }

                component = handler.get();
            }
            catch( Exception e )
            {
                throw new ComponentException( "Could not return a reference to the Component:" + e.getMessage(), e );
            }

            m_used.put( component, handler );

            return component;
        }

        public boolean hasComponent( String role )
        {
            final boolean hasComponent = m_components.has( role, null );

            if( !hasComponent && null != m_parent )
            {
                return m_parent.hasComponent( role );
            }

            return hasComponent;
        }

        public void release( Component component )
        {
            final ComponentHandler handler;

            handler = (ComponentHandler)m_used.remove( component );

            if( null == handler && null != m_parent )
            {
                m_parent.release( component );
                return;
            }

            handler.put( component );
        }
    }

    /**
     * This is the Default ComponentSelector for the Container.  It provides
     * a very simple abstraction, and makes it easy for the Container to manage
     * the references.
     */
    protected static final class ContainerComponentSelector
        implements ComponentSelector
    {
        private final String m_role;
        private final AbstractContainer m_components;
        private final BucketMap m_used;

        public ContainerComponentSelector( final AbstractContainer container, final String role )
        {
            m_role = role;
            m_components = container;
            m_used = new BucketMap();
        }

        public Component select( final Object hint )
            throws ComponentException
        {
            if( null == hint )
            {
                throw new IllegalArgumentException( "hint cannot be null" );
            }

            ComponentHandler handler = (ComponentHandler)m_components.get( m_role, hint );

            if( null == handler )
            {
                throw new ComponentException( "The hint does not exist in the ComponentSelector" );
            }

            final Component component;

            try
            {
                if( !handler.isInitialized() )
                {
                    handler.initialize();
                }

                component = handler.get();
            }
            catch( Exception e )
            {
                throw new ComponentException( "Could not return a reference to the Component", e );
            }

            m_used.put( component, handler );

            return component;
        }

        public boolean hasComponent( Object hint )
        {
            return m_components.has( m_role, hint );
        }

        public void release( Component component )
        {
            final ComponentHandler handler;

            handler = (ComponentHandler)m_used.remove( component );

            handler.put( component );
        }
    }

    /**
     * This is the command class to initialize a ComponentHandler
     */
    protected static final class InitComponentHandlerCommand implements Command
    {
        private final ComponentHandler m_handler;
        private final Logger m_logger;

        protected InitComponentHandlerCommand( ComponentHandler handler, Logger logger )
        {
            m_handler = handler;
            m_logger = logger;
        }

        public void execute()
            throws Exception
        {
            try
            {
                if( !m_handler.isInitialized() )
                {
                    m_handler.initialize();

                    if( m_logger.isDebugEnabled() )
                    {
                        m_logger.debug( "Initialized Handler " + m_handler );
                    }
                }
            }
            catch( Exception e )
            {
                if( m_logger.isErrorEnabled() )
                {
                    m_logger.error( "Could not initialize ComponentHandler", e );
                }

                throw e;
            }
        }
    }

    /**
     * This is the command class to dispose a ComponentHandler
     */
    protected static final class DisposeComponentHandlerCommand implements Command
    {
        private final ComponentHandler m_handler;
        private final Logger m_logger;

        protected DisposeComponentHandlerCommand( ComponentHandler handler, Logger logger )
        {
            m_handler = handler;
            m_logger = logger;
        }

        public void execute()
        {
            m_handler.dispose();

            if( m_logger.isDebugEnabled() )
            {
                m_logger.debug( "Disposed of Handler " + m_handler );
            }
        }
    }
}

