/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.logger.Logger;

/**
 * A ContainerManager is a pocket universe for a container and its
 * components.
 *
 * <p><b>Case 1: Use by a servlet or other "root" entity</b></p>
 *
 * <pre>
 * <code>
 *    //
 *    ContextBuilder contextBuilder = new ContextBuilder ();
 *    contextBuilder.setContainerClass( Thread.currentThread().getContextClassLoader().loadClass( "org.apache.excalibur.fortress.test.TestContainer" ) );
 *    contextBuilder.setContextDirectory( "./" );
 *    contextBuilder.setWorkDirectory( "./" );
 *    contextBuilder.setContainerConfiguration( "resource://org.apache.excalibur.fortress/test/ContainerProfile.xconf" );
 *    contextBuilder.setLoggerManagerConfiguration( "resource://org.apache.excalibur.fortress/test/ContainerProfile.xlog" );
 *
 *    ContextManager contextManager = new ContextManager( null, contextBuilder.getContext(), null );
 *    contextManager.initialize();
 *
 *    ContainerManager cm = new DefaultContainerManager( contextManager.getContextInstance() );
 *    cm.initialize();
 * </code>
 * </pre>
 *
 * Then, for example, wait for a request and pass it on to the container:
 *
 * <pre>
 * <code>
 *    TestContainer container = (TestContainer) cm.getContainer();
 *    container.handleRequest( ... );
 * </code>
 * </pre>
 *
 * When done, dispose of the managers.
 *
 * <pre>
 * <code>
 *    containerManager.dispose();
 *    contextManager.dispose();
 * </code>
 * </pre>
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.37 $ $Date: 2002/04/13 01:39:56 $
 */
public interface ContainerManager extends Disposable, Initializable
{
    /**
     * Get a reference to your Container.  Typically, you would cast this to
     * whatever interface you will use to interact with it.
     */
    Object getContainer();

    /**
     * Get a reference the default Logger.
     */
    Logger getLogger();
}
