/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress;

import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.ConsoleLogger;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.parameters.Parameterizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.excalibur.fortress.util.ContextManager;
import org.apache.excalibur.util.ComponentStateValidator;

/**
 * The ContainerManager is a single point of contact to manage your Container
 * resources.  It takes care of creating the other managers that a Container
 * needs to use, as well as initializing the Container.  It is designed to be
 * directly instantiated by whatever class needs to initialize your system.
 *
 * <p>
 *   The ContainerManager provides some constants used in the initial
 *   <code>Parameters</code> passed into the ContainerManager.  The ContainerManager
 *   uses these values to create all the pieces necessary for the Container.
 *   Below is a table that describes what those options are.
 * </p>
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.9 $ $Date: 2002/05/13 12:17:39 $
 */
public class DefaultContainerManager implements ContainerManager, ContainerManagerConstants
{
    private final Context initParameters;
    private final Context containerContext;
    private Logger logger;
    private Object containerInstance;
    private ComponentStateValidator validator;

    public DefaultContainerManager( ContextManager contextManager )
    {
        this( contextManager, null );
    }

    public DefaultContainerManager( ContextManager contextManager, Logger logger )
    {
        this( contextManager.getContainerManagerContext(), contextManager.getChildContext(), logger );
    }

    public DefaultContainerManager( Context initParameters )
    {
        this( initParameters, null, null );
    }

    public DefaultContainerManager( Context initParameters, Logger logger )
    {
        this( initParameters, null, logger );
    }

    public DefaultContainerManager( Context initParameters, Context containerContext )
    {
        this( initParameters, containerContext, null );
    }

    public DefaultContainerManager( Context initParameters, Context containerContext, Logger logger )
    {
        this.initParameters = initParameters;
        this.containerContext = containerContext != null ? containerContext : initParameters;
        this.logger = logger == null ? createLoggerFromContext( initParameters ) : logger;
    }

    protected Logger createLoggerFromContext( Context initParameters )
    {
        try
        {
            return ( (LoggerManager)initParameters.get( LOGGER_MANAGER ) ).getDefaultLogger();
        }
        catch( ContextException ce )
        {
            Logger consoleLogger = new ConsoleLogger();
            consoleLogger.error( "Could not obtain logger manager from init parameters (this should not happen). Using console instead." );
            return consoleLogger;
        }
    }

    /**
     * Initialize the ContainerManager
     */
    public void initialize()
    {
        initializeContainer();
    }

    protected void initializeContainer()
    {
        if( null == containerInstance )
        {
            Object instance = null;
            try
            {
                instance = ( (Class)initParameters.get( CONTAINER_CLASS ) ).newInstance();
            }
            catch( Exception e )
            {
                instance = null;
                if( getLogger().isFatalErrorEnabled() )
                {
                    getLogger().fatalError( "Cannot set up the Container, this is an error I cannot recover from.", e );
                }
                return;
            }

            validator = new ComponentStateValidator( instance );

            try
            {
                if( instance instanceof LogEnabled )
                {
                    validator.checkLogEnabled();
                    ( (LogEnabled)instance ).enableLogging( logger );
                }

                if( instance instanceof Contextualizable )
                {
                    validator.checkContextualized();
                    ( (Contextualizable)instance ).contextualize( containerContext );
                }

                if( instance instanceof Composable )
                {
                    validator.checkComposed();
                    ( (Composable)instance ).compose( (ComponentManager)initParameters.get( COMPONENT_MANAGER ) );
                }

                if( instance instanceof Configurable )
                {
                    validator.checkConfigured();
                    ( (Configurable)instance ).configure( (Configuration)initParameters.get( CONFIGURATION ) );
                }

                if( instance instanceof Parameterizable )
                {
                    validator.checkParameterized();
                    ( (Parameterizable)instance ).parameterize( (Parameters)initParameters.get( PARAMETERS ) );
                }

                if( instance instanceof Initializable )
                {
                    validator.checkInitialized();
                    ( (Initializable)instance ).initialize();
                }

                if( instance instanceof Startable )
                {
                    validator.checkStarted();
                    ( (Startable)instance ).start();
                }
            }
            catch( Exception e )
            {
                instance = null;
                if( getLogger().isFatalErrorEnabled() )
                {
                    getLogger().fatalError( "Cannot set up the Container, this is an error I cannot recover from.", e );
                }
            }

            containerInstance = instance;
        }
    }

    protected void disposeContainer()
    {
        if( null != containerInstance )
        {
            if( containerInstance instanceof Startable )
            {
                try
                {
                    validator.checkStopped();
                    ( (Startable)containerInstance ).stop();
                }
                catch( Exception e )
                {
                    if( getLogger().isWarnEnabled() )
                    {
                        getLogger().warn( "Caught an exception when stopping the Container, continuing with shutdown", e );
                    }
                }
            }

            if( containerInstance instanceof Disposable )
            {
                validator.checkDisposed();
                ( (Disposable)containerInstance ).dispose();
            }

            validator = null;
            containerInstance = null;
        }
    }

    /**
     * Override this if you have any special needs for the container (such as
     * wanting to use your own class).
     */
    private final void recycleContainer()
        throws InitializationException
    {
        disposeContainer();
        initializeContainer();
    }

    /**
     * Dispose of the ContainerManager and managed Container
     */
    public void dispose()
    {
        disposeContainer();
    }

    /**
     * Get a reference to your Container.  Typically, you would cast this to
     * whatever interface you will use to interact with it.
     */
    public Object getContainer()
    {
        return containerInstance;
    }

    public Logger getLogger()
    {
        return logger;
    }

}
