/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.handler;

import java.lang.reflect.Method;
import org.apache.avalon.excalibur.instrument.Instrument;
import org.apache.avalon.excalibur.instrument.Instrumentable;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.avalon.framework.logger.Logger;
import org.apache.excalibur.fortress.Container;
import org.apache.excalibur.fortress.ContainerConstants;
import org.apache.excalibur.fortress.util.RoleManager;
import org.apache.excalibur.fortress.markers.ComponentMarker;
import org.apache.excalibur.fortress.markers.MarkerManager;

/**
 * AbstractComponentHandler class, ensures components are initialized
 * and destroyed correctly.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:crafterm@apache.org">Marcus Crafter</a>
 * @version CVS $Revision: 1.3 $ $Date: 2002/06/06 04:47:10 $
 * @since 4.0
 */
public abstract class AbstractComponentHandler
    implements ComponentHandler, Configurable, Disposable, Instrumentable
{
    /** The instance of the ComponentFactory that creates and disposes of the Component */
    protected final ComponentFactory m_factory;

    protected String m_name;

    /** State management boolean stating whether the Handler is initialized or not */
    protected boolean m_initialized = false;

    /** State management boolean stating whether the Handler is disposed or not */
    protected boolean m_disposed = false;

    /** Logger for factory */
    protected Logger m_logger;

    /** Logger Manager */
    protected LoggerManager m_logkit;

    /** Marker Manager */
    protected MarkerManager m_markerManager;

    /** Context */
    protected Context m_context;

    /**
     * Create a ComponentHandler that takes care of hiding the details of
     * whether a Component is ThreadSafe, Poolable, or SingleThreaded.
     * It falls back to SingleThreaded if not specified.
     */
    public AbstractComponentHandler( final Class componentClass,
                                     final Configuration config,
                                     final ComponentManager manager,
                                     final Context context )
        throws Exception
    {
        m_logkit = (LoggerManager)context.get( Container.LOGGER_MANAGER );
        m_markerManager = new MarkerManager();
	m_markerManager.enableLogging( m_logkit.getDefaultLogger() );
	m_context = prepareContext( context, config, manager );
        m_factory =
            new ComponentFactory(
                componentClass,
                config,
                manager,
                m_context,
                m_logkit,
                m_markerManager
            );
    }

    public void configure( Configuration config )
        throws ConfigurationException
    {
        if (config != null)
        {
            m_markerManager.configure( config );
        }
        else
        {
            throw new ConfigurationException( "Require MarkerManager configuration" );
        }
    }

    public boolean isInitialized()
    {
        return m_initialized;
    }

    /**
     * Initialize the ComponentHandler.
     */
    public abstract void initialize()
        throws Exception;

    /**
     * Get a reference of the desired Component
     */
    public Component get()
        throws Exception
    {
        if( !m_initialized )
        {
            throw new IllegalStateException(
                "You cannot get a component from an uninitialized holder"
            );
        }

        if( m_disposed )
        {
            throw new IllegalStateException(
                "You cannot get a component from a disposed holder"
            );
        }

        return null; // value not used, satisfies compiler
    }

    /**
     * Return a reference of the desired Component
     */
    public void put( final Component component )
    {
        if( !m_initialized )
        {
            throw new IllegalStateException(
                "You cannot put a component in an uninitialized holder"
            );
        }
    }

    /**
     * Dispose of the ComponentHandler and any associated Pools and Factories.
     */
    public void dispose()
    {
        try
        {
            if( m_factory instanceof Disposable )
            {
                ( (Disposable)m_factory ).dispose();
            }
        }
        catch( final Exception e )
        {
            if( m_logger.isWarnEnabled() )
            {
                m_logger.warn( "Error decommissioning component: " +
                               m_factory.getCreatedClass().getName(), e );
            }
        }

        m_disposed = true;
    }

    private Context prepareContext( Context parent, Configuration config, ComponentManager manager )
    {
        DefaultContext context = new DefaultContext( parent );
        context.put( ContainerConstants.COMPONENT_CONFIGURATION, config );
        context.put( ContainerConstants.COMPONENT_MANAGER, manager );
        return context;
    }

    public final void setInstrumentableName( String name )
    {
        m_name = name;
    }

    public final String getInstrumentableName()
    {
        return m_name;
    }

    public final Instrument[] getInstruments()
    {
        return Instrumentable.EMPTY_INSTRUMENT_ARRAY;
    }

    public final Instrumentable[] getChildInstrumentables()
    {
        return new Instrumentable[]{
            m_factory
        };
    }
}
