/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.handler;

import org.apache.avalon.excalibur.instrument.CounterInstrument;
import org.apache.avalon.excalibur.instrument.Instrument;
import org.apache.avalon.excalibur.instrument.Instrumentable;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.container.ContainerUtil;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.parameters.Parameterizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.excalibur.fortress.markers.MarkerManager;
import org.apache.excalibur.mpool.ObjectFactory;

/**
 * Factory for Avalon components.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:paul@luminas.co.uk">Paul Russell</a>
 * @version CVS $Revision: 1.12 $ $Date: 2002/06/06 04:47:10 $
 * @since 4.0
 */
public class ComponentFactory
    implements ObjectFactory, ThreadSafe, Instrumentable
{
    private CounterInstrument m_newInstance;
    private CounterInstrument m_dispose;

    /** The class which this <code>ComponentFactory</code>
     * should create.
     */
    private Class m_componentClass;

    /** The Context for the component
     */
    private Context m_context;

    /** The component manager for this component.
     */
    private ComponentManager m_componentManager;

    /** The configuration for this component.
     */
    private Configuration m_configuration;

    /** The LogKitManager for child ComponentSelectors
     */
    private LoggerManager m_logManager;

    /** The logger for the ComponentFactory
     */
    private Logger m_logger;

    /** The MarkerManager */
    private MarkerManager m_markerManager;

    /**
     * Construct a new component factory for the specified component.
     *
     * @param componentClass the class to instantiate (must have a default constructor).
     * @param configuration the <code>Configuration</code> object to pass to new instances.
     * @param componentManager the component manager to pass to <code>Composable</code>s.
     * @param context the <code>Context</code> to pass to <code>Contexutalizable</code>s.
     */
    public ComponentFactory( final Class componentClass,
                             final Configuration configuration,
                             final ComponentManager componentManager,
                             final Context context,
                             final LoggerManager logkit,
                             final MarkerManager markerManager )
    {
        m_componentClass = componentClass;
        m_configuration = configuration;
        m_componentManager = componentManager;
        m_context = context;
        m_logManager = logkit;
        m_logger = m_logManager.getLoggerForCategory( "system.factory" );
        m_markerManager = markerManager;

        m_newInstance = new CounterInstrument( "Create " + getInstrumentableName() );
        m_dispose = new CounterInstrument( "Destroy " + getInstrumentableName() );
    }

    public Object newInstance()
        throws Exception
    {
        final Object component = m_componentClass.newInstance();

        if( m_logger.isDebugEnabled() )
        {
            m_logger.debug( "ComponentFactory creating new instance of " +
                            m_componentClass.getName() + "." );
        }

        m_markerManager.executeCreationMarkers( (Component) component, m_context );

        if( m_newInstance.isActive() )
        {
            m_newInstance.increment();
        }

        return component;
    }

    public final Class getCreatedClass()
    {
        return m_componentClass;
    }

    public final void dispose( final Object component )
        throws Exception
    {
        if( m_logger.isDebugEnabled() )
        {
            m_logger.debug( "ComponentFactory decommissioning instance of " +
                            getCreatedClass().getName() + "." );
        }

        if( getCreatedClass().equals( component.getClass() ) )
        {
            m_markerManager.executeDestructionMarkers( (Component) component, m_context );

            if( m_dispose.isActive() )
            {
                m_dispose.increment();
            }
        }
        else
        {
            throw new IllegalArgumentException( "The object given to be disposed does not come from this ObjectFactory" );
        }
    }

    public final void setInstrumentableName( String name )
    {
        // ignore
    }

    public final String getInstrumentableName()
    {
        return getCreatedClass().getName();
    }

    public final Instrument[] getInstruments()
    {
        return new Instrument[]{
            m_newInstance,
            m_dispose
        };
    }

    public final Instrumentable[] getChildInstrumentables()
    {
        return Instrumentable.EMPTY_INSTRUMENTABLE_ARRAY;
    }

}
