/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.handler;

import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;

/**
 * The FactoryComponentHandler to make sure components are initialized
 * and destroyed correctly.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:crafterm@apache.org">Marcus Crafter</a>
 * @version CVS $Revision: 1.16 $ $Date: 2002/06/06 04:28:50 $
 * @since 4.0
 */
public final class FactoryComponentHandler
    extends AbstractComponentHandler
{
    /**
     * Create a ComponentHandler that takes care of hiding the details of
     * whether a Component is ThreadSafe, Poolable, or SingleThreaded.
     * It falls back to SingleThreaded if not specified.
     */
    public FactoryComponentHandler( final Class componentClass,
                                    final Configuration config,
                                    final ComponentManager manager,
                                    final Context context )
        throws Exception
    {
        super( componentClass, config, manager, context );
        m_logger = m_logkit.getLoggerForCategory( "system.handler.factory" );
        m_name = "FactoryComponentHandler";
    }

    /**
     * Initialize the ComponentHandler.
     */
    public void initialize()
    {
        if( m_initialized )
        {
            return;
        }

        if( m_logger.isDebugEnabled() )
        {
            m_logger.debug( "ComponentHandler initialized for: " + this.m_factory.getCreatedClass().getName() );
        }
        m_initialized = true;
    }

    /**
     * Get a reference of the desired Component
     */
    public Component get()
        throws Exception
    {
        super.get();

	Component component = (Component) m_factory.newInstance();
        m_markerManager.executeAccessMarkers( component, m_context );

	return component;
    }

    /**
     * Return a reference of the desired Component
     */
    public void put( final Component component )
    {
        super.put( component );

        try
        {
            m_markerManager.executeReleaseMarkers( component, m_context );
            m_factory.dispose( component );
        }
        catch( final Exception e )
        {
            if( m_logger.isWarnEnabled() )
            {
                m_logger.warn( "Error decommissioning component: " +
                               m_factory.getCreatedClass().getName(), e );
            }
        }
    }
}
