/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.handler;

import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;

/**
 * The ThreadSafeComponentHandler to make sure components are initialized
 * and destroyed correctly.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:crafterm@apache.org">Marcus Crafter</a>
 * @version CVS $Revision: 1.17 $ $Date: 2002/06/06 04:27:57 $
 * @since 4.0
 */
public final class PerThreadComponentHandler
    extends AbstractComponentHandler
{
    private ThreadLocalComponent m_instance;

    /**
     * Create a ComponentHandler that takes care of hiding the details of
     * whether a Component is ThreadSafe, Poolable, or SingleThreaded.
     * It falls back to SingleThreaded if not specified.
     */
    public PerThreadComponentHandler( final Class componentClass,
                                      final Configuration config,
                                      final ComponentManager manager,
                                      final Context context )
        throws Exception
    {
        super( componentClass, config, manager, context );
        m_instance = new ThreadLocalComponent( m_factory );
        m_logger = m_logkit.getLoggerForCategory( "system.handler.perthread" );
        m_name = "PerThreadComponentHandler";
    }

    /**
     * Initialize the ComponentHandler.
     */
    public void initialize()
        throws Exception
    {
        if( m_initialized )
        {
            return;
        }

        if( m_logger.isDebugEnabled() )
        {
            if( this.m_factory != null )
            {
                m_logger.debug( "ComponentHandler initialized for: " + this.m_factory.getCreatedClass().getName() );
            }
            else
            {
                m_logger.debug( "ComponentHandler initialized for: " + this.m_instance.getClass().getName() );
            }
        }

        m_initialized = true;
    }

    /**
     * Get a reference of the desired Component
     */
    public final Component get()
        throws Exception
    {
        super.get();

	Component component = (Component) m_instance.getComponent();
        m_markerManager.executeAccessMarkers( component, m_context );
        return component;
    }

    /**
     * Return a reference of the desired Component
     */
    public void put( final Component component )
    {
        try
        {
            m_markerManager.executeReleaseMarkers( component, m_context );
            super.put( component );
        }
        catch (final Exception e)
        {
            if( m_logger.isWarnEnabled() )
            {
                m_logger.warn( "Error releasing component: " +
                                m_factory.getCreatedClass().getName(), e );
            }
        }
    }

    /**
     * Dispose of the ComponentHandler and any associated Pools and Factories.
     */
    public void dispose()
    {
        super.dispose();

        m_instance = null;
    }

    private static final class ThreadLocalComponent extends ThreadLocal
    {
        private final ComponentFactory m_factory;

        protected ThreadLocalComponent( ComponentFactory factory )
        {
            m_factory = factory;
        }

        protected Object initialValue()
        {
            try
            {
                return m_factory.newInstance();
            }
            catch( Exception e )
            {
                return null;
            }
        }

        public Component getComponent()
        {
            return (Component)this.get();
        }
    }
}
