/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.handler;

import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;
import org.apache.excalibur.fortress.Container;
import org.apache.excalibur.mpool.Pool;
import org.apache.excalibur.mpool.PoolManager;

/**
 * The PoolableComponentHandler to make sure components are initialized
 * and destroyed correctly.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:crafterm@apache.org">Marcus Crafter</a>
 * @version CVS $Revision: 1.20 $ $Date: 2002/06/06 04:27:22 $
 * @since 4.0
 */
public final class PoolableComponentHandler
    extends AbstractComponentHandler
{
    /** The instance of the PoolManager to create the Pool for the Handler */
    private final PoolManager m_poolManager;

    /** The pool of components for <code>Poolable</code> Components */
    private Pool m_pool;

    /** The Config element for the poolable */
    private final int m_poolMin;

    /**
     * Create a ComponentHandler that takes care of hiding the details of
     * whether a Component is ThreadSafe, Poolable, or SingleThreaded.
     * It falls back to SingleThreaded if not specified.
     */
    public PoolableComponentHandler( final Class componentClass,
                                     final Configuration config,
                                     final ComponentManager manager,
                                     final Context context )
        throws Exception
    {
        super( componentClass, config, manager, context );
        m_poolMin = config.getAttributeAsInteger( "pool-min", 10 );
        m_logger = m_logkit.getLoggerForCategory( "system.handler.poolable" );
        m_poolManager = (PoolManager)context.get( Container.POOL_MANAGER );
        m_name = "PoolableComponentHandler";
    }

    /**
     * Initialize the ComponentHandler.
     */
    public void initialize()
        throws Exception
    {
        if( m_initialized )
        {
            return;
        }

        m_pool = m_poolManager.getManagedPool( m_factory, m_poolMin );

        if( m_logger.isDebugEnabled() )
        {
            m_logger.debug( "ComponentHandler initialized for: " +
                            m_factory.getCreatedClass().getName() );
        }
        m_initialized = true;
    }

    /**
     * Get a reference of the desired Component
     */
    public Component get()
        throws Exception
    {
        super.get();

	Component component = (Component) m_pool.acquire();
	m_markerManager.executeAccessMarkers( component, m_context );
        return component;
    }

    /**
     * Return a reference of the desired Component
     */
    public void put( final Component component )
    {
        try
        {
            super.put( component );
            m_markerManager.executeReleaseMarkers( component, m_context );
            m_pool.release( component );
        }
        catch (final Exception e)
        {
            if( m_logger.isWarnEnabled() )
            {
                m_logger.warn( "Error releasing component: " +
                               m_factory.getCreatedClass().getName(), e );
            }
        }
    }

    /**
     * Dispose of the ComponentHandler and any associated Pools and Factories.
     */
    public void dispose()
    {
        super.dispose();
        // REVISIT: release m_pool ? m_pool = null;
    }
}
