/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.handler;

import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.logger.Logger;

/**
 * The ThreadSafeComponentHandler to make sure components are initialized
 * and destroyed correctly.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:crafterm@apache.org">Marcus Crafter</a>
 * @version CVS $Revision: 1.14 $ $Date: 2002/06/05 17:06:48 $
 * @since 4.0
 */
public final class ThreadSafeComponentHandler extends AbstractComponentHandler
{
    private Component m_instance;

    /**
     * Create a ComponentHandler that takes care of hiding the details of
     * whether a Component is ThreadSafe, Poolable, or SingleThreaded.
     * It falls back to SingleThreaded if not specified.
     */
    public ThreadSafeComponentHandler( final Class componentClass,
                                       final Configuration config,
                                       final ComponentManager manager,
                                       final Context context )
        throws Exception
    {
        super( componentClass, config, manager, context );
        m_logger = m_logkit.getLoggerForCategory( "system.handler.threadsafe" );
        m_name = "ThreadSafeComponentHandler";
    }

    /**
     * Initialize the ComponentHandler.
     */
    public void initialize()
        throws Exception
    {
        if( m_initialized )
        {
            return;
        }

        if( m_instance == null )
        {
            m_instance = (Component)this.m_factory.newInstance();
        }

        if( m_logger.isDebugEnabled() )
        {
            if( this.m_factory != null )
            {
                m_logger.debug( "ComponentHandler initialized for: " + this.m_factory.getCreatedClass().getName() );
            }
            else
            {
                m_logger.debug( "ComponentHandler initialized for: " + this.m_instance.getClass().getName() );
            }
        }

        m_initialized = true;
    }

    /**
     * Get a reference of the desired Component
     */
    public final Component get()
        throws Exception
    {
        super.get();
        m_markerManager.executeAccessMarkers( m_instance, m_context );
        return m_instance;
    }

    /**
     * Return a reference of the desired Component
     */
    public void put( final Component component )
    {
        try
        {
            super.put( component );
            m_markerManager.executeReleaseMarkers( m_instance, m_context );
        }
        catch( final Exception e )
        {
            if( m_logger.isWarnEnabled() )
            {
                m_logger.warn( "Error releasing component: " +
                               m_factory.getCreatedClass().getName(), e );
            }
        }
    }

    /**
     * Dispose of the ComponentHandler and any associated Pools and Factories.
     */
    public void dispose()
    {
        try
        {
            if( null != m_factory )
            {
                m_factory.dispose( m_instance );
            }
            else
            {
                if( m_instance instanceof Startable )
                {
                    ( (Startable)m_instance ).stop();
                }

                if( m_instance instanceof Disposable )
                {
                    ( (Disposable)m_instance ).dispose();
                }
            }

            m_instance = null;
        }
        catch( final Exception e )
        {
            if( m_logger.isWarnEnabled() )
            {
                m_logger.warn( "Error decommissioning component: " +
                               m_factory.getCreatedClass().getName(), e );
            }
        }

        m_disposed = true;
    }
}
