/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.markers;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;

/**
 * <code>AbstractMarkerManager</code> class, provides base operations for
 * executing a particular set of component marker interfaces on a 
 * component.
 *
 * @author <a href="mailto:crafterm@apache.org">Marcus Crafter</a>
 * @version 1.0
 */
public abstract class AbstractMarkerManager extends AbstractLogEnabled
	implements Configurable, MarkerManagerConstants
{
	// arrays holding marker configurations
	protected Configuration[] m_accessMarkers;
	protected Configuration[] m_releaseMarkers;
	protected Configuration[] m_creationMarkers;
	protected Configuration[] m_destructionMarkers;

	// internal store which caches marker methods, interfaces, and actions
	private final Map markerMethodMap = new HashMap();
	private final Map markerInterfaceCache = new HashMap();
	private final Map markerActionCache = new HashMap();

	/**
	 * Constructor, creates a marker manager, and caches all marker
	 * <code>Method</code> objects for later use.
	 *
	 * @exception Exception if an error occurs
	 */
	public AbstractMarkerManager()
		throws Exception
	{
		setupMarkerMethodMap();
	}

	/**
	 * <code>configure</code> method, configures this marker manager class
	 * (see markers.xml for example input file).
	 *
	 * @param config a <code>Configuration</code> instance
	 * @exception ConfigurationException if an error occurs
	 */
	public void configure(Configuration config)
		throws ConfigurationException
	{
		// obtain marker definitions from config object
		m_accessMarkers = config.getChild(ACCESS).getChildren(MARKER);
		m_releaseMarkers = config.getChild(RELEASE).getChildren(MARKER);
		m_creationMarkers = config.getChild(CREATION).getChildren(MARKER);
		m_destructionMarkers = config.getChild(DESTRUCTION).getChildren(MARKER);

		// debug some statistics
		if (getLogger().isDebugEnabled())
		{
			getLogger().debug(
				"MarkerManager configured with " + m_accessMarkers.length + 
				" access markers, " + m_releaseMarkers.length + 
				" release markers, " + m_creationMarkers.length +
				" creation markers and " + m_destructionMarkers.length +
				" destruction markers"
			);
		}
	}

	/**
	 * <code>executeMarkers</code> method, executes a given array of marker interfaces
	 * on a given component. Called by subclasses specifying which phase the given
	 * marker interfaces adhere to (see markers.xml for marker and phase definitions).
	 *
	 * @param markers a <code>Configuration[]</code> array containing marker definitions
	 * @param component a <code>Component</code> instance
	 * @param context a <code>Context</code> instance
	 * @param type a <code>String</code> value, indicating the phase the specified
	 * marker array adheres to.
	 * @exception Exception if an error occurs
	 */
	protected void executeMarkers(
		Configuration[] markers, Component component, Context context, String type
	)
		throws Exception
	{
		Method method = (Method) markerMethodMap.get(type);

		for (int i = 0; i < markers.length; ++i)
		{
			executeMarker(markers[i], component, context, method);
		}
	}

	/**
	 * <code>executeMarker</code> method, tests a component to see if it
	 * implements a single given marker interface, and if so, executes the 
	 * applicable marker's 'action' method (as defined in markers.xml).
	 *
	 * @param marker marker definition
	 * @param component a <code>Component</code> instance
	 * @param context a <code>Context</code> instance
	 * @param method a <code>Method</code> object, specifying which method should be
	 * invoked on the marker's <i>action</i> class, should this component implement
	 * the given marker's interface (see markers.xml for definitions of each marker
	 * and their action classes).
	 * @exception Exception if an error occurs
	 */
	private void executeMarker(
		Configuration marker, Component component, Context context, Method method
	)
		throws Exception
	{
		Class interfaze = getMarkerInterfaceClass(marker);

		if (interfaze.isAssignableFrom(component.getClass()))
		{
			if (getLogger().isDebugEnabled())
			{
				getLogger().debug(
					"Executing marker " + interfaze.getName() +
					" on component " + component +
					" (" + method.getName() + " phase)"
				);
			}

			method.invoke(getMarkerAction(marker), new Object[] {component, context});
		}
	}

	/**
	 * Helper method for obtaining a <code>Class</code> object for a 
	 * given ComponentMarkers <i>interface</i> configuration. The ComponentMarker
	 * action name is specified in the 'interface' attribute of the given
	 * configuration
	 *
	 * @param marker marker definition
	 * @return a <code>Class</code> object, representing the 'interface'
	 * the given marker represents
	 * @exception Exception if an error occurs
	 */
	private Class getMarkerInterfaceClass(Configuration marker)
		throws Exception
	{
		String interfaze = marker.getAttribute("interface");

		if (markerInterfaceCache.containsKey(interfaze))
			return (Class) markerInterfaceCache.get(interfaze);
		else
		{
			Class clazz = Class.forName(interfaze);
			markerInterfaceCache.put(interfaze, clazz);
			return clazz;
		}
	}

	/**
	 * Helper method for obtaining a <code>ComponentMarker</code> object
	 * for a given ComponentMarkers <i>action</i> configuration. The ComponentMarker
	 * class name is specified in the 'action' attribute of the given
	 * configuration.
	 *
	 * This method also caches marker actions objects for future use.
	 *
	 * @param marker marker definition
	 * @return a <code>ComponentMarker</code> object, as defined by the
	 * given markers 'action' attribute.
	 * @exception Exception if an error occurs
	 */
	private ComponentMarker getMarkerAction(Configuration marker)
		throws Exception
	{
		String action = marker.getAttribute("action");

		if (markerActionCache.containsKey(action))
			return (ComponentMarker) markerActionCache.get(action);
		else
		{
			ComponentMarker instance =
				(ComponentMarker) Class.forName(action).newInstance();
			markerActionCache.put(action, instance);
			return instance;
		}
	}

	/**
	 * Helper method to cache <code>Method</code> objects for all methods
	 * defined on the <code>ComponentMarker</code> interface. Called by the 
	 * constructor.
	 *
	 * @exception Exception if an error occurs
	 */
	private void setupMarkerMethodMap()
		throws Exception
	{
		final Class[] params = {Component.class, Context.class};
		final String[] methods = { ACCESS, RELEASE, CREATION, DESTRUCTION };

		for (int i = 0; i < methods.length; ++i)
		{
			markerMethodMap.put(
				methods[i], 
				ComponentMarker.class.getDeclaredMethod(methods[i], params)
			);
		}

		// markerMethodMap.makeReadOnly();
	}
}
