/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included  with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.avalon.framework.component.Component;

/**
 * The Excalibur Role Manager is used for Excalibur Role Mappings.  All of the
 * information is hard-coded.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.8 $ $Date: 2002/05/13 12:17:39 $
 * @since 4.1
 */
public abstract class AbstractRoleManager
    implements RoleManager
{
    protected static final String EMPTY_STRING = "";

    protected final ClassLoader m_loader;

    /** Map for shorthand to class mapping */
    protected Map m_shorthands;

    /** Map for role to classname mapping */
    protected Map m_classNames;

    /** Map for role to handler classname mapping */
    protected Map m_handlerNames;

    /** Parent <code>RoleManager</code> for nested resolution */
    protected final RoleManager m_parent;

    /**
     * Default constructor--this RoleManager has no parent.
     */
    public AbstractRoleManager()
    {
        this( null );
    }

    /**
     * Alternate constructor--this RoleManager has the specified
     * parent.
     *
     * @param parent  The parent <code>RoleManager</code>.
     */
    public AbstractRoleManager( RoleManager parent )
    {
        this( parent, Thread.currentThread().getContextClassLoader() );
    }

    /**
     * Alternate constructor--this RoleManager has the specified
     * parent.
     *
     * @param parent  The parent <code>RoleManager</code>.
     */
    public AbstractRoleManager( RoleManager parent, ClassLoader loader )
    {
        ClassLoader thisLoader = loader;

        if( null == thisLoader )
        {
            thisLoader = Thread.currentThread().getContextClassLoader();
        }

        m_loader = thisLoader;
        m_parent = parent;
    }

    protected void setup( Map shorts, Map classes, Map handlers,
                          String shortName, String role, String className,
                          String handlerClassName )
    {
        final Class klass;
        Class handlerKlass;

        try
        {
            klass = m_loader.loadClass( className );

            if( !Component.class.isAssignableFrom( klass ) )
            {
                // Do not store reference if it is not a Component
                return;
            }
        }
        catch( Exception e )
        {
            // Do not store reference if class does not exist.
            return;
        }

        try
        {
            handlerKlass = m_loader.loadClass( handlerClassName );
        }
        catch( Exception e )
        {
            handlerKlass = org.apache.excalibur.fortress.handler.PerThreadComponentHandler.class;
        }

        shorts.put( shortName, klass );
        shorts.put( klass, shortName );
        classes.put( klass,
                     role );

        List classList = (List)classes.get( role );

        if( null == classList )
        {
            classList = new ArrayList( 5 );
        }

        classList.add( klass );
        classes.put( role, classList );

        handlers.put( klass, handlerKlass );
    }

    /**
     * Find the Class for the given shorthand name.  If there is no
     * correspondence between the class and the shorthand name, the method
     * returns <code>null</code>.  If this RoleManager does not have the match,
     * and there is a parent RoleManager, the parent will be asked to resolve
     * the request.
     */
    public final Class getClassForName( final String shorthandName )
    {
        if( shorthandName == null ) return null;

        final Class component = (Class)m_shorthands.get( shorthandName );

        if( null == component && null != m_parent )
        {
            return m_parent.getClassForName( shorthandName );
        }

        return component;
    }

    /**
     * Retrieves the real role name from a shorthand name.  Usually
     * the shorthand name refers to a configuration element name.  If
     * this RoleManager does not have the match, and there is a parent
     * RoleManager, the parent will be asked to resolve the role.
     *
     * @param shorthandName  The shortname that is an alias for the role.
     * @return the official role name.
     */
    public final String getNameForClass( final Class klass )
    {
        final String shorthandName = (String)m_shorthands.get( klass );

        if( null == shorthandName && null != m_parent )
        {
            return m_parent.getNameForClass( klass );
        }

        return shorthandName;
    }

    /**
     * Retrieves the handler class name for the specified class name.  This
     * is called for every ComponentImplementation.  If this RoleManager does
     * not have the match, and there is a parent RoleManager, the parent will be
     * asked to resolve the handler's class name.
     *
     * @param role  The role that has a default implementation.
     * @return the Fully Qualified Class Name (FQCN) for the role.
     */
    public final Class getHandlerClassForClass( final Class className )
    {
        final Class handler = (Class)m_handlerNames.get( className );

        if( null == handler && null != m_parent )
        {
            return m_parent.getHandlerClassForClass( className );
        }

        return handler;
    }

    /**
     * Retrieves the default class name for the specified role.  This
     * is only called when the configuration does not specify the
     * class explicitly.  If this RoleManager does not have the match,
     * and there is a parent RoleManager, the parent will be asked
     * to resolve the class name.
     *
     * @param role  The role that has a default implementation.
     * @return the Fully Qualified Class Name (FQCN) for the role.
     */
    public final Class[] getClassesForRole( final String role )
    {
        final List classes = (List)m_classNames.get( role );

        if( null == classes && null != m_parent )
        {
            return m_parent.getClassesForRole( role );
        }

        return (Class[])classes.toArray( new Class[]{} );
    }

    /**
     * Retrieves a default class name for a role/hint combination.
     * This is only called when a role is mapped to a
     * DefaultComponentSelector, and the configuration elements use
     * shorthand names for the type of component.  If this RoleManager
     * does not have the match, and there is a parent RoleManager, the
     * parent will be asked to resolve the class name.
     *
     * @param klass The class to get rolename for
     * @return the FQCN for the role/hint combination.
     */
    public final String getRoleForClass( final Class klass )
    {
        final String role = (String)m_classNames.get( klass );

        if( null == role )
        {
            if( null != m_parent )
            {
                return m_parent.getRoleForClass( klass );
            }
            else
            {
                return EMPTY_STRING;
            }
        }

        return role;
    }
}
