/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included  with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.util;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;

/**
 * Configurable RoleManager implementation.  It populates the RoleManager
 * from a configuration hierarchy.  This is based on the DefaultRoleManager
 * in the org.apache.avalon.component package.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.9 $ $Date: 2002/05/12 11:13:45 $
 * @since 4.1
 */
public class ConfigurableRoleManager
    extends AbstractRoleManager
    implements Configurable
{
    /**
     * Default constructor--this RoleManager has no parent.
     */
    public ConfigurableRoleManager()
    {
        super( null );
    }

    /**
     * Alternate constructor--this RoleManager has the specified
     * parent.
     *
     * @param parent  The parent <code>RoleManager</code>.
     */
    public ConfigurableRoleManager( RoleManager parent )
    {
        super( parent, Thread.currentThread().getContextClassLoader() );
    }

    /**
     * Alternate constructor--this RoleManager has the specified
     * parent and a classloader.
     *
     * @param parent  The parent <code>RoleManager</code>.
     */
    public ConfigurableRoleManager( RoleManager parent, ClassLoader loader )
    {
        super( parent, loader );
    }

    /**
     * Reads a configuration object and creates the role, shorthand,
     * and class name mapping.
     *
     * @param configuration  The configuration object.
     * @throws ConfigurationException if the configuration is malformed
     */
    public final void configure( final Configuration configuration )
        throws ConfigurationException
    {
        final Map shorts = new HashMap();
        final Map classes = new HashMap();
        final Map handlers = new HashMap();

        final Configuration[] roles = configuration.getChildren( "role" );

        for( int i = 0; i < roles.length; i++ )
        {
            final String role = roles[ i ].getAttribute( "name" );
            Configuration[] components = roles[ i ].getChildren( "component" );

            for( int j = 0; j < components.length; j++ )
            {
                final String shorthand = components[ j ].getAttribute( "shorthand" );
                final String className =
                    components[ j ].getAttribute( "class", null );
                final String handlerClassName =
                    components[ j ].getAttribute( "handler",
                                                  "org.apache.excalibur.fortress.handler.PerThreadComponentHandler" );

                setup( shorts, classes, handlers, shorthand, role, className, handlerClassName );
            }
        }

        m_shorthands = Collections.unmodifiableMap( shorts );
        m_classNames = Collections.unmodifiableMap( classes );
        m_handlerNames = Collections.unmodifiableMap( handlers );
    }
}
