package org.apache.excalibur.fortress.util;

import java.io.File;
import org.apache.avalon.excalibur.instrument.InstrumentManager;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.excalibur.event.Queue;
import org.apache.excalibur.mpool.PoolManager;

/**
 * Helper class to create a context for the ContextManager.
 */
public class ContextBuilder implements ContextManagerConstants
{
    private final DefaultContext context;

    /**
     * Creates a context builder and initializes it with default values.
     * The default values are:
     *
     * <ul>
     * <li>THREADS_CPU =  2</li>
     * <li>THREAD_TIMEOUT = 1000</li>
     * <li>CONTEXT_DIRECTORY = "./"</li>
     * <li>WORK_DIRECTORY = "/tmp"</li>
     * <li>LOG_CATEGORY = "test"</li>
     * <li>CONTEXT_CLASSLOADER = the thread context class loader</li>
     * </ul>
     */
    public ContextBuilder()
    {
        this( createDefaultContext() );
    }

    /**
     * Creates a context builder and initializes it with default values.
     *
     * @param parent parent context with default values.
     */
    public ContextBuilder( Context parent )
    {
        context = new OverridableContext( parent );
    }

    /**
     * Creates a default context.
     */
    public static final Context createDefaultContext()
    {
        return createDefaultContext( Thread.currentThread().getContextClassLoader() );
    }

    /**
     * Creates a default context.
     */
    public static final Context createDefaultContext( ClassLoader contextClassLoader )
    {
        DefaultContext defaultContext = new DefaultContext();

        defaultContext.put( THREADS_CPU, new Integer( 2 ) );
        defaultContext.put( THREAD_TIMEOUT, new Long( 1000 ) );
        defaultContext.put( CONTEXT_DIRECTORY, new File( "./" ) );
        defaultContext.put( WORK_DIRECTORY, new File( "/tmp" ) );
        defaultContext.put( LOG_CATEGORY, "test" );
        defaultContext.put( CONTEXT_CLASSLOADER, contextClassLoader );

        defaultContext.makeReadOnly();

        return defaultContext;
    }

    /**
     * Finalizes the context and returns it.
     */
    public Context getContext()
    {
        context.makeReadOnly();
        return context;
    }

    public void setCommandQueue( Queue commandQueue )
    {
        context.put( COMMAND_QUEUE, commandQueue );
    }

    public void setComponentManagerParent( ComponentManager componentManager )
    {
        context.put( COMPONENT_MANAGER_PARENT, componentManager );
    }

    public void setComponentManager( ComponentManager componentManager )
    {
        context.put( COMPONENT_MANAGER, componentManager );
    }

    public void setContainerClass( String containerClass )
        throws ClassNotFoundException
    {
        ClassLoader classLoader = null;
        try
        {
            classLoader = (ClassLoader)context.get( CONTEXT_CLASSLOADER );
        }
        catch( ContextException ce )
        {
            classLoader = Thread.currentThread().getContextClassLoader();
        }

        context.put( CONTAINER_CLASS, classLoader.loadClass( containerClass ) );
    }

    public void setContainerClass( Class containerClass )
    {
        context.put( CONTAINER_CLASS, containerClass );
    }

    public void setContainerConfiguration( Configuration config )
    {
        context.put( CONFIGURATION, config );
        context.put( CONFIGURATION_URI, null );
    }

    public void setContainerConfiguration( String location )
    {
        context.put( CONFIGURATION_URI, location );
    }

    public void setContextClassLoader( ClassLoader loader )
    {
        context.put( CONTEXT_CLASSLOADER, loader );
    }

    public void setContextDirectory( File file )
    {
        context.put( CONTEXT_DIRECTORY, file );
    }

    public void setContextDirectory( String directory )
    {
        context.put( CONTEXT_DIRECTORY, new File( directory ) );
    }

    public void setLoggerCategory( String category )
    {
        context.put( LOG_CATEGORY, category );
    }

    public void setLoggerManager( LoggerManager logManager )
    {
        context.put( LOGGER_MANAGER, logManager );
        context.put( LOGGER_MANAGER_CONFIGURATION, null );
        context.put( LOGGER_MANAGER_CONFIGURATION_URI, null );
    }

    public void setLoggerManagerConfiguration( Configuration config )
    {
        context.put( LOGGER_MANAGER_CONFIGURATION, config );
        context.put( LOGGER_MANAGER_CONFIGURATION_URI, null );
    }

    public void setLoggerManagerConfiguration( String location )
    {
        context.put( LOGGER_MANAGER_CONFIGURATION_URI, location );
    }

    public void setMarkerManagerConfiguration( String location )
    {
        context.put( MARKER_MANAGER_CONFIGURATION_URI, location );
    }

    public void setMarkerManagerConfiguration( Configuration config )
    {
        context.put( MARKER_MANAGER_CONFIGURATION, config );
        context.put( MARKER_MANAGER_CONFIGURATION_URI, null );
    }

    public void setInstrumentManager( InstrumentManager profiler )
    {
        context.put( INSTRUMENT_MANAGER_CONFIGURATION, profiler );
        context.put( INSTRUMENT_MANAGER_CONFIGURATION, null );
        context.put( INSTRUMENT_MANAGER_CONFIGURATION_URI, null );
    }

    public void setInstrumentManagerConfiguration( Configuration config )
    {
        context.put( INSTRUMENT_MANAGER_CONFIGURATION, config );
        context.put( INSTRUMENT_MANAGER_CONFIGURATION_URI, null );
    }

    public void setInstrumentManagerConfiguration( String location )
    {
        context.put( INSTRUMENT_MANAGER_CONFIGURATION_URI, location );
    }

    public void setNumberOfThreadsPerCPU( int numberOfThreads )
    {
        context.put( THREADS_CPU, new Integer( numberOfThreads ) );
    }

    public void setPoolManager( PoolManager poolManager )
    {
        context.put( POOL_MANAGER, poolManager );
    }

    public void setRoleManager( RoleManager roleManager )
    {
        context.put( ROLE_MANAGER, roleManager );
    }

    public void setRoleManagerConfiguration( Configuration config )
    {
        context.put( ROLE_MANAGER_CONFIGURATION, config );
        context.put( ROLE_MANAGER_CONFIGURATION_URI, null );
    }

    public void setRoleManagerConfiguration( String location )
    {
        context.put( ROLE_MANAGER_CONFIGURATION_URI, location );
    }

    public void setThreadTimeout( long timeout )
    {
        context.put( THREAD_TIMEOUT, new Long( timeout ) );
    }

    public void setWorkDirectory( File file )
    {
        context.put( WORK_DIRECTORY, file );
    }

    public void setWorkDirectory( String directory )
    {
        setWorkDirectory( new File( directory ) );
    }
}
