/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included  with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.util;

import java.util.ArrayList;
import java.util.Iterator;
import org.apache.avalon.excalibur.instrument.manager.DefaultInstrumentManager;
import org.apache.avalon.excalibur.logger.LogKitLoggerManager;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.DefaultComponentManager;
import org.apache.avalon.framework.component.DefaultComponentSelector;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.configuration.DefaultConfigurationSerializer;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.avalon.framework.logger.ConsoleLogger;
import org.apache.avalon.framework.logger.Logger;
import org.apache.excalibur.event.Queue;
import org.apache.excalibur.event.command.CommandManager;
import org.apache.excalibur.mpool.DefaultPoolManager;
import org.apache.excalibur.mpool.PoolManager;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.ResourceSourceFactory;
import org.apache.excalibur.source.impl.SourceResolverImpl;

/**
 * The ContextManager is used to manage the values in a Container's Context.
 * Use this helper class to create the initial context to pass into the
 * ContainerManager.  Its purpose is to add the default values, and give
 * convenient methods to override those defaults.  Once you get an instance
 * of the Context, it is made read-only and returned.  No further operations
 * will be possible on it.
 *
 * <p>You can get two different contexts from the ContextManager: the child
 * context and the container manager context. The former contains all managers,
 * such as the pool manager etc. necessary for a child container to create
 * additional child containers. The container manager context contains all
 * of the child context, but also initialization parameters for the
 * container, such as a Configuration object, a ComponentManager, etc., that
 * the container wants, but does not want to pass on to its children.
 *
 * <p>You would typically use the container manager context to initialize
 * the container manager, and let it pass the child context on to the container.
 *
 * <p>The ContextManager will sometimes create new components, such as a component
 * manager, a pool manager, etc. It will manage these components and dispose
 * of them properly when it itself is disposed .
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:leo.sutic@inspireinfrastructure.com">Leo Sutic</a>
 * @author <a href="mailto:proyal@apache.org">Peter Royal</a>
 * @version CVS $Revision: 1.19 $ $Date: 2002/05/13 12:17:39 $
 * @since 4.1
 */
public class ContextManager implements ContextManagerConstants
{

    private static final Configuration EMPTY_CONFIG;

    static
    {
        DefaultConfiguration config = new DefaultConfiguration( "", "", "", "" );
        config.makeReadOnly();
        EMPTY_CONFIG = config;
    }

    /**
     * The root context.
     */
    private final Context rootContext;

    /**
     * The context of the new container. This context has the rootContext
     * as its parent. Put everything here that you want the new container
     * to have in its own context.
     */
    private final DefaultContext childContext;

    /**
     * Container manager's context. This context has the child context
     * as parent. Put things here that you want the container manager
     * to see, but do not wish to expose to the container.
     */
    private final DefaultContext containerManagerContext;

    /**
     * New context passed in, maybe from a ContextBuilder.
     * This context may not have objects, but rather URI's or
     * other pointers. The ContextManager inspects it for
     * elements it can use to create component managers etc.
     */
    private Logger logger;
    private final Logger primordialLogger;
    private ComponentManager manager;

    /**
     * The components that are "owned" by this context and should
     * be disposed by it. Any manager that is created as a result
     * of it not being in the rootContext, or having been created
     * by the ContextManager should go in here.
     */
    private final ArrayList ownedComponents = new ArrayList();

    private final DefaultConfigurationBuilder configBuilder =
        new DefaultConfigurationBuilder();
    private final DefaultConfigurationSerializer configSerializer =
        new DefaultConfigurationSerializer();

    /**
     * Create a new ContextManager.
     *
     * @param rootContext the default values.
     * @param overrides   values to be overridden in the root context. This parameter is typically
     *                    created with a ContextBuilder.
     * @param logger      logger to use when creating new components.
     */
    public ContextManager( Context rootContext, Logger logger )
    {
        this.rootContext = rootContext;
        this.childContext = new OverridableContext( this.rootContext );
        this.containerManagerContext = new OverridableContext( childContext );
        this.logger = logger;

        this.primordialLogger = new ConsoleLogger();
    }

    public void assumeOwnership( Object o )
    {
        if( o == null )
        {
            throw new IllegalArgumentException( "Can not assume ownership of a null!" );
        }
        ownedComponents.add( o );
    }

    public void initialize() throws Exception
    {
        initializeOwnComponentManager();
        initializeLoggerManager();
        initializeRoleManager();
        initializeComponentManager();
        initializeCommandQueue();
        initializePoolManager();
        initializeContext();
        initializeInstrumentManager();
        initializeConfiguration();
        initializeMarkerManager();

        childContext.makeReadOnly();
        containerManagerContext.makeReadOnly();
    }

    protected void initializeConfiguration() throws Exception
    {
        try
        {
            containerManagerContext.put( CONFIGURATION, rootContext.get( CONFIGURATION ) );
        }
        catch( ContextException ce )
        {
            Configuration containerConfig = getConfiguration( CONFIGURATION, CONFIGURATION_URI );
            if( containerConfig == null )
            {
                // No config.
                // Does the parent supply a logger manager?
                try
                {
                    containerManagerContext.get( CONFIGURATION );

                    // OK, done.
                    return;
                }
                catch( ContextException cex )
                {
                    // Guess there is none.
                    return;
                }
            }
            else
            {
                containerManagerContext.put( CONFIGURATION, containerConfig );
            }
        }
    }

    protected void initializeMarkerManager() throws Exception
    {
        childContext.put(
            MARKER_MANAGER_CONFIGURATION,
            getConfiguration( MARKER_MANAGER_CONFIGURATION, MARKER_MANAGER_CONFIGURATION_URI )
        );
    }

    protected void initializeContext() throws Exception
    {
        copyEntry( CONTAINER_CLASS );
        copyEntry( PARAMETERS );
    }

    protected void copyEntry( String key ) throws Exception
    {
        try
        {
            containerManagerContext.put( key, rootContext.get( key ) );
        }
        catch( ContextException ce )
        {
        }
    }

    /**
     * Disposes all items that this ContextManager has created.
     */
    public void dispose()
    {
        // Dispose owned components
        Iterator ownedComponentsIter = ownedComponents.iterator();
        while( ownedComponentsIter.hasNext() )
        {
            Object o = ownedComponentsIter.next();

            try
            {
                if( o instanceof Startable )
                {
                    ( (Startable)o ).stop();
                }

                if( o instanceof Disposable )
                {
                    ( (Disposable)o ).dispose();
                }
            }
            catch( Exception e )
            {
                getLogger().warn( "Unable to dispose of owned component " + o.getClass().getName(), e );
            }
        }
        ownedComponents.clear();
    }

    protected Object get( Context context, String key, Object defaultValue )
    {
        try
        {
            return context.get( key );
        }
        catch( ContextException ce )
        {
            return defaultValue;
        }
    }

    /**
     * Will set up a ComponentManager if none is supplied.
     *
     * The postcondition is that childContext.get( Container.COMPONENT_MANAGER )
     * should return a valid logger manager.
     */
    protected void initializeComponentManager() throws Exception
    {
        try
        {
            childContext.put( COMPONENT_MANAGER, rootContext.get( COMPONENT_MANAGER ) );
            return;
        }
        catch( ContextException ce )
        {
        }

        // See if we can inherit from the parent...
        try
        {
            childContext.get( COMPONENT_MANAGER );

            // OK, done.
            return;
        }
        catch( ContextException ce )
        {
            // No ComponentManager available anywhere. (Set one up.)
        }

        //
        // TODO: Load configuration from COMPONENT_MANAGER_CONFIGURATION or COMPONENT_MANAGER_CONFIGURATION_URI
        //       and create a proper CM from those (ECM?). Need a container CM for that, though. /LS
        //
        ComponentManager cm = new DefaultComponentManager(
            (ComponentManager)get( rootContext, COMPONENT_MANAGER_PARENT, null ) );
        assumeOwnership( cm );
        containerManagerContext.put( COMPONENT_MANAGER, cm );
    }

    protected void initializeCommandQueue() throws Exception
    {
        try
        {
            childContext.put( COMMAND_QUEUE, rootContext.get( COMMAND_QUEUE ) );
            return;
        }
        catch( ContextException ce )
        {
        }

        try
        {
            childContext.get( COMMAND_QUEUE );
            return;
        }
        catch( ContextException ce )
        {
        }

        CommandManager cm = new CommandManager();
        assumeOwnership( cm );
        childContext.put( COMMAND_QUEUE, cm.getCommandQueue() );
    }

    protected void initializePoolManager() throws Exception
    {
        try
        {
            childContext.put( POOL_MANAGER, rootContext.get( POOL_MANAGER ) );
            return;
        }
        catch( ContextException ce )
        {
        }

        PoolManager pm = new DefaultPoolManager( (Queue)childContext.get( COMMAND_QUEUE ) );
        assumeOwnership( pm );
        childContext.put( POOL_MANAGER, pm );
    }

    protected void initializeRoleManager() throws Exception
    {
        try
        {
            childContext.put( ROLE_MANAGER, rootContext.get( ROLE_MANAGER ) );
            return;
        }
        catch( ContextException ce )
        {
        }

        Configuration config = getConfiguration( ROLE_MANAGER_CONFIGURATION, ROLE_MANAGER_CONFIGURATION_URI );

        if( config == null )
        {
            // See if we can inherit from the parent...
            try
            {
                childContext.get( ROLE_MANAGER );

                // OK, done.
                return;
            }
            catch( ContextException ce )
            {
                // No RoleManager available anywhere. (Set one up.)

                RoleManager rm = new ExcaliburRoleManager();

                assumeOwnership( rm );
                childContext.put( ROLE_MANAGER, rm );

                return;
            }
        }

        ConfigurableRoleManager rm = new ConfigurableRoleManager( new ExcaliburRoleManager() );

        rm.configure( config );

        assumeOwnership( rm );
        childContext.put( ROLE_MANAGER, rm );
    }

    /**
     * Get a reference to the initial ComponentManager used by the ContainerManager
     * to hold the Components used for parsing the config files and setting up the
     * environment.
     */
    protected void initializeOwnComponentManager() throws Exception
    {
        DefaultComponentManager manager = new DefaultComponentManager();

        DefaultComponentSelector selector = new DefaultComponentSelector();
        ResourceSourceFactory resource = new ResourceSourceFactory();
        resource.enableLogging( getLogger() );
        selector.put( "resource", resource );

        manager.put( resource.ROLE + "Selector", selector );

        SourceResolverImpl resolver = new SourceResolverImpl();
        resolver.enableLogging( getLogger() );
        resolver.contextualize( this.childContext );
        resolver.compose( manager );

        manager.put( resolver.ROLE, resolver );

        manager.makeReadOnly();

        assumeOwnership( manager );

        this.manager = manager;
    }

    protected Configuration getConfiguration( String configKey, String uriKey )
    {
        Configuration config = null;
        SourceResolver resolver = null;
        Source src = null;

        try
        {
            return (Configuration)rootContext.get( configKey );
        }
        catch( ContextException ce )
        {
        }

        String configUri = null;
        try
        {
            configUri = (String)rootContext.get( uriKey );
        }
        catch( ContextException ce )
        {
            return null;
        }

        try
        {
            resolver = (SourceResolver)manager.lookup( SourceResolver.ROLE );
            src = resolver.resolveURI( configUri );

            return configBuilder.build( src.getInputStream() );
        }
        catch( Exception e )
        {
            getLogger().warn( "Could not read configuration file: " + configUri, e );
            return null;
        }
        finally
        {
            resolver.release( src );
            manager.release( resolver );
        }
    }

    /**
     * Finalizes and returns the context.
     */
    public Context getContainerManagerContext()
    {
        return containerManagerContext;
    }

    /**
     * Finalizes and returns the context.
     */
    public Context getChildContext()
    {
        return childContext;
    }

    /**
     * Get a reference the Logger.
     */
    protected Logger getLogger()
    {
        if( logger == null )
        {
            return primordialLogger;
        }
        else
        {
            return logger;
        }
    }

    /**
     * Will set up a LogKitLoggerManager if none is supplied.  This can be
     * overridden if you don't want a LogKitLoggerManager.
     *
     * The postcondition is that childContext.get( Container.LOGGER_MANAGER )
     * should return a valid logger manager.
     */
    protected void initializeLoggerManager() throws Exception
    {
        try
        {
            // Try copying an already existing logger manager from the override context.

            childContext.put( LOGGER_MANAGER, rootContext.get( LOGGER_MANAGER ) );
        }
        catch( ContextException ce )
        {
            // Should we set one up?
            // Try to get a configuration for it...
            Configuration loggerManagerConfig = getConfiguration( LOGGER_MANAGER_CONFIGURATION, LOGGER_MANAGER_CONFIGURATION_URI );
            if( loggerManagerConfig == null )
            {
                // No config.
                // Does the parent supply a logger manager?
                try
                {
                    childContext.get( LOGGER_MANAGER );

                    // OK, done.
                    return;
                }
                catch( ContextException cex )
                {
                }
            }

            String logCategory = (String)get( rootContext, LOG_CATEGORY, get( childContext, LOG_CATEGORY, null ) );

            LogKitLoggerManager logManager = new LogKitLoggerManager( logCategory );

            logManager.contextualize( rootContext );
            logManager.configure( loggerManagerConfig );

            assumeOwnership( logManager );

            childContext.put( LOGGER_MANAGER, logManager );
        }
        finally
        {
            // Since we now have a LoggerManager, we can update the this.logger field
            // if it is null and start logging to the "right" logger.

            if( this.logger == null )
            {
                getLogger().info( "Switching to default Logger provided by LoggerManager." );

                LoggerManager loggerManager = (LoggerManager)childContext.get( LOGGER_MANAGER );
                this.logger = loggerManager.getDefaultLogger();
            }
        }
    }

    /**
     * Will set up a LogKitLoggerManager if none is supplied.  This can be
     * overridden if you don't want a LogKitLoggerManager.
     *
     * The postcondition is that childContext.get( Container.LOGGER_MANAGER )
     * should return a valid logger manager.
     */
    protected void initializeInstrumentManager() throws Exception
    {
        try
        {
            // Try copying an already existing logger manager from the override context.

            childContext.put( INSTRUMENT_MANAGER, rootContext.get( INSTRUMENT_MANAGER ) );
        }
        catch( ContextException ce )
        {
            // Should we set one up?
            // Try to get a configuration for it...
            Configuration profilerConfig = getConfiguration( INSTRUMENT_MANAGER_CONFIGURATION, INSTRUMENT_MANAGER_CONFIGURATION_URI );
            if( profilerConfig == null )
            {
                // No config.
                // Does the parent supply a logger manager?
                try
                {
                    childContext.get( INSTRUMENT_MANAGER );

                    // OK, done.
                    return;
                }
                catch( ContextException cex )
                {
                }

                profilerConfig = EMPTY_CONFIG;
            }

            DefaultInstrumentManager instrumentManager = new DefaultInstrumentManager( INSTRUMENT_MANAGER_NAME );
            instrumentManager.enableLogging( getLogger() );
            instrumentManager.configure( profilerConfig );
            instrumentManager.initialize();

            assumeOwnership( instrumentManager );

            childContext.put( INSTRUMENT_MANAGER, instrumentManager );
        }
        finally
        {
            // Since we now have a LoggerManager, we can update the this.logger field
            // if it is null and start logging to the "right" logger.

            if( this.logger == null )
            {
                getLogger().info( "Switching to default Logger provided by LoggerManager." );

                LoggerManager loggerManager = (LoggerManager)childContext.get( LOGGER_MANAGER );
                this.logger = loggerManager.getDefaultLogger();
            }
        }
    }
}
