/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included  with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.util;

/**
 * RoleManager Interface, use this to specify the Components and how they
 * correspond to easy shorthand names.  The RoleManager assumes a flat
 * relationship of shorthand names to classes, and classes to roles.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.4 $ $Date: 2002/04/04 16:04:28 $
 * @since 4.1
 */
public interface RoleManager
{
    /**
     * Find the Class for the given shorthand name.  If there is no
     * correspondence between the class and the shorthand name, the method
     * returns <code>null</code>.  If this RoleManager does not have the match,
     * and there is a parent RoleManager, the parent will be asked to resolve
     * the request.
     */
    Class getClassForName( final String shorthandName );

    /**
     * This method is merely a hint for serialization.  If this RoleManager does
     * not have the match, and there is a parent RoleManager, the parent will be
     * asked to resolve the request.
     */
    String getNameForClass( final Class component );

    /**
     * Get the Role name for a specific class.  If the class does not belong to
     * a Component, or the Role is not easily determinable, this method will return
     * <code>null</code>.  If this RoleManager does not have the match, and
     * there is a parent RoleManager, the parent will be asked to resolve the
     * request.
     */
    String getRoleForClass( final Class component );

    /**
     * Get an array of classes registered with the role manager that implement a
     * role.  If this RoleManager does not have the match, and there is a parent
     * RoleManager, the parent will be asked to resolve the request.
     */
    Class[] getClassesForRole( final String role );

    /**
     * Retrieves the handler class name for the specified class.  This
     * is called for every Component Implementation.  If this RoleManager does
     * not have the match, and there is a parent RoleManager, the parent will be
     * asked to resolve the handler's class name.
     *
     * @param class  The class of the Component in question.
     * @return the Class instance of the ComponentHandler.
     */
    Class getHandlerClassForClass( final Class className );
}
