/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.excalibur.fortress.test;

import java.net.URL;
import junit.framework.TestCase;
import junit.textui.TestRunner;
import org.apache.avalon.excalibur.component.ExcaliburComponentManager;
import org.apache.avalon.excalibur.datasource.DataSourceComponent;
import org.apache.avalon.excalibur.logger.DefaultLogKitManager;
import org.apache.avalon.excalibur.monitor.Monitor;
import org.apache.avalon.excalibur.testcase.CascadingAssertionFailedError;
import org.apache.avalon.excalibur.testcase.LatchedThreadGroup;
import org.apache.avalon.excalibur.xml.Parser;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.avalon.framework.logger.LogKitLogger;
import org.apache.avalon.framework.logger.Logger;
import org.apache.excalibur.fortress.ContainerManager;
import org.apache.excalibur.fortress.DefaultContainerManager;
import org.apache.excalibur.fortress.util.ContextBuilder;
import org.apache.excalibur.fortress.util.ContextManager;

/**
 * Used as a basis for the PoolComparisonProfile Tests
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version $Id: ContainerProfile.java,v 1.18 2002/06/05 16:45:54 crafterm Exp $
 */
public final class ContainerProfile
    extends TestCase
{
    /**
     * The TEST_SIZE defines the overall size of the tests.  Decreasing this will
     *  decrease the time the test takes to run, but also decrease its efficiency.
     */
    protected static final int TEST_SIZE = 5000;
    protected static final int THREADS = 1;

    protected static Throwable m_throwable = null;
    protected static int m_getCount = 0;

    protected Logger m_logger;
    protected org.apache.log.Logger m_logKitLogger;

    /*---------------------------------------------------------------
     * Constructors
     *-------------------------------------------------------------*/
    public ContainerProfile( String name )
    {
        super( name );

        // Set to debug to see more useful information.
        org.apache.log.Logger logger =
            org.apache.log.Hierarchy.getDefaultHierarchy().getLoggerFor( "test" );
        logger.setPriority( org.apache.log.Priority.INFO );
        m_logKitLogger = logger;
        m_logger = new LogKitLogger( logger );
    }

    /*---------------------------------------------------------------
     * ECM vs. ContainerManager StartTimes
     *-------------------------------------------------------------*/
    /**
     * Compare the ECM and ContainerManager start times.
     */
    public void testCompare_ECM_ContainerManager_UseageTime()
        throws Exception
    {
        resetMemory(); // Start clean

        long ecmStart = System.currentTimeMillis();
        ExcaliburComponentManager manager = new ExcaliburComponentManager();
        Context context = new DefaultContext();
        manager.setLogger( m_logKitLogger );
        manager.contextualize( context );
        DefaultLogKitManager logmanager = new DefaultLogKitManager();
        logmanager.setLogger( m_logKitLogger );
        logmanager.contextualize( context );
        logmanager.configure( getLogKitConfig() );
        manager.setLogKitManager( logmanager );
        manager.configure( getContainerConfig() );
        manager.initialize();
        long ecmStop = System.currentTimeMillis();
        long ecmDuration = ecmStop - ecmStart;

        resetMemory(); // Start clean

        long cmStart = System.currentTimeMillis();
        ContextBuilder contextBuilder = new ContextBuilder();
        contextBuilder.setContainerClass( "org.apache.excalibur.fortress.test.TestContainer" );
        contextBuilder.setContextDirectory( "./" );
        contextBuilder.setWorkDirectory( "./" );
	contextBuilder.setMarkerManagerConfiguration( "resource://org/apache/excalibur/fortress/markers.xml" );
        contextBuilder.setContainerConfiguration( "resource://org/apache/excalibur/fortress/test/ContainerProfile.xconf" );
        contextBuilder.setLoggerManagerConfiguration( "resource://org/apache/excalibur/fortress/test/ContainerProfile.xlog" );
        contextBuilder.setRoleManagerConfiguration( "resource://org/apache/excalibur/fortress/test/ContainerProfile.roles" );

        ContextManager contextManager = new ContextManager( contextBuilder.getContext(), null );
        contextManager.initialize();

        ContainerManager cm = new DefaultContainerManager( contextManager );
        cm.initialize();
        TestContainer container = (TestContainer)cm.getContainer();
        assertNotNull( container );
        long cmStop = System.currentTimeMillis();
        long cmDuration = cmStop - cmStart;

        // Show a summary
        if( m_logger.isInfoEnabled() )
        {
            m_logger.info( "Test Case: ECM_ContainerManager_StartTime" );
            m_logger.info( "     ECM time = " + ecmDuration + "ms." );
            m_logger.info( "     ContainerManager time = " + cmDuration + "ms." );

            double mult;
            mult = ( cmDuration > 0 ? ( ecmDuration * 100 / cmDuration ) / 100.0 : Float.POSITIVE_INFINITY );
            m_logger.info( "  => ContainerManager is " + mult + " X as fast as ExcaliburComponentManager on init." );
            mult = ( ecmDuration > 0 ? ( cmDuration * 100 / ecmDuration ) / 100.0 : Float.POSITIVE_INFINITY );
            m_logger.info( "  => ExcaliburComponentManager is " + mult + " X as fast as ContainerManager on init." );
        }

        resetMemory();

        lookupTest( "Test Case: ECM_ContainerManager_UseageTime", container.getCM(), manager );

        resetMemory();

        ecmStart = System.currentTimeMillis();
        manager.dispose();
        ecmStop = System.currentTimeMillis();
        ecmDuration = ecmStop - ecmStart;

        resetMemory();

        cmStart = System.currentTimeMillis();
        cm.dispose();
        cmStop = System.currentTimeMillis();
        cmDuration = cmStop - cmStart;

        // Show a summary
        if( m_logger.isInfoEnabled() )
        {
            m_logger.info( "Test Case: ECM_ContainerManager_KillTime" );
            m_logger.info( "     ECM time = " + ecmDuration + "ms." );
            m_logger.info( "     ContainerManager time = " + cmDuration + "ms." );

            double mult;
            mult = ( cmDuration > 0 ? ( ecmDuration * 100 / cmDuration ) / 100.0 : Float.POSITIVE_INFINITY );
            m_logger.info( "  => ContainerManager is " + mult + " X as fast as ExcaliburComponentManager on dispose." );
            mult = ( ecmDuration > 0 ? ( cmDuration * 100 / ecmDuration ) / 100.0 : Float.POSITIVE_INFINITY );
            m_logger.info( "  => ExcaliburComponentManager is " + mult + " X as fast as ContainerManager on dispose." );
        }
    }

    /*---------------------------------------------------------------
     * Utility Methods
     *-------------------------------------------------------------*/
    protected void resetMemory()
    {
        System.gc();
        System.gc();

        // Let the system settle down.
        try
        {
            Thread.sleep( 50 );
        }
        catch( InterruptedException e )
        {
        }
        Runtime runtime = Runtime.getRuntime();
        m_logger.debug( "Memory: " + ( runtime.totalMemory() - runtime.freeMemory() ) );
    }

    /**
     * Get the LogKitManager Config file
     */
    protected Configuration getLogKitConfig()
        throws Exception
    {
        final String resourceName = this.getClass().getName().replace( '.', '/' ) + ".xlog";
        URL resource = this.getClass().getClassLoader().getResource( resourceName );
        return loadConfig( resource );
    }

    /**
     * Get the Container Config file
     */
    protected Configuration getContainerConfig()
        throws Exception
    {
        final String resourceName = this.getClass().getName().replace( '.', '/' ) + ".xconf";
        URL resource = this.getClass().getClassLoader().getResource( resourceName );
        return loadConfig( resource );
    }

    /**
     * Load Config
     */
    protected Configuration loadConfig( URL path )
        throws Exception
    {
        final DefaultConfigurationBuilder builder = new DefaultConfigurationBuilder();
        return builder.build( path.openStream() );
    }

    /**
     * Get the short class name
     */
    protected String getShortClassName( Object o )
    {
        String name = o.getClass().getName();
        int pos = name.lastIndexOf( '.' );
        if( pos > 0 )
        {
            name = name.substring( pos + 1 );
        }
        return name;
    }

    /**
     * The guts of the various test cases.  Will dispose the pools
     */
    protected void lookupTest( String name, ComponentManager cmA, ComponentManager cmB )
        throws Exception
    {
        m_logger.info( "Test Case: " + name );

        // Get the short class names
        final String cmAName = getShortClassName( cmA );
        final String cmBName = getShortClassName( cmB );

        // Start clean
        resetMemory();


        // Get the time for ecm
        final long cmADuration = getLookupRunTime( cmA );
        m_logger.info( "     " + cmAName + " time = " + cmADuration + "ms. to use " + TEST_SIZE + " calls on 3 components." );
        resetMemory();


        // Get the time for manager
        final long cmBDuration = getLookupRunTime( cmB );
        m_logger.info( "     " + cmBName + " time = " + cmBDuration + "ms. to use " + TEST_SIZE + " calls on 3 components." );
        resetMemory();

        // Show a summary
        if( m_logger.isInfoEnabled() )
        {
            double mult;
            mult = ( cmADuration > 0 ? ( cmBDuration * 100 / cmADuration ) / 100.0 : Float.POSITIVE_INFINITY );
            m_logger.info( "  => " + cmAName + " is " + mult + " X as fast as " + cmBName + "." );

            mult = ( cmBDuration > 0 ? ( cmADuration * 100 / cmBDuration ) / 100.0 : Float.POSITIVE_INFINITY );
            m_logger.info( "  => " + cmBName + " is " + mult + " X as fast as " + cmAName + "." );
        }
    }

    protected long getLookupRunTime( ComponentManager manager )
    {
        // Create the runnable
        LookupRunner runnable = new LookupRunner( manager, m_logger );

        LatchedThreadGroup group = new LatchedThreadGroup( runnable, THREADS );
        group.enableLogging( m_logger );

        long duration;
        try
        {
            duration = group.go();
        }
        catch( Throwable t )
        {
            // Throwable could have been thrown by one of the tests.
            if( m_throwable == null )
            {
                m_throwable = t;
            }
            duration = 0;
        }

        if( m_throwable != null )
        {
            throw new CascadingAssertionFailedError( "Exception in test thread.", m_throwable );
        }

        assertTrue( "m_getCount == 0 (" + m_getCount + ")", m_getCount == 0 );

        return duration;
    }

    private static class LookupRunner implements Runnable
    {
        private Logger m_logger;
        private ComponentManager m_manager;
        private int m_getCount = 0;
        private Throwable m_throwable = null;

        public LookupRunner( ComponentManager manager, Logger logger )
        {
            m_manager = manager;
            m_logger = logger;
        }

        public int getCount()
        {
            return m_getCount;
        }

        public Throwable getThrowable()
        {
            return m_throwable;
        }

        public void run()
        {
            // Perform this threads part of the test.
            final int loops = ( TEST_SIZE / THREADS );
            for( int i = 0; i < loops; i++ )
            {
                Parser parser = null;
                DataSourceComponent datasource = null;
                Monitor monitor = null;

                try
                {
                    parser = (Parser)m_manager.lookup( Parser.ROLE );

                    // Make the loops hold the components longer than they are released, but only slightly.
                    Thread.yield();
                }
                catch( Throwable t )
                {
                    m_logger.error( "Unexpected error after " + m_getCount +
                                    " iterations retrieved for Parser", t );

                    if( m_throwable == null )
                    {
                        m_throwable = t;
                    }
                    return;
                }
                finally
                {
                    if( null != parser )
                    {
                        m_manager.release( parser );
                    }
                }
                /*
                try
                {
                    datasource = (DataSourceComponent) m_manager.lookup(DataSourceComponent.ROLE);

                    // Make the loops hold the components longer than they are released, but only slightly.
                    Thread.yield();
                }
                catch (Throwable t)
                {
                    m_logger.error( "Unexpected error after " + m_getCount +
                        " iterations retrieved for DataSourceComponent", t );

                    if (m_throwable == null) {
                        m_throwable = t;
                    }
                    return;
                }
                finally
                {
                    if ( null != datasource )
                    {
                        m_manager.release( datasource );
                    }
                }
                */
                try
                {
                    monitor = (Monitor)m_manager.lookup( Monitor.ROLE );

                    // Make the loops hold the components longer than they are released, but only slightly.
                    Thread.yield();
                }
                catch( Throwable t )
                {
                    m_logger.error( "Unexpected error after " + m_getCount +
                                    " iterations retrieved for DataSourceComponent", t );

                    if( m_throwable == null )
                    {
                        m_throwable = t;
                    }
                    return;
                }
                finally
                {
                    if( null != monitor )
                    {
                        m_manager.release( monitor );
                    }
                }
            }
        }
    }

    public static final void main( String[] args )
    {
        TestRunner.run( ContainerProfile.class );
    }
}

