package org.apache.bcel.proxy;

import org.apache.bcel.generic.*;
import org.apache.bcel.util.Repository;
import org.apache.bcel.Constants;
import org.apache.bcel.classfile.*;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class ProxyCreator {
    private JavaClass clazz = null;
    private InstructionFactory    _factory;
    private ConstantPoolGen     _cp;
    private ClassGen                _cg;
    private static String className;
    private static final String fieldName = "handler";
    private static final String fieldSetterName = "setInvocationHandler";
    private JavaClass[] interfaces = null;
    private Repository repository = null;
    private Map classFieldNames = null;
    private Map methodMap = null; //method name + method signature -> method object
    private Map interfaceMap = null; //method name + method signature -> interface object
    private Map exceptionMap = null; //method name + method signature -> exception list String[]
    private String indent = "";

    public ProxyCreator(String className, Repository repository, Class[] interfaces) throws ClassNotFoundException {
        if (isDebug()) debug(className + " implements");
        if (isDebug()) increaseIndent(4);
        JavaClass[] interfacesJc = new JavaClass[interfaces.length];
        String[] interfacesStr = new String[interfaces.length];
        for (int i = 0; i < interfaces.length; i++) {
            interfacesJc[i] = repository.loadClass(interfaces[i]);
            interfacesStr[i] = interfaces[i].getName();
            if (isDebug()) debug(interfacesStr[i]);
        }
        if (isDebug()) decreaseIndent(4);
        if (isDebug()) debug("{ // " + className);
        init(className, repository, interfacesJc, interfacesStr);
        if (isDebug()) debug("} // " + className + "\n");
    }
    
    public ProxyCreator(String className, Repository repository, JavaClass[] interfaces) {
        if (isDebug()) debug(className + " implements");
        if (isDebug()) increaseIndent(4);
        String[] interfacesStr = new String[interfaces.length];
        for (int i = 0; i < interfaces.length; i++) {
            interfacesStr[i] = interfaces[i].getClassName();
            if (isDebug()) debug(interfacesStr[i]);
        }
        if (isDebug()) decreaseIndent(4);
        if (isDebug()) debug("{ // " + className);
        init(className, repository, interfaces, interfacesStr);
        if (isDebug()) debug("} // " + className + "\n");
    }
    
    public ProxyCreator(String className, Repository repository, String[] interfaces) throws ClassNotFoundException {
        if (isDebug()) debug(className + " implements");
        if (isDebug()) increaseIndent(4);
        JavaClass[] interfacesJc = new JavaClass[interfaces.length];
        String[] interfacesStr = new String[interfaces.length];
        for (int i = 0; i < interfaces.length; i++) {
            interfacesJc[i] = repository.loadClass(interfaces[i]);
        }
        if (isDebug()) decreaseIndent(4);
        if (isDebug()) debug("{ // " + className);
        init(className, repository, interfacesJc, interfaces);
        if (isDebug()) debug("} // " + className + "\n");
    }

    private void init(String className, Repository repository, JavaClass[] interfaces, String[] interfacesStr) {
        this.className = className;
        this.repository = repository;
        this.interfaces = interfaces;
        _cg = new ClassGen(className, Proxy.class.getName(), "", Constants.ACC_PUBLIC | Constants.ACC_FINAL, interfacesStr);
        classFieldNames = new HashMap(interfaces.length * 2 + 5);
        methodMap = new HashMap(interfaces.length * 2 + 5);
        exceptionMap = new HashMap(interfaces.length * 2 + 5);
        interfaceMap = new HashMap(interfaces.length * 2 + 5);
        
        _cp = _cg.getConstantPool();
        _factory = new InstructionFactory(_cg, _cp);
        if (isDebug()) increaseIndent();
        clazz = create();
        if (isDebug()) decreaseIndent();
    }

    public JavaClass getJavaClass() {
        return this.clazz;
    }

    private JavaClass create() {
        if (isDebug()) debug("//CREATE FIELDS");
        createFields();
        if (isDebug()) debug("//FIELDS CREATED");
        
        /*
         * One method for initializer.
         * And one method for each interface method.
         */
        createInit();
        createSetter();
        inspectMethods(this.interfaces);
        Method[] methods = getMethods();
        for (int i = 0; i < methods.length; i++) {
            createMethod(methods[i]);
        }
        return _cg.getJavaClass();
    }

    // java.lang.Object special case: hashCode(), equals() and toString() will be proxied
    private static final Set proxiedMethodsOnObject = new HashSet(5);
    static {
        proxiedMethodsOnObject.add("hashCode()I");
        proxiedMethodsOnObject.add("equals(Ljava/lang/Object;)Z");
        proxiedMethodsOnObject.add("toString()Ljava/lang/String;");
    }

    private void inspectMethods(JavaClass[] interfaces) {
        JavaClass o;
        try {
            o = this.repository.loadClass("java.lang.Object");
        } catch (ClassNotFoundException e) {
            throw new Error(e);
        }
        Method[] oMethods = o.getMethods();
        for (int j = 0; j < oMethods.length; j++) {
            if (proxiedMethodsOnObject.contains(oMethods[j].getName() + oMethods[j].getSignature()) && oMethods[j].isPublic()) 
                inspectMethod(o, oMethods[j]);
        }
        for (int i = 0; i < interfaces.length; i++) {
            if (isDebug()) debug("//interface: " + interfaces[i].getClassName());
            Method[] methods = interfaces[i].getMethods();
            for (int j = 0; j < methods.length; j++) {
                inspectMethod(interfaces[i], methods[j]);
            }
        }
    }

    private void inspectMethod(JavaClass javaClass, Method method) {
        String sig = method.getName() + method.getSignature();
        if (isDebug()) debug("//method: " + sig);
        if (!methodMap.containsKey(sig)) {
            methodMap.put(sig, method);
            interfaceMap.put(sig, javaClass);
            String[] en = null;
            ExceptionTable et = method.getExceptionTable();
            if (et != null) en = et.getExceptionNames();
            if (en == null) en = new String[] {};
            exceptionMap.put(sig, en);
        } else {
            // if duplicate method signatures, then the exceptions thrown by the method is a union of the exceptions 
            // thrown by these duplicate methods
            ExceptionTable et = method.getExceptionTable();
            if (et != null) {
                String[] newList = joinArrays((String[]) exceptionMap.get(sig), 
                                                           et.getExceptionNames());
                exceptionMap.put(sig, newList);
            }
        }
    }

    private Method[] getMethods() {
        Set set =  methodMap.keySet();
        Method[] results = new Method[set.size()];
        int i = 0;
        for (Iterator iter = set.iterator(); iter.hasNext();) {
            results[i++] = (Method) methodMap.get(iter.next());
        }
        return results;
    }

    private static String[] joinArrays(String[] base, String[] add) {
        Set es = new HashSet((int)(base.length + add.length * 2.5));
        for (int i = 0; i < base.length; i++) {
            es.add(base[i]);
        }
        for (int i = 0; i < add.length; i++) {
            es.add(add[i]);
        }
        String[] results = new String[es.size()];
        return (String[]) es.toArray(results);
    }

    private void putClassFieldName(String key, int index) {
        String classFieldName = "class$" + index;
        classFieldNames.put(key, classFieldName);
        if (isDebug()) debug("//put classFieldName " + key + " -> " + classFieldName);
    }

    private void createFields() {
        /*
         * One private field for holding the invocation handler.
         * And one private and static field for each reference to a Class type (e.g. String.class).
         */
        FieldGen field;

        int index = 0;
        putClassFieldName("java.lang.Object", index++);
        for (int i = 0; i < this.interfaces.length; i++) {
            putClassFieldName(interfaces[i].getClassName(), index++);
            Method[] methods = interfaces[i].getMethods();
            for (int j = 0; j < methods.length; j++) {
                Method method = methods[j];
                Type[] argumentTypes = method.getArgumentTypes();
                for (int k = 0; k < argumentTypes.length; k++) {
                    Type type = argumentTypes[k];
                    String key = type.getSignature();
                    if (key != null && key.length() > 1) key = key.substring(1, key.length()-1).replace('/','.');
                    if (type.getType() > Constants.T_VOID && !classFieldNames.containsKey(key)) {
                        putClassFieldName(key, index++);
                    }
                }
            }
        }

        for (Iterator iter = classFieldNames.keySet().iterator(); iter.hasNext();) {
            Object key = iter.next();
            field = new FieldGen(Constants.ACC_PRIVATE | Constants.ACC_STATIC, new ObjectType("java.lang.Class"), (String) classFieldNames.get(key), _cp);
            _cg.addField(field.getField());
            if (isDebug()) debug("private static java.lang.Class " + classFieldNames.get(key) + ";");
            classFieldNames.put(key, field.getName());
        }

        if (isDebug()) debug();
        if (isDebug()) debug("private " + InvocationHandler.class.getName() + " " + fieldName + ";\n");
        field = new FieldGen(Constants.ACC_PRIVATE, new ObjectType(InvocationHandler.class.getName()), fieldName, _cp);
        _cg.addField(field.getField());
    }

    private void createInit() {
        InstructionList il = new InstructionList();
        MethodGen mg = new MethodGen(Constants.ACC_PUBLIC, Type.VOID, Type.NO_ARGS, new String[] {}, "<init>", className, il, _cp);
        if (isDebug()) debug("init (" + mg.getSignature() + ") {");
        if (isDebug()) increaseIndent();

        if (isDebug()) debug("super();");
        il.append(_factory.createLoad(Type.OBJECT, 0));
        il.append(_factory.createInvoke(Proxy.class.getName(), "<init>", Type.VOID, Type.NO_ARGS, Constants.INVOKESPECIAL));

        if (isDebug()) debug("this." + fieldName + " = null;");
        il.append(_factory.createLoad(Type.OBJECT, 0));
        il.append(InstructionConstants.ACONST_NULL);
        il.append(_factory.createFieldAccess(className, fieldName, new ObjectType(InvocationHandler.class.getName()), Constants.PUTFIELD));

        if (isDebug()) debug("return;");
        il.append(InstructionConstants.RETURN);

        mg.setMaxStack();
        mg.setMaxLocals();
        _cg.addMethod(mg.getMethod());
        il.dispose();
        if (isDebug()) decreaseIndent();
        if (isDebug()) debug("} // init(" + mg.getSignature() + ")\n");
    }
    
    private void createSetter() {
        InstructionList il = new InstructionList();
        String localVariableName = "handler";
        MethodGen mg = new MethodGen(Constants.ACC_PUBLIC, Type.VOID, new Type[] { new ObjectType(InvocationHandler.class.getName()) }, new String[] { localVariableName }, fieldSetterName, className, il, _cp);
        if (isDebug()) debug(fieldSetterName +" (" + mg.getSignature() + ") {");
        if (isDebug()) increaseIndent();

        if (isDebug()) debug("if (this." + fieldName + " != " + localVariableName + ") {");
        il.append(_factory.createLoad(Type.OBJECT, 0));
        il.append(_factory.createFieldAccess(className, fieldName, new ObjectType(InvocationHandler.class.getName()), Constants.GETFIELD));
        BranchInstruction ifnonnull_then_return = _factory.createBranchInstruction(Constants.IFNONNULL, null);

        if (isDebug()) debug("this." + fieldName + " = " + localVariableName + ";");
        il.append(_factory.createLoad(Type.OBJECT, 0));
        il.append(_factory.createLoad(Type.OBJECT, 1));
        il.append(_factory.createFieldAccess(className, fieldName, new ObjectType(InvocationHandler.class.getName()), Constants.PUTFIELD));

        if (isDebug()) debug("} //if (this." + fieldName + " != " + localVariableName + ")");

        if (isDebug()) debug("return;");
        InstructionHandle ih_return = il.append(InstructionConstants.RETURN);
        ifnonnull_then_return.setTarget(ih_return);

        mg.setMaxStack();
        mg.setMaxLocals();
        _cg.addMethod(mg.getMethod());
        il.dispose();
        if (isDebug()) decreaseIndent();
        if (isDebug()) debug("} // " + fieldSetterName + "(" + mg.getSignature() + ")\n");
    }

    private void createMethod(Method method) {
        if (isDebug()) debug("//method: " + method.getName() + method.getSignature());
        ExceptionTable et = method.getExceptionTable();
        InstructionList il = new InstructionList();
        //LocalVariableTable lvt = method.getLocalVariableTable();
        //LocalVariable[] arguments = lvt != null ? lvt.getLocalVariableTable() : new LocalVariable[] {};
        Type[] argumentTypes = method.getArgumentTypes();
        String[] argumentNames = new String[argumentTypes.length];
        String resultClassName = null;
        Type returnType = method.getReturnType();
        if (returnType.getType() < Constants.T_VOID) {
            resultClassName = getPrimitiveClassName(returnType.getType());
        } else if (returnType.getType() != Constants.T_VOID) {
            resultClassName = ((ObjectType) returnType).getClassName();
        }
        Type resultType = new ObjectType(resultClassName);
        String methodSignature = method.getName() + method.getSignature();
        JavaClass interfaceClass = (JavaClass) interfaceMap.get(methodSignature);
        String[] exceptions = (String[]) exceptionMap.get(methodSignature);

        for (int i = 0; i < argumentTypes.length; i++) {
            Type type = argumentTypes[i];
            argumentNames[i] = "arg" + (i+1);
        }
        MethodGen mg = new MethodGen(Constants.ACC_PUBLIC | Constants.ACC_FINAL, method.getReturnType(), method.getArgumentTypes(), argumentNames, method.getName(), className, il, _cp);
        if (exceptions.length == 0) {
            if (isDebug()) debug(method.getSignature() + " {");
        } else {
            if (isDebug()) debug(method.getSignature() + " throws");
            if (isDebug()) increaseIndent(4);
            for (int i = 0; i < exceptions.length; i++) {
                mg.addException(exceptions[i]);
                if (isDebug()) debug(exceptions[i]);
            }
            if (isDebug()) decreaseIndent(4);
            if (isDebug()) debug("{ // " + method.getSignature());
        }
        if (isDebug()) increaseIndent();

        String localVariableName = "result";
        int result = -1;
        if (method.getReturnType() != Type.VOID) {
            //ReturnType result = null;
            if (isDebug()) debug(resultClassName + " " + localVariableName + " = null;");
            il.append(InstructionConstants.ACONST_NULL);
            LocalVariableGen lg = mg.addLocalVariable(localVariableName, resultType, null, null);
            result = lg.getIndex();
            if (isDebug()) debug("// result index " + result);
            lg.setStart(il.append(new ASTORE(result)));
        }

        //load handler
        InstructionHandle ih_start_try = il.append(_factory.createLoad(Type.OBJECT, 0));
        il.append(_factory.createFieldAccess(className, fieldName, new ObjectType(InvocationHandler.class.getName()), Constants.GETFIELD));
        il.append(_factory.createLoad(Type.OBJECT, 0));
        //load method interface class
        loadObjectClass(mg, il, interfaceClass.getClassName());
        //add method name
        il.append(new PUSH(_cp, method.getName()));

        if (argumentTypes.length > 0) {
            //create array of argument types
            if (isDebug()) debug("Class[] argumentTypes = new Class[" + argumentTypes.length + "];");
            il.append(new PUSH(_cp, argumentTypes.length));
            il.append(_factory.createNewArray(new ObjectType("java.lang.Class"), (short) 1));
            il.append(InstructionConstants.DUP);
    
            //load each argument type and save to array
            for (int i = 0; i < argumentTypes.length; i++) {
                if (isDebug()) debug("argumentTypes[" + i + "] = loadType(" + argumentTypes[i].getSignature() + ");");
                il.append(new PUSH(_cp, i));
                loadType(mg, il, argumentTypes[i]);
                il.append(InstructionConstants.AASTORE);
            }
        } else {
            if (isDebug()) debug("Class[] argumentTypes = null;");
            il.append(InstructionConstants.ACONST_NULL);
        }
        //get the method object to be passed to delegate, using reflection API
        il.append(_factory.createInvoke("java.lang.Class", "getMethod", new ObjectType("java.lang.reflect.Method"), new Type[] { Type.STRING, new ArrayType(new ObjectType("java.lang.Class"), 1) }, Constants.INVOKEVIRTUAL));

        if (argumentNames.length > 0) {
            //create new array of argument values to be passed to delegate
            if (isDebug()) debug("Object[] arguments = new Object[" + argumentNames.length + "];");
            il.append(new PUSH(_cp, argumentNames.length));
            il.append(_factory.createNewArray(Type.OBJECT, (short) 1));
            il.append(InstructionConstants.DUP);
            //argument value array created
    
            //loop over local arguments and store them in array
            for (int i = 0; i < argumentNames.length; i++) {
                if (isDebug()) debug("arguments[" + i + "] = " + argumentNames[i] + ";");
                il.append(new PUSH(_cp, i));
                if (argumentTypes[i].getType() < Constants.T_VOID) {
                    String primitiveClassName = getPrimitiveClassName(argumentTypes[i].getType());
                    il.append(_factory.createNew(primitiveClassName));
                    il.append(InstructionConstants.DUP);
                    il.append(_factory.createLoad(argumentTypes[i], i+1));
                    il.append(_factory.createInvoke(primitiveClassName, "<init>", Type.VOID, new Type[] { argumentTypes[i] }, Constants.INVOKESPECIAL));
                } else {
                    il.append(_factory.createLoad(argumentTypes[i], i+1));
                }
                il.append(InstructionConstants.AASTORE);
            }
            //local arguments stored
        } else {
            if (isDebug()) debug("Class[] arguments = null;");
            il.append(InstructionConstants.ACONST_NULL);
        }

        //invoke the delegator method
        if (isDebug()) debug("handler.invoke();");
        il.append(_factory.createInvoke(InvocationHandler.class.getName(), "invoke", Type.OBJECT, new Type[] { Type.OBJECT, new ObjectType("java.lang.reflect.Method"), new ArrayType(Type.OBJECT, 1) }, Constants.INVOKEINTERFACE));

        if (method.getReturnType() != Type.VOID) {
            //save the result
            if (isDebug()) debug("//check cast: " + resultClassName);
            il.append(_factory.createCheckCast(new ObjectType(resultClassName)));
            if (isDebug()) debug("//store to result");
            il.append(_factory.createStore(Type.OBJECT, result));
        }

        //goto return
        BranchInstruction goto_return = _factory.createBranchInstruction(Constants.GOTO, null);
        InstructionHandle ih_end_try = il.append(goto_return);
        
        //standard catch blocks
        createReThrow(mg, il, ih_start_try, ih_end_try, "java.lang.ClassCastException");
        createNestedThrow(mg, il, ih_start_try, ih_end_try, "java.lang.NoSuchMethodException", "java.lang.Error");
        createReThrow(mg, il, ih_start_try, ih_end_try, "java.lang.RuntimeException");

        //declared exception catch blocks, will be re-thrown
        for (int i = 0; i < exceptions.length; i++) {
            createReThrow(mg, il, ih_start_try, ih_end_try, exceptions[i]);
        }

        //standard catch blocks, continued...
        createReThrow(mg, il, ih_start_try, ih_end_try, "java.lang.Error");
        createNestedThrow(mg, il, ih_start_try, ih_end_try, "java.lang.Throwable", "java.lang.reflect.UndeclaredThrowableException");

        //load the result and return it, or, just return
        InstructionHandle ih_return = null;
        if (method.getReturnType().getType() < Constants.T_VOID) {
            // primitive type
            if (isDebug()) debug("return primitive " + localVariableName +  ";");
            ih_return = il.append(_factory.createLoad(resultType, result));
            createPrimitiveConvert(il, method.getReturnType());
            il.append(_factory.createReturn(method.getReturnType()));
        } else if (method.getReturnType().getType() != Constants.T_VOID) {
            // object
            if (isDebug()) debug("return " + localVariableName +  ";");
            ih_return = il.append(_factory.createLoad(Type.OBJECT, result));
            il.append(_factory.createReturn(method.getReturnType()));
        } else {
            // void
            if (isDebug()) debug("return;");
            ih_return = il.append(InstructionConstants.RETURN);
        }
        goto_return.setTarget(ih_return);

        mg.setMaxStack();
        mg.setMaxLocals();
        _cg.addMethod(mg.getMethod());
        il.dispose();
        if (isDebug()) decreaseIndent();
        if (isDebug()) debug("} //" + method.getSignature() + "\n");
    }

    private static String getPrimitiveClassName(byte type) {
        switch (type) {
            case Constants.T_BOOLEAN: return "java.lang.Boolean";
            case Constants.T_BYTE: return "java.lang.Byte";
            case Constants.T_CHAR: return "java.lang.Character";
            case Constants.T_DOUBLE: return "java.lang.Double";
            case Constants.T_FLOAT: return "java.lang.Float";
            case Constants.T_INT: return "java.lang.Integer";
            case Constants.T_LONG: return "java.lang.Long";
            case Constants.T_SHORT: return "java.lang.Short";
        }
        return null;
    }

    private static String getPrimitiveMethodName(byte type) {
        switch (type) {
            case Constants.T_BOOLEAN: return "booleanValue";
            case Constants.T_BYTE: return "byteValue";
            case Constants.T_CHAR: return "charValue";
            case Constants.T_DOUBLE: return "doubleValue";
            case Constants.T_FLOAT: return "floatValue";
            case Constants.T_INT: return "intValue";
            case Constants.T_LONG: return "longValue";
            case Constants.T_SHORT: return "shortValue";
        }
        return null;
    }

    private void createPrimitiveConvert(InstructionList il, Type returnType) {
        il.append(_factory.createInvoke(getPrimitiveClassName(returnType.getType()), getPrimitiveMethodName(returnType.getType()), 
                                                       returnType, Type.NO_ARGS, Constants.INVOKEVIRTUAL));
    }

    private void createReThrow(MethodGen mg, InstructionList il, InstructionHandle ih_start_try, InstructionHandle ih_end_try, String handledException) {
        if (isDebug()) debug("catch (" + handledException + " e) {");
        if (isDebug()) increaseIndent();
        LocalVariableGen lg = mg.addLocalVariable("e", Type.OBJECT, null, null);
        int e = lg.getIndex();
        if (isDebug()) debug("// e index " + e);
        InstructionHandle ih = il.append(new ASTORE(e));
        lg.setStart(ih);
        il.append(_factory.createLoad(Type.OBJECT, e));
        if (isDebug()) debug("throw e;");
        lg.setEnd(il.append(InstructionConstants.ATHROW));
        mg.addExceptionHandler(ih_start_try, ih_end_try, ih, new ObjectType(handledException));
        if (isDebug()) decreaseIndent();
        if (isDebug()) debug("} //catch " + handledException);
    }

    private void createNestedThrow(MethodGen mg, InstructionList il, InstructionHandle ih_start_try, InstructionHandle ih_end_try, 
                                                      String handledException, String thrownException) {
        if (isDebug()) debug("catch (" + handledException + " e) {");
        if (isDebug()) increaseIndent();
        LocalVariableGen lg = mg.addLocalVariable("e", Type.OBJECT, null, null);
        int e = lg.getIndex();
        if (isDebug()) debug("// e index " + e);
        InstructionHandle ih = il.append(new ASTORE(e));
        lg.setStart(ih);

        il.append(_factory.createNew(thrownException));
        il.append(InstructionConstants.DUP);

        il.append(_factory.createLoad(Type.OBJECT, e));
        il.append(_factory.createInvoke(thrownException, "<init>", Type.VOID, new Type[] { new ObjectType("java.lang.Throwable") }, Constants.INVOKESPECIAL));
        if (isDebug()) debug("throw new " + thrownException + "(e);");
        lg.setEnd(il.append(InstructionConstants.ATHROW));
        mg.addExceptionHandler(ih_start_try, ih_end_try, ih, new ObjectType(handledException));
        if (isDebug()) decreaseIndent();
        if (isDebug()) debug("} //catch " + handledException);
    }

    private void loadObjectClass(MethodGen mg, InstructionList il, String classNameForLoading) {
        if (isDebug()) debug("loadObjectClass (" + classNameForLoading + ") {");
        if (isDebug()) increaseIndent();

        String classFieldName = (String) this.classFieldNames.get(classNameForLoading);
        if (isDebug()) debug("//get classFieldName " + classNameForLoading + " -> " + classFieldName);
        //il.append(_factory.createLoad(Type.OBJECT, 0));
        il.append(_factory.createFieldAccess(className, classFieldName, new ObjectType("java.lang.Class"), Constants.GETSTATIC));
        il.append(InstructionConstants.DUP);
        BranchInstruction ifnonnull_then_ok = _factory.createBranchInstruction(Constants.IFNONNULL, null);
        il.append(ifnonnull_then_ok);
        il.append(InstructionConstants.POP);

        // if class field is null, then init that field
        // try
        InstructionHandle ih_start_try = il.append(new PUSH(_cp, classNameForLoading));
        InstructionHandle ih_end_try = il.append(_factory.createInvoke("java.lang.Class", "forName", new ObjectType("java.lang.Class"), new Type[] { Type.STRING }, Constants.INVOKESTATIC));
        // catch ClassNotFoundException, goto ih_catch: throw new NoClassDefFoundError
 
        // save the Class object to static field
        il.append(InstructionConstants.DUP);
        il.append(_factory.createFieldAccess(this.className, classFieldName, new ObjectType("java.lang.Class"), Constants.PUTSTATIC));
        BranchInstruction goto_ok = _factory.createBranchInstruction(Constants.GOTO, null);
        il.append(goto_ok);

        //handle ClassNotFoundException, throw new NoClassDefFoundError
        InstructionHandle ih_catch = il.append(_factory.createNew("java.lang.NoClassDefFoundError"));
        il.append(InstructionConstants.DUP_X1);
        il.append(InstructionConstants.SWAP);
        il.append(_factory.createInvoke("java.lang.Throwable", "getMessage", Type.STRING, Type.NO_ARGS, Constants.INVOKEVIRTUAL));
        il.append(_factory.createInvoke("java.lang.NoClassDefFoundError", "<init>", Type.VOID, new Type[] { Type.STRING }, Constants.INVOKESPECIAL));
        il.append(InstructionConstants.ATHROW);
        //new NoClassDefFoundError throwed

        InstructionHandle ih_ok = il.append(InstructionConstants.NOP);
        ifnonnull_then_ok.setTarget(ih_ok);
        goto_ok.setTarget(ih_ok);

        mg.addExceptionHandler(ih_start_try, ih_end_try, ih_catch, new ObjectType("java.lang.ClassNotFoundException"));
        if (isDebug()) decreaseIndent();
        if (isDebug()) debug("} //loadObjectClass " + classNameForLoading);
    }

    private void loadType(MethodGen mg, InstructionList il, Type type) {
        if (isDebug()) debug("loadType (" + type.getSignature() + ") {");
        if (isDebug()) increaseIndent();

        if (type.getType() < Constants.T_VOID) {
            if (isDebug()) debug("// primitive: " + getPrimitiveClassName(type.getType()));
            il.append(_factory.createFieldAccess(getPrimitiveClassName(type.getType()), "TYPE", new ObjectType("java.lang.Class"), Constants.GETSTATIC));
        } else {
            ObjectType objectType = (ObjectType) type;
            if (isDebug()) debug("// object: " + objectType.getClassName() + ", calling loadObjectClass");
            loadObjectClass(mg, il, objectType.getClassName());
        }
        if (isDebug()) decreaseIndent();
        if (isDebug()) debug("} //loadType " + type.getSignature());
    }

    private void debug() {
        System.out.println();
    }

    private void debug(String msg) {
        System.out.println(this.indent + msg);
    }
    
    private void increaseIndent() {
        increaseIndent(1);
    }
    
    private void increaseIndent(int n) {
        if (n > 1) {
            StringBuffer sb = new StringBuffer(this.indent.length() + n * 4);
            sb.append(this.indent);
            for (int i = 0; i < n; i++) sb.append("    ");
            this.indent = sb.toString();
        } else {
            this.indent += "    ";
        }
    }

    private void decreaseIndent() {
        decreaseIndent(1);
    }
    
    private void decreaseIndent(int n) {
        this.indent = this.indent.substring(4*n);
    }

    private boolean isDebug() {
        return true;
    }

}
