
package org.k2d2.framework.serverframework.core;

/**
 * This abstract class should be inherited from to provide your own customized
 * command that processes tasks in its in-queue and puts them in the out-queue.
 * You need to override the execute() method to provide your application specific
 * functionality.
 * @author Karthik Rangaraju
 */
public abstract class ServerCommand implements Cloneable
{
	private int mNumThreads = 1;
	
	public static class ThreadingModelType
	{
		private ThreadingModelType()
		{
		}
	}
	public static final ThreadingModelType SINGLE_THREADED = new ThreadingModelType();
	public static final ThreadingModelType MULTI_THREADED = new ThreadingModelType();
	
	private ThreadingModelType mModel = MULTI_THREADED;

	/**
	 * Lets you set the command's threading model. If your execute method is
	 * thread safe, set this to ServerCommand.MULTI_THREADED, otherwise set it
	 * to ServerCommand.SINGLE_THREADED. If you don't call this method, the
	 * default threading model (ServerCommand.MULTI_THREADED) is used.
	 */
	public final void setThreadingModelType(ThreadingModelType pModel)
	{
		mModel = pModel;
	}

	/**
	 * Utility method to get the threading model type
	 * @return SeverCommand.ThreadingModelType - can be ServerCommand.SINGLE_THREADED
	 * or ServerCommand.MULTI_THREADED
	 */
	public final ServerCommand.ThreadingModelType getThreadingModelType()
	{
		return mModel;
	}

	/**
	 * Sets the number of threads you want to use to execute this command. The
	 * default is 1
	 */
	public final void setNumberOfThreads(int pNumThreads)
	{
		mNumThreads = pNumThreads;
	}
		
	/**
	 * Returns the number of threads that will be used to execute this command
	 * @return int representing number of threads that will run this command
	 */
	public final int getNumberOfThreads()
	{
		return mNumThreads;
	}
	
	/**
	 * For single threaded commands (whose execute() method is not thread safe,
	 * the server creates clones of the command object. Use this method to
	 * perform any specific cloning work that the default Java implementation
	 * cannot cover (anything beyond a shallow copy of the object)
	 * @return Object - a reference to a cloned ServerCommand object
	 */
	public Object clone()
	{
		try
		{
			return super.clone();
		}
		catch (CloneNotSupportedException e)
		{
			e.printStackTrace();
			// What do I do with this exception?
			// Exception should not occur
			// k2: I think I should log the error (whenever I have
			// logging implemented) and exit!
			return null;
		}
	}

	/**
	 * This method lets the command perform one-time initialization. This method is
	 * called once per server start() call.
	 */
	public void init()
	{
		
	}
	
	/**
	 * Your class that derives from ServerCommand should override this method.
	 * The method is executed for every object in the server's in-queue that
	 * needs to be processed.
	 * @param pInObj a reference to an object from the server's in-queue that
	 * needs to be processed.
	 * @return Object - if you return a non-null object, the returned object is
	 * added to the server's out-queue. If you return null, nothing is added
	 * to the server's out-queue
	 */
	public Object execute(Object pInObj)
	{
		return null;
	}
	
}
