
package org.k2d2.framework.threadpackage.semaphore;

/**
 * This class implements a POSIX style "Event" object. The difference
 * between the ConditionalEvent and the java wait()/notify() technique is in
 * handling of event state. If a ConditionalEvent is signalled, a thread
 * that subsequently waits on it is immediately released. In case of auto
 * reset EventObjects, the object resets (unsignalled) itself as soon as it
 * is signalled and waiting thread(s) are released (based on whether signal()
 * or signalAll() was called).
 * <B>When to use</b>
 * When porting code from a POSIX environment or when writing complex multi-threaded
 * applications where you need to work with higher level abstractions
 *
 * @author Karthik Rangaraju
 */
public class ConditionalEvent
{
    private boolean mState = false;
    private boolean mAutoReset = false;
    
    // TODO: Need to add methods that block until a specified time and
    // return (though in real-life, I've never known what to do if a thread
    // timesout other than call the method again!
    
    /**
     * Creates a manual reset ConditionalEvent with a specified initial state
     * @param pInitialState Sets the initial state of the ConditionalEvent.
     * Signalled if pInitialState is true, unsignalled otherwise.
     */
    public ConditionalEvent(boolean pInitialState)
    {
        mState = pInitialState;
    }
    
    /**
     * Creates a ConditionalEvent wit the defined initial state
     * @param pInitialState if true, the ConditionalEvent is signalled when
     * created.
     * @param pAutoReset if true creates an auto-reset ConditionalEvent
     */
    public ConditionalEvent(boolean pInitialState, boolean pAutoReset)
    {
        mState = pInitialState;
        mAutoReset = pAutoReset;
    }
    
    /**
     * Checks if the event is signalled. Does not block on the operation
     * @return true is event is signalled, false otherwise. Does not reset
     * an autoreset event
     */
    public boolean isSignalled()
    {
        return mState;
    }

    /**
     * Signals the event. A single thread blocked on waitForSignal() is released
     * @see #signalAll()
     * @see #waitForSignal()
     */
    public synchronized void signal()
    {
        mState = true;
        notify();
    }

    /**
     * Current implementation only works with manual reset events. Releases
     * all threads blocked on waitForSignal()
     * @see #waitForSignal()
     */
    public synchronized void signalAll()
    {
        mState = true;
        notifyAll();
    }

    /**
     * Resets the event to an unsignalled state
     */
    public synchronized void reset()
    {
        mState = false;
    }
    
    /**
     * If the event is signalled, this method returns immediately resetting the
     * signal, otherwise it blocks until the event is signalled.
     * @throws InterruptedException if the thread is interrupted when blocked
     */
    public synchronized void waitForSignal() throws InterruptedException
    {
        while (mState == false)
        {
            wait();
        }
        if (mAutoReset == true)
        {
            mState = false;
        }
    }

}
