
package org.k2d2.framework.threadpackage.semaphore;

/**
 * A thread barrier blocks all threads hitting it until a pre-defined number of threads
 * arrive at the barrier. This is useful for implementing release consistent concurrency
 * where you don't want to take the performance penalty of providing mutual exclusion
 * to shared resources
 * <b>When to use</b>
 * To contain the complexity of multi-threaded application code, use this class instead
 * of implementing the functionality using Java primitves wait() and notify() with
 * counters.
 *
 * @author Karthik Rangaraju
 */
public class ThreadBarrier
{
    private int mThreshold;
    private int mCount;

    /**
     * Initializes a thread barrier object with a given thread count
     * @param pCount is the number of threads that need to block on barrierSynchronize()
     * before they will be allowed to pass through
     * @see #barrierSynchronize()
     */
    public ThreadBarrier(int pCount)
    {
        mThreshold = pCount;
        mCount = 0;
    }

    /**
     * This method blocks all threads calling it until the threshold number of threads
     * have called it. It then releases all threads blocked by it
     * @throws InterruptedException if any thread blocked during the call is interrupted
     */
    public synchronized void barrierSynchronize() throws InterruptedException
    {
        if (mCount != mThreshold - 1)
        {
            mCount++;
            wait();
        }
        else
        {
            mCount = 0;
            notifyAll();
        }
    }

}
