
package org.k2d2.framework.threadpackage.util;

/**
 * Interface for a BlockingQueue. Unlike a normal queue, in a BlockingQueue,
 * threads trying to dequeue() from an empty queue are blocked until the operation
 * can be successful.
 *
 * @author Karthik Rangaraju
 */
public interface BlockingQueue
{
    /**
     * This method adds an object to the queue.
     * @param pObj the object to add to the queue
     * @throws InterruptedException if a thread blocked on this method is
     * interrupted
     * @throws QueueClosedException if the queue is closed when this operation
     * is invoked.
     */
    public void enqueue(Object pObj) throws InterruptedException, QueueClosedException;
    
    /**
     * This method dequeues an object from the queue. If the queue is empty
     * it blocks until something is added to the queue. The blocked thread can be
     * released by calling close() which causes QueueClosedException to be thrown.
     * @return An object from the queue of type java.lang.Object
     * @throws InterruptedException if a thread blocked on this call is
     * interrupted.
     * @throws QueueClosedException if the thread exited the method because
     * someone called close()
     * @see #close()
     */
    public Object dequeue() throws InterruptedException, QueueClosedException;
    
    /**
     * This method forces threads blocked on a dequeue() call to exit throwing
     * the QueueClosedException exception. Once this method has been called,
     * to reuse the queue, one must call the reset() method. Note: This method
     * does not wait for the blocked threads to exit
     * @see #reset()
     */
    public void close();
    
    /**
     * This method returns the state of the queue. Note: In a multi-threaded
     * environment, the return value of this method does not necessarily
     * reflect the latest state of the queue
     * @return true if the queue is closed, false otherwise
     */
    public boolean isClose();
    
    /**
     * This method removes all elements from the queue and resets the queue for use
     * after a call to close().
     * @see #close()
     */
    public void reset();
    
    /**
     * This method returns the size of the queue. Since this queue is designed
     * to be accessed concurrently, you cannot rely on this return value unless
     * you guard access to the queue. This method is only provided as a
     * convenience. Most clients of a blocking queue should never need to check
     * the size of the queue
     * @return an integer representing the size of the queue
     */
    public int size();
    
}

