
package org.k2d2.framework.threadpackage.util;

/**
 * This is an implementation of a bounded blocking queue. Threads that try to
 * enqueue to a full queue or dequeue from an empty queue are blocked until the
 * operation can be successful. The implementation also provides for a way to
 * force the exit of threads blocked on an empty queue thus providing graceful
 * shutdown.
 *
 * @author Karthik Rangaraju
 */
public class BoundedBlockingQueue extends BoundlessBlockingQueue
{
    private int mQueueSize;
        
    /**
     * Creates a blocking queue of a given size.
     * @param pQueueSize The number of elements the queue can hold (its size)
     */
    public BoundedBlockingQueue(int pQueueSize)
    {
        super(pQueueSize);
        
        mQueueSize = pQueueSize;
    }
    
    /**
     * This method adds an object to the queue. If the queue is full, it
     * blocks until there is place in the queue. If the queue is closed,
     * it throws QueueClosedException.
     * @param pObj the object to add to the queue
     * @throws InterruptedException if a thread blocked on this method is
     * interrupted
     * @throws QueueClosedException if the queue is closed when the method
     * is called
     */
    public void enqueue(Object pObj) throws InterruptedException,
                                            QueueClosedException
    {
        synchronized (this)
        {
            if (isClosed() == true)
            {
                throw new QueueClosedException();
            }
            while (super.size() == mQueueSize)
            {
                wait();
            }
            super.enqueue(pObj);
        }
    }

    /**
     * This method dequeues an object from the queue. If the queue is empty
     * it blocks until something is added to the queue. The block can be
     * released when close() is called
     * @return An object from the queue of type java.lang.Object
     * @throws InterruptedException if a thread blocked on this call is
     * interrupted.
     * @throws QueueClosedException if the thread exited the method because
     * someone called close()
     * @see #close()
     */
    public synchronized Object dequeue() throws InterruptedException,
                                                QueueClosedException
    {
        Object obj;
        
        synchronized (this)
        {
            obj = super.dequeue();
            notify();
            return obj;
        }
    }
    
    /**
     * This method checks if the queue is full or not
     * @return true if full, false otherwise
     */
    public boolean isFull()
    {
        return (super.size() == mQueueSize);
    }
    
}
