#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include "config.h"
#include "debug.h"
#include "servo_input.h"
#include "example.h"

#define NEUTRAL_TICKS	(1500*(F_CPU/1000000))
#define MAXSPEED_TICKS	(500*(F_CPU/1000000))

/**************
 * Basic theory
 **************
 * We want to determine the length of a pulse
 * We start by triggering on a positive edge and grabbing a timestamp
 * We then trigger on a negative edge and calculate the difference between
 * the current and stored timestamps.  The resulting time is the length
 * of the pulse in clock ticks.  This value is then scaled to a speed.
 */

void servo_init() {
	/* Set the watchdog timer
	 * The constants really don't reflect real-world time.  This
	 * is actually 512K clock ticks.  For 1 Mhz, that would be 0.52 seconds.
	 */
	wdt_enable(WDTO_500MS);

	/**************************
	 * Turn on the capture unit
	 **************************
	 * Start with the positive edge being detected
	 * and set the timer clock equal to the system clock.
	 * Also turn on noise cancelling for the input capture
	 * pin.
	 */
	TCCR1B = _BV(ICES1) | _BV(CS10) | _BV(ICNC1);

	/* Enable input capture interrupt */
	TIMSK = _BV(TICIE1);
}

/* Timer 1 input capture interrupt
 * This happens on each edge of a server pulse
 */
ISR(TIMER1_CAPT_vect)
{
	static uint16_t		positiveEdge;
	uint16_t		timestamp;

	/* Reset the watchdog timer every time we get a pulse */
	wdt_reset();

	/* Grab the current value */
	timestamp = ICR1;

	/* If ICES1 is set, we tripped on the positive edge */
	if (bit_is_set(TCCR1B,ICES1)) {
		/* Store the current value */
		positiveEdge = timestamp;

		/* Switch to trigger on the negative edge */
		TCCR1B &= ~(_BV(ICES1));
	} else {
		uint16_t pulseLength;

		/* Switch to trigger on the positive edge */
		TCCR1B |= _BV(ICES1);

		/* Calcuate the difference between the two edges */
		if (timestamp < positiveEdge) {
			pulseLength = (0xFFFF - positiveEdge + 1) + timestamp;
		} else {
			pulseLength = timestamp - positiveEdge;
		}

		/* We need to know 2 things: direction and speed.
		 * Since the length is centered around 1.5ms, we
		 * can figure out the direction by simply comparing
		 * against the number of clock cycles that 1.5ms
		 * would be.  Then, we can determine the magnitude
		 * with a little bit of math.
		 */
		if (pulseLength < NEUTRAL_TICKS) {
			/* Forward */
			motor_direction = FORWARD;
			pulseLength = (NEUTRAL_TICKS - pulseLength);
		} else {
			/* Reverse */
			motor_direction = REVERSE;
			pulseLength = pulseLength - NEUTRAL_TICKS;
		}

		/* Clamp the speed to the range from 0 to MAXSPEED */
		if (pulseLength > MAXSPEED_TICKS) {
			pulseLength = MAXSPEED_TICKS;
		}

		/* Scale the speed to an 8-bit value
		 * where 0 = Stop and 255 = Full speed
		 */
		motor_speed = (pulseLength / (MAXSPEED_TICKS/255)) & 0xFF;
	}	
}
