/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *    Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Axis" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.axis.transport.http;

import org.apache.axis.AxisEngine;
import org.apache.axis.AxisFault;
import org.apache.axis.MessageContext;
import org.apache.axis.client.Call;
import org.apache.axis.client.Transport;

//
// Internals of those packages are subject to change.
//
import org.apache.axis.transport.http.HTTPConstants;
import org.apache.axis.utils.JavaUtils;


/**
 * Extends Transport by implementing the setupMessageContext function to
 * set HTTP-specific message context fields and transport chains.
 * May not even be necessary if we arrange things differently somehow.
 * Can hold state relating to URL properties.
 *
 * @author Rob Jellinghaus (robj@unrealities.com)
 * @author Doug Davis (dug@us.ibm.com)
 * @author Glen Daniels (gdaniels@allaire.com)
 * @author Slawomir Lisznianski (slisznianski@asyncnet.com)
 */
public class HTTPTransport extends Transport
{
    public static final String DEFAULT_TRANSPORT_NAME = "http";
    
    /**
     * HTTP properties
     */
    public static final String URL = MessageContext.TRANS_URL;

    private String action;
    private AxisEngine engine_;
    
    public HTTPTransport () {
        transportName = DEFAULT_TRANSPORT_NAME;
    }
    
    /**
     * helper constructor
     */
    public HTTPTransport (String url, String action)
    {
        transportName = DEFAULT_TRANSPORT_NAME;
        this.url = url;
        this.action = action;
    }
    
    /**
     * Set up any transport-specific derived properties in the message context.
     * @param context the context to set up
     * @param message the client service instance
     * @param engine the engine containing the registries
     * @throws AxisFault if service cannot be found
     */
    public void setupMessageContextImpl(MessageContext mc,
                                        Call call,
                                        AxisEngine engine)
        throws AxisFault
    {
			//
			// Set current engine.
			//
			if (engine == null)
				throw new AxisFault("HTTPTransport.setupMessageContextImpl",
                            JavaUtils.getMessage("noEngine00"), null, null);
			engine_ = engine;
			
			//
			// Set any SOAP properties.
			//
			if (action != null) {
					mc.setUseSOAPAction(true);
					mc.setSOAPActionURI(action);
			}

			//
			// Set up any cookies from the "engine session".
			//
			Object cookie__ = engine_.getApplicationSession().get(HTTPConstants.HEADER_COOKIE);
			if (cookie__ != null)
				mc.setProperty(HTTPConstants.HEADER_COOKIE, cookie__);

			Object cookie2__ = engine_.getApplicationSession().get(HTTPConstants.HEADER_COOKIE2);
			if (cookie2__ != null)
				mc.setProperty(HTTPConstants.HEADER_COOKIE2, cookie2__);        

			// Allow the SOAPAction to determine the service, if the service
			// (a) has not already been determined, and (b) if a service matching
			// the soap action has been deployed.
			if (mc.getService() == null) {
					mc.setTargetService( (String)mc.getSOAPActionURI() );
			}
    }

    public void processReturnedMessageContext(MessageContext context) 
    {
			//
			// Store any cookies from the response into the "engine session".
			// 
			Object cookie__ = context.getProperty(HTTPConstants.HEADER_COOKIE);
			Object cookie2__ = context.getProperty(HTTPConstants.HEADER_COOKIE2);

						
			// 
			// Throw only if there would be something to store.
			//
			// Note: It should really throw AxisFault, however we are constrained
			// by the base method here.
			//
			if ((cookie__ != null || cookie2__ != null) && engine_ == null)
				throw new RuntimeException(new AxisFault("HTTPTransport.processReturnedMessageContext",
                       													 JavaUtils.getMessage("noEngine00"), null, null));			
			if (cookie__ != null)
				engine_.getApplicationSession().set(HTTPConstants.HEADER_COOKIE, cookie__);

			if (cookie2__ != null)
				engine_.getApplicationSession().set(HTTPConstants.HEADER_COOKIE2, cookie2__);
    }
}
