package Apache2::TomKit::Cache::SQLite;

use strict;
use File::Spec;
use DBI;

use Apache2::TomKit::Cache::DefaultKeyBuilder;
use Apache2::TomKit::Cache::Entry;

sub new {
	my $class      = shift;
	my $logger     = shift;
	my $config     = shift;
	my $keyBuilder = shift;

	my $key      = $keyBuilder->buildKey();
	my $cacheDir = $config->getCacheDir();

	if ( !defined $cacheDir ) {
		$cacheDir = File::Spec->tmpdir();
	}

	my $dbfile = "$cacheDir/axkitCache.sqlite";

	my $this = bless {
		logger => $logger,
		config => $config,
		key    => $key,
		dbfile => $dbfile
	}, $class;

	$this->{logger}->debug( 9, "Setting up the cache with SQLLite" );

	my $createDb = !-e $dbfile;
	if ($createDb) {
		$this->setUpDb();
	}

	return $this;
}

sub deliverFromCache {
	my $this           = shift;
	my $mtime          = shift;
	my $processorChain = shift;

	## TODO investigate if two connections can live next to each other
	my $dbh = DBI->connect( "dbi:SQLite:dbname=" . $this->{dbfile}, "", "" );
	my $keyBuilder = new Apache2::TomKit::Cache::DefaultKeyBuilder( $this->{logger}, $this->{config} );

	my $key = $keyBuilder->buildKey();

	my $sth = $dbh->prepare("SELECT d_id, d_request_md5, d_transformed_doc, d_transform_timestamp FROM delivered_documents WHERE d_request_md5 = ?");
	$sth->execute($key);

	my $rv = new Apache2::TomKit::Cache::Entry( $this->{logger}, $key, $this, $processorChain );
	my $useCached = 1;

	if ( my @deliveredDocument = $sth->fetchrow_array() ) {
		$rv->{isNew} = 0;

		$this->{logger}->debug( 9,"Found cached entry, comparing mtimes: $mtime < $deliveredDocument[3]");

		if ( $mtime < $deliveredDocument[3] ) {
			$this->{logger}->debug( 9,"Checking processors for " . $deliveredDocument[0] . ".");
			
			my $sth2 = $dbh->prepare("SELECT dp_timestamp FROM document_processors WHERE dp_ref_document = ? ");
			$sth2->execute( $deliveredDocument[0] );

			my $i = 0;

			while ( my $time = ( $sth2->fetchrow_array() )[0] ) {
				if ( $time > $processorChain->{chain}->[$i]->getMTime() )
				{
					$useCached = 1;
				}
				else {
					$useCached = 0;
				}

				$this->{logger}->debug( 9, "Comparing: $time > " . $processorChain->[$i]->getMTime() . ":" . $useCached );
				last if !$useCached;
				
				## TODO ADD CHECKS FOR DOCUMENT-FILES
			}
			
			$sth2->finish();
			
			if ($useCached) {
				$this->{logger}->debug( 7, "We can use the cached file" );
				$rv->{isCached} = 1;
				$rv->{content}  = $deliveredDocument[2];
			}
		}
	}

	$sth->finish();
	$dbh->disconnect();

	return $rv;
}

sub restore {
	my $this       = shift;
	my $cacheEntry = shift;

	$this->{logger}->debug( 9, "Restoring the content in the cache for later retrevial" );

	## TODO investigate if two connections can live next to each other
	my $dbh = DBI->connect( "dbi:SQLite:dbname=" . $this->{dbfile}, "", "" );
	my $sth;

	my $time = time;

	if ( ! $cacheEntry->{isNew} ) {
		$this->{logger}->debug( 9, "Erasing old content");
		$dbh->do( "DELETE FROM delivered_documents WHERE d_request_md5  = '" . $cacheEntry->{key} . "'" );	
		$dbh->do( "DELETE FROM document_processors WHERE dp_request_md5 = '" . $cacheEntry->{key} . "'" );
	}

	$sth = $dbh->prepare("INSERT INTO delivered_documents ( d_request_md5, d_transformed_doc, d_transform_timestamp) VALUES (?,?,?)");
	$sth->execute( $cacheEntry->{key}, $cacheEntry->{content}, $time );
	$sth->finish();


	my $id = $dbh->func('last_insert_rowid');
	
	$sth = $dbh->prepare("INSERT INTO document_processors ( dp_ref_document, dp_request_md5, dp_key, dp_timestamp ) VALUES (?,?,?,?)");

	foreach ( @{ $cacheEntry->{processorChain}->{chain} } ) {
		$this->{logger}->debug( 9, "Restoring processor for document-id ($id): " . $_->getKey()  );
		$sth->execute( $id, $cacheEntry->{key}, $_->getKey(), $time );
	}

	$sth->finish();

	## TODO ADD DOCUMENT-FILES

	$dbh->disconnect();
}


sub setUpDb {
	my $this = shift;

	$this->{logger}->debug( 9, "Oh there is no cache created until now." );

	## TODO investigate if two connections can live next to each other
	my $dbh = DBI->connect( "dbi:SQLite:dbname=" . $this->{dbfile}, "", "" );

	$dbh->do("CREATE TABLE delivered_documents ( d_id INTEGER AUTO_INCREMENT PRIMARY KEY DEFAULT '1', d_request_md5 CHAR(32), d_transformed_doc BINARY, d_transform_timestamp BIGINTEGER )");
	$dbh->do("CREATE TABLE document_processors ( dp_id INTEGER AUTO_INCREMENT PRIMARY KEY DEFAULT '1', dp_key CHAR(32), dp_request_md5 CHAR(32), dp_ref_document INTEGER, dp_timestamp BIGINTEGER )");
	$dbh->do("CREATE TABLE document_files ( df_id INTEGER AUTO_INCREMENT PRIMARY KEY DEFAULT '1', dp_request_md5 CHAR(32), df_ref_document INTEGER, df_ref_processor INTEGER, df_filename VARCHAR(255), df_timestamp BIGINTEGER )");

	$dbh->disconnect();
}

1;