///
/// \file	r_sms.cc
///		Record parsing class for the SMS database.
///

/*
    Copyright (C) 2005-2009, Net Direct Inc. (http://www.netdirect.ca/)
    Copyright (C) 2009, Ryan Li(ryan@ryanium.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

    See the GNU General Public License in the COPYING file at the
    root directory of this project for more details.
*/

#include "r_sms.h"
#include "record-internal.h"
#include "protostructs.h"
#include "data.h"
#include "time.h"
#include "debug.h"
#include "iconv.h"
#include "errno.h"
#include <ostream>
#include <iomanip>
#include <string.h>

using namespace std;
using namespace Barry::Protocol;

namespace Barry {

///////////////////////////////////////////////////////////////////////////////
// Sms Class

// SMS Field Codes
#define SMSFC_METADATA 0x01
#define SMSFC_PHONE_NUMBER 0x02
#define SMSFC_CONTENT 0x04

// SMS Field Sizes and Header Sizes
#define SMS_METADATA_SIZE 0x40
#define SMS_PHONE_NUMBER_HEADER_SIZE 0x04

// SMS Field Indices
#define SMS_FLAGS 0x01
#define SMS_NEW 0x02
#define SMS_STATUS 0x05
#define SMS_ERRORID 0x09
#define SMS_TIMESTAMP 0x0d
#define SMS_SENT_TIMESTAMP 0x15
#define SMS_ENCODING 0x1d

// SMS Flags and Field Values
#define SMS_FLG_NEW_CONVERSATION 0x20
#define SMS_FLG_SAVED 0x10
#define SMS_FLG_DELETED 0x08
#define SMS_FLG_OPENED 0x01
#define SMS_STA_RECEIVED 0x000007ff
#define SMS_STA_DRAFT 0x7fffffff
#define SMS_ENC_7BIT 0x00
#define SMS_ENC_VCARD 0x01
#define SMS_ENC_UCS2 0x02

std::string Sms::UCS2Preprocess(const std::string &ucs2) const
{
	std::string ret(ucs2);
	for (unsigned i = 0; i < ret.size(); ++i)
		if (i % 2)
		{
			// swap
			char temp = ret[i - 1];
			ret[i - 1] = ret[i];
			ret[i] = temp;
		}
	return ret;
}

time_t Sms::GetTime() const
{
	return (time_t)(Timestamp / 1000);
}

time_t Sms::GetSentTime() const
{
	return (time_t)(SentTimestamp / 1000);
}

Sms::Sms()
{
	Clear();
}

Sms::~Sms()
{
}

const unsigned char* Sms::ParseField(const unsigned char *begin,
				     const unsigned char *end,
				     const IConverter *ic)
{
	const CommonField *field = (const CommonField *)begin;

	// advance and check size
	begin += COMMON_FIELD_HEADER_SIZE + btohs(field->size);
	if (begin > end) // if begin==end, we are ok
		return begin;

	if (!btohs(field->size)) // if field has no size, something's up
		return begin;

	switch (field->type)
	{
		case SMSFC_METADATA:
		{
			if (btohs(field->size) != SMS_METADATA_SIZE)
				break; // size not match

			const unsigned char *metadata = (const unsigned char *)field->u.raw;

			NewConversation = metadata[SMS_FLAGS] & SMS_FLG_NEW_CONVERSATION;
			Saved = metadata[SMS_FLAGS] & SMS_FLG_SAVED;
			Deleted = metadata[SMS_FLAGS] & SMS_FLG_DELETED;
			Opened = metadata[SMS_FLAGS] & SMS_FLG_OPENED;

			IsNew = metadata[SMS_NEW];

			uint32_t status = *((uint32_t *) (metadata + SMS_STATUS));
			switch (status)
			{
				case SMS_STA_RECEIVED:
					MessageStatus = Received;
					break;
				case SMS_STA_DRAFT:
					MessageStatus = Draft;
					break;
				default:
					MessageStatus = Sent; // consider all others as sent
			}

			ErrorId = *((uint32_t *) (metadata + SMS_ERRORID));

			Timestamp = *((uint64_t *) (metadata + SMS_TIMESTAMP));

			SentTimestamp = *((uint64_t *) (metadata + SMS_SENT_TIMESTAMP));

			switch (metadata[SMS_ENCODING])
			{
				case SMS_ENC_7BIT:
					Encoding = SevenBit;
					break;
				case SMS_ENC_VCARD:
					Encoding = VCard;
					break;
				case SMS_ENC_UCS2:
					Encoding = UCS2;
					break;
				default:
					Encoding = SevenBit; // consider all unknowns as 7bit
			}

			return begin;
		}

		case SMSFC_PHONE_NUMBER:
		{
			uint16_t length = btohs(field->size);
			if (length < SMS_PHONE_NUMBER_HEADER_SIZE + 1) // trailing '\0'
				break; // too short
			
			length -= SMS_PHONE_NUMBER_HEADER_SIZE;
			const char *number = (const char *)field->u.raw + SMS_PHONE_NUMBER_HEADER_SIZE;
			PhoneNumbers.push_back(std::string(number, length));
			return begin;
		}

		case SMSFC_CONTENT:
		{
			const char *content_begin = (const char *)field->u.raw;
			Content = std::string(content_begin, content_begin + btohs(field->size));
			if (Encoding == UCS2)
			{
				iconv_t ucs2utf8;
				ucs2utf8 = iconv_open("UTF-8", "UCS-2");
				if (ucs2utf8 == (iconv_t)(-1))
					throw ErrnoError("iconv_open failed (ucs2 to utf8)", errno);
				IConverter iconv_ucs("UTF-8", true);

				Content = UCS2Preprocess(Content);
				Content = iconv_ucs.Convert(ucs2utf8, Content);
			}
			return begin;
		}
	}

	// if still not handled, add to the Unknowns list
	UnknownField uf;
	uf.type = field->type;
	uf.data.assign((const char*)field->u.raw, btohs(field->size));
	Unknowns.push_back(uf);

	// return new pointer for next field
	return begin;
}

void Sms::ParseHeader(const Data &data, size_t &offset)
{
	// no header in SMS records
}

void Sms::ParseFields(const Data &data, size_t &offset, const IConverter *ic)
{
	const unsigned char *finish = ParseCommonFields(*this,
		data.GetData() + offset, data.GetData() + data.GetSize(), ic);
	offset += finish - (data.GetData() + offset);
}

void Sms::Clear()
{
	MessageStatus = Unknown;
	DeliveryStatus = NoReport;
	Encoding = SevenBit;

	IsNew = NewConversation = Saved = Deleted = Opened = false;

	Timestamp = SentTimestamp = 0;
	ErrorId = 0;

	PhoneNumbers.clear();
	Content.clear();

	Unknowns.clear();
}

void Sms::Dump(std::ostream &os) const
{

	os << "SMS record: 0x" << setbase(16) << RecordId
		<< " (" << (unsigned int)RecType << ")\n";
	time_t t = GetTime();
	os << "\tTime: " << ctime(&t);

	if (MessageStatus == Received)
	{
		t = GetSentTime();
		os << "\tSent at: " << ctime(&t);
		// FIXME - since the ISP may use a time zone other than UTC, this time probably has an offset.
	}

	if (ErrorId)
		os << "\tSend Error: 0x" << setbase(16) << ErrorId << "\n";

	switch (MessageStatus)
	{
		case Received:
			os << "\tReceived From:\n";
			break;
		case Sent:
			os << "\tSent to:\n";
			break;
		case Draft:
			os << "\tDraft for:\n";
			break;
		case Unknown:
			os << "\tUnknown status for:\n";
			break;
	}

	os << "\t";
	for (std::vector<std::string>::const_iterator Iterator = PhoneNumbers.begin(); Iterator < PhoneNumbers.end(); ++Iterator)
	{
		if (Iterator != PhoneNumbers.begin())
			os << ", ";
		os << *Iterator;
	}
	os << "\n";

	if (IsNew || Opened || Saved || Deleted || NewConversation)
	{
		os << "\t";
		if (IsNew)
			os << "New ";
		if (Opened)
			os << "Opened ";
		if (Saved)
			os << "Saved ";
		if (Deleted)
			os << "Deleted ";
		os << "Message" << (NewConversation ? " that starts a new conversation" : "") << "\n";
	}
	os << "\tContent: " << Content << "\n";
	os << "\n";
}

} // namespace Barry
