/* killpg.c: bash loadable builtin wrapper of killpg(2) syscall
 */
#include <config.h>

#if defined (HAVE_UNISTD_H)
#  include <unistd.h>
#endif
#include <errno.h>
#include <string.h>

#include "bashtypes.h"
#include "posixtime.h"

#include <stdio.h>

#include "builtins.h"
#include "shell.h"
#include "common.h"

#include "bashgetopt.h"

#if !defined (NSIG)
#  define NSIG 64
#endif

#ifndef OK
# define OK 0
#endif

extern const char *signal_names[NSIG+2];

#undef FIRSTSIG
#define FIRSTSIG 1
#undef LASTSIG
#define LASTSIG  NSIG

int
killpg_builtin
( WORD_LIST *list )
{
  register WORD_LIST *word=list;
  intmax_t pgid_arg=0, sig_arg=0;
  int   pgid=0
    ,   sig=0
    ;
  if( word != NULL )
  {
    if( (word->word != NULL)
      &&(word->word->word != NULL)
      )
    {
      if( ! legal_number( word->word->word, &pgid_arg ) )
      {
        builtin_error("First 'pgid' argument must be numeric.");         
        return (EXECUTION_FAILURE);
      }
      if( pgid_arg < 0 )
      {
        builtin_error("negative 'pgid' values not allowed.");         
        return (EXECUTION_FAILURE);
      }
      word=word->next;
    }
    
    if( (word != NULL)
      &&(word->word != NULL)
      &&(word->word->word != NULL)
      )
    {
      if( ! legal_number( word->word->word, &sig_arg ) )
      {
        register const char *p=word->word->word;
        register int sig=-1;          
        if(strncasecmp("sig",p,3) == 0)
        {
          for(sig=FIRSTSIG; sig < LASTSIG; sig+=1)
            if(strcasecmp(signal_names[sig]+3,p+3) == 0)
              break;
        }
        else
        {
          for(sig=FIRSTSIG; sig < LASTSIG; sig+=1)
            if(strcasecmp(signal_names[sig]+3,p) == 0)
              break;
        }       
      }else
      {
        sig = sig_arg;
        if( (sig < 0 ) || (sig >= NSIG) )
        {
          builtin_error("signal argument >0 <64 expected.");
          return (EXECUTION_FAILURE);
        }
      }
    }
  }
  if( pgid_arg > 0 )
    pgid = pgid_arg;
  if( sig <= 0 )
    sig = SIGTERM;
  if(killpg(pgid,sig)!=OK)
  {
    builtin_error("killpg(%d,%d) failed: %d %s",pgid,sig, errno, strerror(errno));
    return (EXECUTION_FAILURE);
  }
  return (EXECUTION_SUCCESS);
}
           
const char *killpg_doc[]=
{
  "killpg: wrapper of killpg(2) system call"
, ""
, "Arguments:"
, "[ <pgid:integer\>\=0 process group> <sig:( integer\>\=0 | [SIG|sig]{hup|HUP,int|INT...} )> ]"
, "If pgid is 0, kills the current process group; if sig is 0, SIGTERM is used."
, 0
};

struct builtin killpg_struct =
{ "killpg"
, killpg_builtin
, BUILTIN_ENABLED
, (char**)killpg_doc
, "killpg <pgid> <sigspec>"
, 0
};
