/**
 * @brief Description of the SLIP input grammar.
 * Syntax and tokens for SlipReadList.lex.cpp
 * Sublist Declaration
 * sublist sublistName;                        // sublist of forward declarations
 * sublist sublistName, sublistName ...;       // semi-colon required
 *                                             // list Definition with user mark and Description list
 * list listName{ mark <Descriptor list> ( ) };
 * list listName{ <Descriptor list> ( ) };     // list Definition with Description list
 * list listName{ mark ( ) };                  // list Definition with user mark
 * list listName { ( ) };                      // list Definition empty list
 * list listName { };                          // list Definition empty list
 *
 * User Data Declaration
 * user userName;                              // Single user data name
 * user userName, ...;                         // sublist of user data names
 *
 * #include "path/name"                        // Include file
 *
 * list Syntax                                 // list Stuff
 *
 * 'c'                                         // char           SlipDatum((char)'c')
 * c                                           // char           SlipDatum((char)'c')
 * "string"                                    // string         SlipDatum((string)string, false)
 * true                                        // bool           SlipDatum((bool)true)
 * false                                       // bool           SlipDatum((bool)true)
 * numberUC                                    // unsigned char  SlipDatum((UCHAR)number)
 * numberC                                     // signed char    SlipDatum((CHAR)number)
 * numberUL                                    // unsigned long  SlipDatum((ULONG)number)
 * numberL                                     // signed long    SlipDatum((LONG)number)
 * number                                      // signed long    SlipDatum((LONG)number)
 * double        1.0E+0                        // double         SlipDatum(DOUBLE)number)
 * double        1.E+0                         // double         SlipDatum(DOUBLE)number)
 * double        1.                            // double         SlipDatum(DOUBLE)number)
 * double         .1E+0                        // double         SlipDatum(DOUBLE)number)
 * double         .1                           // double         SlipDatum(DOUBLE)number)
 * double        1E+0                          // double         SlipDatum(DOUBLE)number)
 * {mark <Description list> ( ) }              // Anonymous list SlipHeader()
 * {<Description list>( ) }                    // Anonymous list SlipHeader()
 * {mark ( ) }                                 // Anonymous list SlipHeader()
 *
 * {listName}                                  // sublist expansion SlipSublist(listName)
 *
 * Comments
 * /*   * /                                   // 'C' comments
 * //                                         // 'C++' comments
 *
 * Special Symbols  '(' ')' '{' '}' ',' ';' '/' '<' '>' '\"'
 * Keywords         include sublist user
 *
 * EXAMPLE
 *
 * sublist mysublist;
 *
 * #include "myFileName"
 *
 * sublist sublist1, sublist2;
 *
 * user myData;
 *
 * list list3 { 0x15 < c (1 2 {list1} 3)> () };
 *              |    | |  |                |
 *              |    | |  |                . empty sublist
 *              |    | |  |
 *              |    | |  . associated value in <key v>
 *              |    | |
 *              |    | . key in <k v>
 *              |    |
 *              |    . Description sublist
 *              |
 *              . user mark
 * sublist sublist4 { ( myData{ ( 3 4 ) } };
 * sublist sublist5 { 230 ( able ( baker charlie (foxtrot) one {sublist1} ) ) };
 *
 * ( this "is" a "sublist" {sublist1} (1 2 3) {sublist5} {10 <1 2> (able)} )
 *
 * Backup Naur Form (BNF)
 *
 * %define api.value.type { struct token stuff } or
 * #define YYSTYPE struct token stuff
 */
 
 /*
  * bison -d --no-lines --warnings=all --language=c++ slip.y
 */
 
 /*************************************
  *    Declarations and Options
  *************************************/
 
%{
# include <cstdint>
# include <string>
# include "lex.Slip.h"
//void yyerror(char const *s);
%}

 %file-prefix "Slip.bison"
 %language "C++"
 %name-prefix "Slip"
 %output "Slip.Bison.cpp"

%union YYSTYPE {                               //!< Token values
   bool          BOOL;                         //!< token boolean true / false
   int8_t        CHAR;                         //!< 8-bit  token value
   double        FLOAT;                        //!< 64-bit token value
   int32_t       INTEGER;                      //!< 32-bit token value
   string*       STRING;                       //!< pointer to new string()
   uint8_t       UCHAR;                        //!< 8-bit unsigned value
   uint32_t      UINTEGER;                     //!< 32-bit unsigned value
}

 %start sublistDeclarations_Definition
 %token <BOOL>          BOOL                   /* boolean value TRUE or FALSE       */
 %token <INTEGER>       EOL                    /* End-Of-File indication            */
 %token <CHAR>          CHAR                   /* a character 'c'                   */
 %token <CHAR>          CHARS                  /* a signed 8-bit number             */
 %token <UCHAR>         CHARU                  /* an unsigned 8-bit number          */
 %token <FLOAT>         FLOAT                  /* double value                      */
 %token <INTEGER>       INCLUDE                /* the literal 'include'             */
 %token <INTEGER>       INTEGER                /* a signed long integer             */
 %token <string>        NAME                   /* a string [a-zA-Z_$][a-zA-Z0-9_$]* */
 %token <string>        STRING                 /* a string "string"                 */
 %token <INTEGER>       SUBLIST                /* the literal 'sublist'             */
 %token <UCHAR>         UCHAR                  /* an unsigned character 'c'         */
 %token <UINTEGER>      UINTEGER               /* an unsigned long integer          */
 %token <INTEGER>       USER                   /* the literal 'user'                */

// %define api.value.type union
// %token <bool>          BOOL                   /* boolean value TRUE or FALSE       */
// %token <int>           EOL                    /* End-Of-File indication            */
// %token <char>          CHAR                   /* a character 'c'                   */
// %token <char>          CHARS                  /* a signed 8-bit number             */
// %token <unsigned char> CHARU                  /* an unsigned 8-bit number          */
// %token <double>        FLOAT                  /* double value                      */
// %token <int>           INCLUDE                /* the literal 'include'             */
// %token <int>           INTEGER                /* a signed long integer             */
// %token <string>        NAME                   /* a string [a-zA-Z_$][a-zA-Z0-9_$]* */
// %token <string>        STRING                 /* a string "string"                 */
// %token <int>           SUBLIST                /* the literal 'sublist'             */
// %token <unsigned char> UCHAR                  /* an unsigned character 'c'         */
// %token <unsigned int>  UINTEGER               /* an unsigned long integer          */
// %token <int>           USER                   /* the literal 'user'                */
// 
// %type <int>            sublistDeclarations_Definition
// %type <int>            declarations
// %type <int>            declarationItem
// %type <int>            forwardReferenceList
// %type <int>            sublistDefinition
// %type <int>            descriptionList
// %type <int>            descriptionItemlist
// %type <int>            descriptionItem
// %type <int>            descriptionElement
// %type <int>            include
// %type <string>         pathList
// %type <int>            userDataList
// %type <int>            listDefinition
// %type <int>            listItemList
// %type <int>            listItem
// %type <int>            expandName
// %type <int>            userData
// %type <int>            datum
// %type <int>            number

 %%
 /*************************************
  *              BNF
  *************************************/
sublistDeclarations_Definition:   
            declarations listDefinition
          | declarations                       /* declaration section               */
          | listDefinition                     /* list                              */
          | EOL                                /* End-Of-File handling              */
          ;

declarations:       
            declarations declarationItem       /* sublist of declarations           */
          | declarationItem                    /* single declaration                */
          ;

declarationItem:
            SUBLIST forwardReferenceList ';'   /* forward sublist declarations      */
          | NAME listDefinition ';'            /* sublist definition                */
          | USER userDataList ';'              /* user data names                   */
          | include                            /* include file def'n                */
          ;

forwardReferenceList:
            forwardReferenceList ',' NAME      /* forward sublist declarations list */
          | NAME                               /* forward sublist declaration       */
          ;

sublistDefinition:   
                                               /* full sublist declaration          */
            '{' number descriptionList listDefinition '}'
                                               /* list declaration w/o user mark    */
          | '{' descriptionList listDefinition '}'
          | '{' number listDefinition '}'      /* list declaration w/o description list */
          | '{' listDefinition '}'             /* simple sublist declaration        */
          ;

descriptionList:   
            '<' descriptionItemlist '>'        /* Association list                  */
          | '<' '>'                            /* empty association list            */
          ;

descriptionItemlist:
                                               /* Description List                  */
            descriptionItemlist descriptionItem
          | descriptionItem                    /* description item                  */
          ;  

descriptionItem:
                                               /* association entry                 */
          descriptionElement ',' descriptionElement
          ;

descriptionElement:
            expandName                         /* {NAME} sublist expansion          */
          | datum                              /* SlipDatum()                       */
          | listDefinition                     /* {NAME} SlipDatum()                */
          ;

include:    '#' INCLUDE '"' pathList '"'       /* include file path and name        */
          ;

pathList: pathList'/' NAME                     /* include file path                 */
        | NAME                                 /* include file name                 */
        ;
          
userDataList:                                  /* list of User Data names           */
            userDataList ',' NAME
          | NAME
          ;

listDefinition:       
            '(' listItemList ')'               /* list definition                   */
          | sublistDefinition                  /* SlipSubsublist(list)              */
          ;

listItemList:
            listItemList listItem              /* listHeader.enqueue(listItem)      */
          | listItem                           /* listHeader.enqueue(listItem)      */
          ;

listItem:   
            datum                              /* SlipDatum(datum)                  */
          | expandName                         /* SlipSubsublist(expandName)        */
          | userData                           /* SlipDatum(PTR)                    */
          | listDefinition                     /* SlipSubsublist(list)              */
          | '(' ')'                            /* SlipSublist(new SlipHeader() )    */
          ;

expandName: '{' NAME '}'                       /* SlipSubsublist(name)              */
          ;

userData:   NAME '{' listDefinition '}'        /* SlipDatum(PTR)                    */
          ;

datum:      BOOL                               /* SlipDatum((bool)boolValue)        */
          | CHAR                               /* SlipDatum((char)chr)              */
          | UCHAR                              /* SlipDatum((unsigned char)unsignedChr)*/
          | number                             /* discrete number                   */
          | FLOAT                              /* SlipDatum((double)floatNum)       */
          | STRING                             /* SlipDatum((string)string)         */
          ;
          
number:     INTEGER                         /* SlipDatum((long)integer)          */
          | UINTEGER                        /* SlipDatum((unsigned long)unsignedInteger)*/
          | CHARS                           /* SlipDatum((char)chr)              */
          | CHARU                           /* SlipDatum((unsigned char)unsignedChr)*/
          ;
%%

inline
int yylex(YYSTYPE * lval) { return Sliplex(YYSTYPE * lval); 
//void yyerror(char const *s) {
//	fprintf(stderr, "%s\n", s);
//}; // yyerror(char const *s)