/* ddrescueLog2badblocks V.0.02.01 2011-01-08
   compute badblocks as required by: e2fsck -vf -L badBlockList.txt       */
/* Copyright  2011 Valerio Messina */

/* This file is part of ddrescue.
   ddrescue is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   ddrescue is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with ddrescue.  If not, see <http://www.gnu.org/licenses/>.      */

/* input  data: pos [Bytes hex], size [Bytes hex], blockSize [Bytes]
   output data: list of unique blocks [#] */

// for each "-" line of ddrescue log file:
//    posBlocks=rint(pos/blockSize)
//    end=pos+size-1
//    endBlocks=rint(end/blockSize)
//    output all bad blocks in range: endBlocks-posBlocks inclusive
//    the list must be unique sorted to remove duplicates

#include <stdio.h>
#include <stdlib.h> // malloc, free
#include <string.h> // strlen, strstr

#define OK            0
#define ERROR        -1
#define WARN         -2
#define DEBUG        -3

#define PRINTOFF      0
#define PRINTERROR    1
#define PRINTWARN     2
#define PRINTBATCH    3
#define PRINTF        4
#define PRINTDEBUG    5
#define PRINTVERBOSE  6
#define PRINTALL      7

#define MaxSize ((((unsigned long int)1)<<31)-1) /* 2^31-1 = 2147483647 */
#define TERM    '\0'    /* String Terminator */

#define Lines            80ULL   // number of badZones "-" in ddrescue log
#define BytesPerSector  512ULL   // size in Bytes of one HDD sector
#define BytesPerBlock  4096ULL   // size in Bytes of one ext3 block
#define SectorPerCluster BytesPerBlock/BytesPerSector   // in number of sectors
                                                        // like ddrescue -c

typedef unsigned char           u08;
typedef unsigned long       int u32;
typedef unsigned long long  int u64;

typedef signed              int errOk;

typedef struct { // one badZone of ddrescue log
                   u64 pos;
                   u64 size;
               } badZone;

badZone* badZones; // will point to an array of ddrescue log badZones
u64* blocks; // will point to an array of badblocks with duplications
u64* unique; // will point to an array of unique badblocks

char debug=0; // set to 1 for verbose output, set to 2 for debug
u08 dbgLev=PRINTBATCH;

int printNchar (char* startPtr, u64 num) { /* print N char from start */
   u32 pos;
   if (startPtr == NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: startPtr point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   if (num == 0) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: num must be >0\n", __FUNCTION__);
      return (ERROR);
   }
   if (dbgLev>=PRINTVERBOSE) printf ("%2lluChar:'%c", num, *startPtr);
   for (pos=0; pos<num; pos++) {
      if (dbgLev>=PRINTF) printf ("%c", *(startPtr+pos));
      if (*(startPtr+pos)=='\0') break;
   }
   if (dbgLev>=PRINTDEBUG) printf ("'@%p\n", startPtr);
   if (dbgLev>=PRINTVERBOSE) printf ("\n");
   return (OK);
}

/* open a fileName in ReadOnly and return the filePtr or NULL on ERROR */
FILE* openRead (char* fileName) {
   FILE* out;
   if (fileName==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: fileName point to NULL\n", __FUNCTION__);
      return (NULL);
   }
   out = fopen (fileName, "r");
   if (out == NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: cannot open File:\"%s\"\n", __FUNCTION__, fileName);
   }
   return (out);
}

/* open a fileName in WriteOnly and return the filePtr or NULL on ERROR */
FILE* openWrite (char* fileName) {
   FILE* out;
   if (fileName==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: fileName point to NULL\n", __FUNCTION__);
      return (NULL);
   }
   out = fopen (fileName, "w");
   if (out == NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: cannot open File:\"%s\"\n", __FUNCTION__, fileName);
   }
   return (out);
}

/* open a file in ReadOnly and return the filePtr, file size or ERROR */
off_t getFileSize (char* fileName, FILE** filePtrPtr) {
   FILE* filePtr;
   long int len;  /* signed long. On SUS there is off_t as signed long long */
   int out;
   long int prev; /* signed long. On SUS there is off_t as signed long long */
   if (fileName==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: fileName point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   if (filePtrPtr==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: filePtrPtr point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   filePtr = openRead (fileName);
   if (filePtr==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: openRead returned NULL on File:\"%s\"\n", __FUNCTION__, fileName);
      return (ERROR);
   }
   // ftello is in SingleUnixSpecification not Posix, so no support in Cygwin
   prev = ftell (filePtr); // read current position (start)
   if (prev<0) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: ftell failed on File:\"%s\"\n", __FUNCTION__, fileName);
      return (ERROR);
   }
   out = fseek (filePtr, 0, SEEK_END); // go to end of file
   if (out!=0) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: seek failed on File:\"%s\"\n", __FUNCTION__, fileName);
      return (ERROR);
   }
   len = ftell (filePtr); // read current position (end)
   if (dbgLev>=PRINTVERBOSE) printf ("File length:\"%s\" is %ld\n", fileName, len);
   if (len<0) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: ftell failed on File:\"%s\"\n", __FUNCTION__, fileName);
      return (ERROR);
   }
   out = fseek (filePtr, prev, SEEK_SET); // seek to start
   if (out != prev) {
      if (dbgLev>=PRINTWARN) printf ("WARN %s: seek to prev:%ld failed on File:\"%s\"\n", __FUNCTION__, prev, fileName);
      return (ERROR);
   }
   *filePtrPtr = filePtr;
   return (len);
}

/* open and copy a file in RAM and return bufferPtr, buffer size or ERROR */
/* The file is closed. The user must free the bufferPtr at end of use */
off_t readFile (char* fileName, char** bufferPtrPtr) {
   FILE* filePtr;
   long int len;  /* signed long. On SUS there is off_t as signed long long */
   size_t Nch;          /* U32: like an unsigned int, not an unsigned long! */
   int out;
   if (fileName==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: fileName point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   if (bufferPtrPtr==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: bufferPtrPtr point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   len = getFileSize (fileName, &filePtr);
   if (len<0) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: getFileSize returned:%ld on File:\"%s\"\n", __FUNCTION__, len, fileName);
      return (ERROR);
   }
   if (dbgLev>=PRINTVERBOSE) printf ("file length:\"%s\" is %ld\n", fileName, len);
   if (len>=MaxSize) { /* limit memory allocation to 2GB */
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: file \"%s\" too big:%ld\n", __FUNCTION__, fileName, len);
      return (ERROR);
   }
   *bufferPtrPtr = (char*) malloc (len+1);               /* room for NULL */
   if (*bufferPtrPtr==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: cannot allocate %ld bytes of memory\n", __FUNCTION__, len);
      return (ERROR);
   }
   if (dbgLev>=PRINTVERBOSE) printf ("%s: buffer allocated at %p\n", __FUNCTION__, *bufferPtrPtr);
   if ( len >= (((u64)1)<<32) ) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: cannot read so much:%ld with fread(1)\n", __FUNCTION__, len);
      return (ERROR);
   }
   Nch = fread (*bufferPtrPtr, 1, len, filePtr);
/*   if (Nch != len) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: cannot read Nch/len:%u/%ld from file:\"%s\"\n", __FUNCTION__, Nch, len, fileName);
      return (Nch);
   }*/
   if (dbgLev>=PRINTVERBOSE) printf ("%s: file:'%s' of '%u' chars readed in buffer\n", __FUNCTION__, fileName, Nch);
   //if (dbgLev>=PRINTVERBOSE) printNchar ( ((*bufferPtrPtr)+len-5), 5 );
   *((*bufferPtrPtr)+len) = TERM; /* insert terminator */
   if (dbgLev>=PRINTVERBOSE) printf ("%s line:%u\n", __FUNCTION__, __LINE__);
   out = fclose (filePtr);
   if (out != 0) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: cannot close file:\"%s\"\n", __FUNCTION__, fileName);
      return (ERROR);
   }
   return (len);
}

/* copy RAM on created file and return written bytes or ERROR */
size_t writeFile (char* fileName, char* bufferPtr) { /* copy from RAM to file */
   FILE* filePtr;
   size_t Nblk;                      /* U32: like an unsigned int, not long */
   size_t len;
   int out;
   if (fileName==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: fileName point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   if (bufferPtr==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: bufferPtr point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   filePtr = openWrite (fileName);
   if (filePtr==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: Cannot write to file:\"%s\"\n", __FUNCTION__, fileName);
      return (ERROR);
   }
   len = strlen (bufferPtr);
   if (dbgLev>=PRINTVERBOSE) printf ("File:\"%s\" lenght is %d\n", fileName, len);
   if (dbgLev>=PRINTVERBOSE) printf ("Buffer to transfer at %p\n", bufferPtr);
   Nblk = fwrite (bufferPtr, 1, len, filePtr);
   if (Nblk != len) {
      if (dbgLev>=PRINTERROR) printf ("WARN %s: Cannot write Nblk/len:%d/%d sul file:\"%s\"\n", __FUNCTION__, Nblk, len, fileName);
      return (Nblk);
   }
   if (dbgLev>=PRINTVERBOSE) printf ("File of '%d' char written from buffer\n", Nblk);
   if (dbgLev>=PRINTVERBOSE) printNchar ( ((bufferPtr)+len-5), 5 );
   /**((*buffer)+len) = TERM;*//* inserisce il terminatore al termine del buffer */
   if (dbgLev>=PRINTVERBOSE) printf ("line:%u\n", __LINE__);
   out = fclose (filePtr);
   if (out != 0) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: Cannot close file:\"%s\"\n", __FUNCTION__, fileName);
      return (ERROR);
   }
   return (Nblk);
}

/* lines are the total lines with "-" in ddrescue logfile.
   blockSize is the ext3 cluster size in Bytes
   the output is written in unique[] */
int calcBlocks (u64 lines, u64 blockSize) {
   u64 pos, size, end;
   u64 posBlocks, endBlocks;
   u64 p=0, u=0;
   u64 b;
   
   if (blockSize==0) return -1;
   if (debug>=1) printf ("ddrescue bad zones list:\n");
   if (debug>=1) printf ("------------------------\n");
   for (u64 l=0; l<lines; l++) {
      
      // read one ddrescue "-" log line. Fill variables: pos & size, then ...
      pos=badZones[l].pos;
      size=badZones[l].size;
      if (debug>=1) printf ("l:%02llu pos:0x%llX size:0x%llX ", l, pos, size);
      if (debug>=1) printf ("\n");
      // end read one ddrescue log line.
      
      posBlocks=pos/blockSize; // integer division will trunc the result
      end=pos+size-1;
      endBlocks=end/blockSize; // integer division will trunc the result
      if (debug>=2) printf ("posBlocks:%llu endBlocks:%llu\n", posBlocks, endBlocks);
      for (b=posBlocks; b<=endBlocks; b++) {
         if (debug>=2) printf ("p:%02llu %llu ", p, b);
         if (debug>=2) printf ("\n");
         p++; // here determine only the number of blocks;
      }
   }
   if (debug>=3) printf ("p:%02llu\n", p);
   if (debug>=2) printf ("\n");
   blocks = malloc (sizeof(u64)*p);   // reserve space for the array
   if (debug>=2) printf ("blocks:%p\n", blocks);
   if (blocks==NULL) return -2;
   unique = malloc (sizeof(u64)*p);   // reserve space for the array
   if (debug>=2) printf ("unique:%p\n", unique);
   if (unique==NULL) return -2;
   if (debug>=2) printf ("\n");
   p=0;
   for (u64 l=0; l<lines; l++) {
      
      // here add the read one ddrescue log line. Fill variables: pos & size, then ...
      pos=badZones[l].pos;
      size=badZones[l].size;
      if (debug>=1) printf ("l:%02llu pos:0x%llX size:0x%llX ", l, pos, size);
      if (debug>=1) printf ("\n");
      // end read one ddrescue log line.
      
      posBlocks=pos/blockSize; // integer division will trunc the result
      end=pos+size-1;
      endBlocks=end/blockSize; // integer division will trunc the result
      if (debug>=2) printf ("posBlocks:%llu endBlocks:%llu\n", posBlocks, endBlocks);
      for (b=posBlocks; b<=endBlocks; b++) {
         
         // output all bad blocks in range: endBlocks-posBlocks inclusive
         // the list must be unique sorted to remove duplicates
         blocks[p]=b;
         if (debug>=2) printf ("p:%02llu %llu ", p, blocks[p]);
         if (debug>=2) printf ("\n");
         p++;
         // end output all bad blocks in range
         
      }
   }
   if (debug>=1) printf ("\n");
   if (debug>=1) printf ("e2fsck bad blocks list with duplicates:\n");
   if (debug>=1) printf ("---------------------------------------\n");
   for (b=0; b<p; b++) {
      if (debug>=1) printf ("b:%02llu %llu ", b, blocks[b]);
      if (debug>=1) printf ("\n");
   }
   if (debug>=1) printf ("\n");
   if (debug>=2) printf ("e2fsck bad blocks list:\n");
   if (debug>=2) printf ("-----------------------\n");
   blocks[p]=0;
   for (b=0; b<p; b++) {
      if (blocks[b]!=blocks[b+1]) {
         unique[u]=blocks[b];
         if (debug>=2) printf ("b:%02llu u:%02llu %llu ", b, u, unique[u]);
         if (debug>=2) printf ("\n");
         u++;
      }
   }
   if (debug>=2) printf ("\n");
   if (debug>=1) printf ("e2fsck bad blocks list:\n");
   if (debug>=1) printf ("-----------------------\n");
   for (b=0; b<u; b++) {
      printf ("%llu ", unique[b]);
      printf ("\n");
   }
   if (debug>=2) printf ("unique:%p\n", unique);
   //free (unique);
   if (debug>=2) printf ("blocks:%p\n", blocks);
   //free (blocks);
   //free (badZones);
   return 0;
}

/* parse of ddrescue log for "-" lines. Return ? or ERROR */
errOk parseLog (char* chPtr, off_t len, char** badZonesPtr) {
   char* chPos;
   u64 pos, size;
   char state;
   u64 m=0;
   
   if (chPtr==NULL) {
      if (dbgLev>=PRINTERROR) printf ("ERROR %s: chPtr point to NULL\n", __FUNCTION__);
      return (ERROR);
   }
   chPos=chPtr;
   for (u08 l=1; l<=4; l++) {
      chPos = strstr (chPos, "\n");
      chPos++;
   }
   for ( ; chPos-chPtr<len; ) {
      //printNchar (chPos, 26);
      //printf ("\n");
      pos=strtoll (chPos, &chPos, 16);
      //printNchar (chPos, 26);
      //printf ("\n");
      size=strtoll (chPos, &chPos, 16);
      //printNchar (chPos, 26);
      //printf ("\n");
      state=*(chPos+2);
      //printf ("pos:0x%010llX size:0x%010llX state:'%c'\n", pos, size, state);
      if (state=='-') {
         m++;
      }
      chPos = strstr (chPos, "\n");
      chPos++;
   }
   //printf ("m:%llu\n", m);
   badZones = malloc (sizeof(badZone)*m);   // reserve space for the array
   // now restart from beginning and fill the array
   m=0;
   chPos=chPtr;
   for (u08 l=1; l<=4; l++) {
      chPos = strstr (chPos, "\n");
      chPos++;
   }
   for ( ; chPos-chPtr<len; ) {
      pos=strtoll (chPos, &chPos, 16);
      size=strtoll (chPos, &chPos, 16);
      state=*(chPos+2);
      //printf ("pos:0x%010llX size:0x%010llX state:'%c'\n", pos, size, state);
      if (state=='-') {
         badZones[m].pos=pos;
         badZones[m].size=size;
         m++;
      }
      chPos = strstr (chPos, "\n");
      chPos++;
   }
   if (debug>=1) printf ("ddrescue log file bad zones list:\n");
   if (debug>=1) printf ("---------------------------------\n");
   for (u64 b=0; b<m; b++) {
      if (debug>=1) printf ("pos:0x%010llX size:0x%010llX\n", badZones[b].pos, badZones[b].size);
   }
   return (OK);
}

// logFile badblocksListFile. if badblocksListFile='-' send output on stdout
int main (int argc, char** argv) {
   int ret;
   off_t len;
   char* bufferPtr;
   char* badZonesPtr;
   // add command line argument parser
   len = readFile ("logDdrescueBackup07.txt", &bufferPtr);
   if (len==-1) {
      printf ("readFile returned:%d not OK\n", ret);
      return -1;
   }
   ret = parseLog (bufferPtr, len, &badZonesPtr);
   if (ret==-1) {
      printf ("parseLog returned:%d not OK\n", ret);
      return -1;
   }
   if (debug>=1) printf ("\n");
   ret = calcBlocks (Lines, BytesPerBlock);
   if (ret==-1) return -1;
   if (ret==-2) return -2;
   return 0;
}
