#include <gsl/gsl_errno.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_interp.h>
#include <gsl/gsl_spline.h>
#include "gsl_interp_ext.h"

int gsl_interp_batch_cspline_ext( gsl_spline *spline, 
        const double x[], const double y[], size_t n_in, 
        double x_out[], double y_out[], size_t n_out, 
        double deriv2_ratio, gsl_interp_accel *acc)
/* generalized, const. for 1 value, linear for 2 values, otherwise 
 *     cspline_ext speed-up the search using linear search 
 * xout need to be incremental to gain advantage of this function
 * spline and accel are pre-alloced because this function may be repeatly 
 *     called for the same dimension
 */
{ 
    size_t i;
    gsl_interp* interp = NULL; /* for linear interp */
    
    if(n_in == 1){
       for(i=0; i<n_out; i++)
           y_out[i] = y[0];
       return 1;
    }
    
    gsl_interp_accel_reset(acc); /* reset the index */
    acc->search_type = GSL_INTERP_LSEARCH; /* force linear search since we know xout[] is incremental */
    
    if(n_in == 2){ /* two points, use linear interpolation */
        interp = gsl_interp_alloc(gsl_interp_linear, n_in);
        interp->allow_extrap = 1;
        for(i=0; i<n_out; i++)
            y_out[i] = gsl_interp_eval(interp, x, y, x_out[i], acc);
        gsl_interp_free(interp);
        return 1;
    }
    
    spline->interp->allow_extrap = 1;
    ((cspline_state_t *)spline->interp->state)->ext_info = &deriv2_ratio;
    gsl_spline_init(spline, x, y, n_in);
    
    for(i=0; i<n_out; i++)
        y_out[i] = gsl_spline_eval(spline, x_out[i], acc);
    return 1;
}

int gsl_interp_grid2d_cspline_ext(gsl_spline *splines[], const double *z, 
        const double x[], size_t nx,
        const double y[], double zwrk[], size_t ny,
        const double x_out[], const double y_out[], double z_out[], size_t n_out, 
        double deriv2_ratio, gsl_interp_accel *accx, gsl_interp_accel *accy) 
/* z is data value of nx*ny size, along x first
 * spline[ny+1] is array of pre-alloc'ed spline buffer, the last one for y-direction
 * accx, accy are x & y accelerators
 * zwrk[ny] is the buffer for workspace along y
 */
{   
    size_t iy, i;
    gsl_interp* interp = NULL; // for linear interp along y
    if(nx < 3){
       GSL_ERROR ("nx must be >= 3", GSL_EINVAL); 
    }

    /* init, could be split into a separate function */
    for(iy = 0; iy < ny; iy++){
        splines[iy]->interp->allow_extrap = 1;
        ((cspline_state_t *)splines[iy]->interp->state)->ext_info = &deriv2_ratio;
        gsl_spline_init(splines[iy], x, z+iy*nx, nx);
    }
    
    /* evaluate, could be separated to another function for single evaluation */
    if(ny == 2){
        interp = gsl_interp_alloc(gsl_interp_linear, ny);
        interp->allow_extrap = 1;
    }
    else if (ny >= 3){
        splines[ny]->interp->allow_extrap = 1;
        ((cspline_state_t *)splines[ny]->interp->state)->ext_info = &deriv2_ratio;
    }
    
    for(i=0; i<n_out; i++){
        for(iy = 0; iy < ny; iy++){
            zwrk[iy] = gsl_spline_eval(splines[iy], x_out[i], accx);
        }
        
        if(ny == 1)
            z_out[i] = zwrk[0];
        else if(ny==2){
            z_out[i] = gsl_interp_eval(interp, y, zwrk, y_out[i], accy);
        }
        else {
            gsl_spline_init(splines[ny], y, zwrk, ny);
            z_out[i] = gsl_spline_eval(splines[ny], y_out[i], accy);
        }
    }
    if(interp){
        gsl_interp_free(interp);
    }
    return 1;
}

/* modifications for interpolation/cspline.c */

/* BEGINNING OF THE EXT part, by Feng Chen, 2011 */
/* parabolic runout: y"[1]=y"[2], y"[m]=y"[m+1]
 * adjust parameter: ((cspline_state_t *)spline->interp->state)->ext_info = &deriv2_ratio;
 */
static int
cspline_init_ext (void * vstate, const double xa[], const double ya[],
              size_t size)
{
  double deriv2_ratio = 1.0; /* default value for parabolic runout */
  cspline_state_t *state = (cspline_state_t *) vstate;

  size_t i;
  size_t num_points = size;
  size_t max_index = num_points - 1;  /* Engeln-Mullges + Uhlig "n" */
  size_t sys_size = max_index - 1;    /* linear system is sys_size x sys_size */

  state->c[0] = 0.0;
  state->c[max_index] = 0.0;

  for (i = 0; i < sys_size; i++)
    {
      const double h_i   = xa[i + 1] - xa[i];
      const double h_ip1 = xa[i + 2] - xa[i + 1];
      const double ydiff_i   = ya[i + 1] - ya[i];
      const double ydiff_ip1 = ya[i + 2] - ya[i + 1];
      const double g_i = (h_i != 0.0) ? 1.0 / h_i : 0.0;
      const double g_ip1 = (h_ip1 != 0.0) ? 1.0 / h_ip1 : 0.0;
      state->offdiag[i] = h_ip1;
      state->diag[i] = 2.0 * (h_ip1 + h_i);
      state->g[i] = 3.0 * (ydiff_ip1 * g_ip1 -  ydiff_i * g_i);
    }
  /* deriv2_ratio = 0: natural cspline 
   * deriv2_ratio = 1:parabolic runout boundary conditions
   * other values (including negative): ratio of the two 2nd-derivatives near the boundary
   */
  if(state->ext_info)
      deriv2_ratio = *((double *)state->ext_info);
  state->diag[0] += deriv2_ratio * state->offdiag[i];
  state->diag[sys_size-1] += deriv2_ratio * state->offdiag[sys_size-1];

  if (sys_size == 1)
    {
      state->c[1] = state->g[0] / state->diag[0];
      return GSL_SUCCESS;
    }
  else
    {
      gsl_vector_view g_vec = gsl_vector_view_array(state->g, sys_size);
      gsl_vector_view diag_vec = gsl_vector_view_array(state->diag, sys_size);
      gsl_vector_view offdiag_vec = gsl_vector_view_array(state->offdiag, sys_size - 1);
      gsl_vector_view solution_vec = gsl_vector_view_array ((state->c) + 1, sys_size);
      
      int status = gsl_linalg_solve_symm_tridiag(&diag_vec.vector, 
                                                 &offdiag_vec.vector, 
                                                 &g_vec.vector, 
                                                 &solution_vec.vector);
      return status;
    }
}


static const gsl_interp_type cspline_ext_type = 
{
  "cspline-extended", 
  3,
  &cspline_alloc,
  &cspline_init_ext,
  &cspline_eval,
  &cspline_eval_deriv,
  &cspline_eval_deriv2,
  &cspline_eval_integ,
  &cspline_free
};

const gsl_interp_type * gsl_interp_cspline_ext = &cspline_ext_type;


