#!/usr/bin/env -S guile \\
-e main -s 
!#

(use-modules (ice-9 binary-ports)
             (ice-9 pretty-print)
             (ice-9 sandbox)
             (ice-9 match)
             (rnrs bytevectors)
             (wasm assemble)
             (wasm resolve)
             (wasm link)
             (wasm wat))

(define dbg? (make-parameter #t))
(define *include-depth* (make-parameter 1))

(define (help progname args)
  (format #t "~a [file] ~%" progname)
  (format #t "  ~a --help:~%\tprint this message~%" progname)
  (format #t "  ~a file:~%\tevaluate then assemble WAT source as a quasiquoted scheme datum~%" progname)
  (format #t "\tThe output of the quasiquote must be a valid GWAT file in this iteration.~%")
  (format #t "\tWe may support shelling out to wabt and such in future~%")

  (format #t "~%Exit Codes:~%")
  (format #t "| 1 => CLI didn't parse.~%")
  (format #t "| 2 => form after `module' in called SWAT file.~%")
  (format #t "| 3 => unknown or disallowed top level form.~%")
  (format #t "| 4 => file exited before wasm was defined.~%")
  (format #t "| 5 => *include-depth* exceeded for macro files.~%")
  (format #t "| 6 => parameters failed to apply from command-line.~%")
  (format #t "| 7 => file passed in from CLI doesn't exist.~%")
  (format #t "~%| -1 => unimplemented~%")
  (exit 0))

(define (format-error fmt . args)
  (apply format `(,(current-error-port) ,fmt ,@args)))

(define (format-exit code fmt . args)
  (apply format `(,(current-error-port) ,fmt ,@args))
  (exit code))

(define (top-level-group? datum)
  (or (eq? (car datum) 'define)
      (eq? (car datum) 'define-syntax)
      (eq? (car datum) 'begin)
      (eq? (car datum) 'eval-when)
      (eq? (car datum) 'let)))

(define* (macros-include f m #:optional (depth 1))
  (call-with-input-file f
    (lambda (f)
      (let lp ((datum (read f))
               (depth depth))
        (cond
          ((eof-object? datum) 
           #t)
          ((> depth (*include-depth*))
           (format-exit 5
             "[err]: include depth exceeded in file ~a!~%"
             (port-filename f)))

          ((top-level-group? datum)
           ;; do we collect params in headers also ??
	   (when (dbg?) 
            (format-error "module_~a eval: ~%    ~S~%" m datum))
           (eval datum m)
           (lp (read f) depth))
    
          ((eq? (car datum) 'include)
           (macros-include f m (1+ depth))
           (lp (read f) depth))
    
          (else
           (format-error "[errinfo]: ~S~%" datum)
           (format-exit 2
            (string-append
             "[err]: top level forms must be `define' or `define-syntax', `let'"
             " or `begin' in file ~a!~%")
            (port-filename f))))))))

(define-inlinable (length= l n)  (= (length l) n))
(define-inlinable (length>= l n) (>= (length l) n))

(define (all-identifiers modspec)
  (module-map (lambda (i x) i) (resolve-interface modspec)))

(define src #f)
(define (process-swat swat parameter-overrides)
  (call-with-input-file swat
    (lambda (f)
      (let ((m (make-sandbox-module
                 (append
                   `(;((ice-9 binary-ports) . )
                     ((rnrs bytevectors) ,@(all-identifiers '(rnrs bytevectors)))
                     ((srfi srfi-43) ,@(all-identifiers '(srfi srfi-43)))
                     ((srfi srfi-1) ,@(all-identifiers '(srfi srfi-1)))
                     ((guile) set!))
                   all-pure-and-impure-bindings))))
        (let ((src-lit (read f)))
          (let lp ((datum src-lit)
                   (parameters '()))
            (cond
              ((eof-object? datum)
               (unless src
                 (format-exit 4
                   "[err]: EOF before module form in the file ~a!~%"
                   (port-filename f))))

              ((or (eq? (car datum) 'module)
                   (and (eq? (car datum) 'quasiquote)
                        (eq? (caadr datum) 'module)))
               (set! src-lit datum)
               (when (eq? (car src-lit) 'module)
                 (set! src-lit (list 'quasiquote src-lit)))

               (when (dbg?)
                 (format-error "unexpanded wasm: ~%")
                 (pretty-print src-lit
                   #:port (current-error-port)))

               (set! datum (read f))
               (unless (eof-object? datum)
                 (format-exit 2
                   "[err]: `module` is not the last form in the file ~a!~%"
                   (port-filename f)))

               ;; ideally if this were `eval-in-sandbox' again, it would be
               ;; a (let () ...) wrapping over all the defines and such, with
               ;; the parameter-overrides injected before the quasiquote

               ;; set parameters here
               (map (lambda (s)
                      (eval
                       (list (list s)
                             (assoc s parameter-overrides))
                       m))
                    parameters)

               (set! src (eval src-lit m)))

              ((top-level-group? datum)
               (let ((res (eval datum m)))
                 (lp (read f)
                     (if (parameter? res)
                         (cons (cadr datum) parameters)
                         parameters))))

              ((or (eq? (car datum) 'include)
                   (eq? (car datum) 'load))
               (when (dbg?)
                 (format-error "including file ~a!~%" (cadr datum)))
               (macros-include (cadr datum) m)
               (lp (read f) parameters))

              (else
                (format-exit 3
                  "[err]: top level forms must be `define`, `define-syntax` or `module` in file ~a!~%"
                  (port-filename f))))))))))

(define (main args)
  (define progname (car args))
  (format (current-error-port) "~a:args -> ~S~%" progname (cdr args))
  (let cli-lp ((args (cdr args))
               (options '()))
    (match args
      ([(or "--help" "-h") . args]
       (help progname args)
       ;; to be sure
       (exit 0))

      (["--barf-wat" . args]
       (cli-lp args (cons (cons 'barf-wat #t) options)))

      ([(? string? file) . args]
       (unless (file-exists? file)
         (format-exit 7
           "[err]: file passed does not exist ~a!~%"
           file))
       (cli-lp args (cons (cons 'file file) options)))

      ([(or "--params=" "--overrides=") (? string? paramlist) . args]
       (let ((datum (with-input-from-string paramlist
                      (lambda ()
                        (read))))
             (current-overrides
              (assoc 'params options)))
         (cond
          ((list? datum) ;; TODO: figure out how to multiply assoc-set!
           (set! current-overrides
                 (let lp ((d datum) (co current-overrides))
                  (cond
                   ((null? d) co)
                   (else co)))))
          ((pair? datum)
           (set! current-overrides (assoc-set! current-overrides (car datum) (cdr datum))))
          (else
           (format-exit 6
            "[err]: unknown datum for parameter override: ~S~%"
            datum)))
           
         (cli-lp args
                 (assq-set! options 'params current-overrides))))

      (_ (identity identity)))
    ;; DO NOT CALL `cli-lp' FROM BEYOND THIS LINE
    (let ((file (and=> (assoc 'file options) cdr))
          (barfp (and=> (assoc 'barf-wat options) cdr)))
      (format (current-error-port) "file:=~S~%" file)
      (format (current-error-port) "barfp:=~S~%" barfp)
      (unless file (help progname '()))
      (process-swat file (assoc 'params options))

      (cond (barfp
             (pretty-print src)
             (exit 0))
            (else
             (define wasm
               ;; apparently `lower-wasm' doesn't want to work here.
               ((compose assemble-wasm resolve-wasm wat->wasm)
                src))
             (put-bytevector (current-output-port) wasm)))
      (exit 0))))
