
/* ------------------------------------------------------------------- */
/* Copyright (c) 2007-2008 Hanzo Archives Limited.                     */
/*                                                                     */
/* Licensed under the Apache License, Version 2.0 (the "License");     */
/* you may not use this file except in compliance with the License.    */
/* You may obtain a copy of the License at                             */
/*                                                                     */
/*     http://www.apache.org/licenses/LICENSE-2.0                      */
/*                                                                     */
/* Unless required by applicable law or agreed to in writing, software */
/* distributed under the License is distributed on an "AS IS" BASIS,   */
/* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or     */
/* implied.                                                            */
/* See the License for the specific language governing permissions and */
/* limitations under the License.                                      */
/*                                                                     */
/* You may find more information about Hanzo Archives at               */
/*                                                                     */
/*     http://www.hanzoarchives.com/                                   */
/*                                                                     */
/* You may find more information about the WARC Tools project at       */
/*                                                                     */
/*     http://code.google.com/p/warc-tools/                            */
/* ------------------------------------------------------------------- */
/**
 * @param source input file handle to compress
 * @param dest ouput file handle (compressed data will be there)
 * @param _level compression level
 * @param csize: size of the compressed data
 *
 * @return Z_OK (i.e. 0) if the operation succeeds, 
 * non zero value otherwise
 *
 * Compress the data pointed by file handle "source" with 
 * compression level "_level" and store the result in file handle 
 * pointed by "dest". Store the size of the compessed data in "csize".
 */

WPUBLIC warc_i32_t compress (FILE * source, FILE * dest,
                             const warc_gzlevel_t _level, 
                             warc_u32_t * csize)
{
    unsigned   char in  [IN_BUFFER_SIZE];
    unsigned   char out [IN_BUFFER_SIZE];
    z_stream   strm;
    warc_i32_t ret;
    warc_u32_t have;
    warc_u32_t flush;
    warc_u32_t level;

    /* preconditions */
    assert  (source);
    assert  (dest);


    /* ensure that the compressed file is in binary mode */
    SET_BINARY_MODE (dest);
    
    /* force the compressed size "csize" to zero */
    * csize = 0;

    /* allocate deflate state */
    strm.zalloc = Z_NULL;
    strm.zfree  = Z_NULL;
    strm.opaque = Z_NULL;

    /* map between user compression level and Zlib one */
    switch (_level) 
      {
      case WARC_GZIP_DEFAULT_COMPRESSION: 
        level = Z_DEFAULT_COMPRESSION; break;
      case WARC_GZIP_NO_COMPRESSION:      
        level = Z_NO_COMPRESSION;      break;
      case WARC_GZIP_BEST_SPEED:          
        level = Z_BEST_SPEED;          break;
      case WARC_GZIP_BEST_COMPRESSION :   
        level = Z_BEST_COMPRESSION;    break;
      default:                            
        level = Z_DEFAULT_COMPRESSION;
      }
    
    /* ret = deflateInit(&strm, level); */
    ret = deflateInit2 (& strm, 
                        level, 
                        Z_DEFLATED,
                        methodBits (GZIP_METHOD),  
                        8, 
                        Z_DEFAULT_STRATEGY);

    if (ret != Z_OK)
      goto RET;

    /* compress until end of file */
    do {
        strm.avail_in = w_fread (in, 1, IN_BUFFER_SIZE, source);
        if (w_ferror (source)) 
          {
            ret = Z_ERRNO;
            goto END;
          }

        flush = w_feof (source) ? Z_FINISH : Z_NO_FLUSH;
        strm.next_in = in;

        /* run deflate() on input until output buffer not full, finish
           compression if all of source has been read in */
        do {
            strm.avail_out = IN_BUFFER_SIZE;
            strm.next_out  = out;

            ret = deflate (& strm, flush);    /* no bad return value */
            assert (Z_STREAM_ERROR != ret);   /* state not clobbered */

            have = IN_BUFFER_SIZE - strm.avail_out;

            /* how many bytes were compressed */
            * csize += have;

            if (w_fwrite (out, 1, have, dest) != have || w_ferror (dest))
              {
                ret = Z_ERRNO;
                goto END;
              }

            if (fflush (dest)) 
              {
                ret = Z_WARC_FLUSH;
                goto END;
              }

        } while (strm.avail_out == 0);

        assert(strm.avail_in == 0);     /* all input will be used */

        /* done when last data in file processed */
    } while (flush != Z_FINISH);

    assert (Z_STREAM_END == ret);       /* stream will be complete */

 END:
    /* clean up and return */
    deflateEnd (& strm);
    
    /* got Z_STREAM_END is OK */
    if (ret == Z_STREAM_END)  
      ret = Z_OK;

    if (ret != Z_OK)
      ret = 0;
    
 RET:

    return (ret);
}
