#include <mailutils/mailutils.h>

mu_list_t sender;
size_t match_count;

static int
sender_cmp (const void *a, const void *b)
{
  return mu_c_strcasecmp ((char const *)a, (char const *)b);
}

static void
cli_from (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  if (!sender)
    MU_ASSERT (mu_list_create (&sender));

  MU_ASSERT (mu_list_append (sender, mu_strdup (arg)));
}

struct mu_option lastmsg_options[] = {
  { "from", 'f', "EMAIL", MU_OPTION_DEFAULT,
    "sender email",
    mu_c_string, NULL, cli_from },
  { NULL }
}, *options[] = { lastmsg_options, NULL };

static struct mu_cli_setup cli = {
  options,
  NULL, /* Add configuration parameters, if necessary */
  "identify mail folders with last message sent from given address(es)",
  "MBOX..."
};

static char *capa[] = {
  "debug",
  "mailbox",
  NULL
};

enum
  {
    EX_OK = 0,
    EX_SOME_MATCH,
    EX_NO_MATCH,
    EX_ERROR,
    EX_USAGE = 64
  };

static int code = EX_OK;

static inline void
set_code (int c)
{
  if (code < c)
    code = c;
}

static int
match_envelope (mu_message_t msg)
{
  mu_envelope_t env;
  char const *s = NULL;
  
  mu_message_get_envelope (msg, &env);
  mu_envelope_sget_sender (env, &s);
  if (!s)
    return 0;
  return mu_list_locate (sender, (void*) s, NULL) == 0;
}
    
static int
match_header (mu_message_t msg)
{
  mu_header_t hdr;
  static char const *hn[] = {
    MU_HEADER_SENDER,
    MU_HEADER_FROM,
  };
  int i;
    
  mu_message_get_header (msg, &hdr);

  for (i = 0; i < MU_ARRAY_SIZE (hn); i++)
    {
      char const *val;
      if (mu_header_sget_value (hdr, hn[0], &val) == 0
	  && mu_list_locate (sender, (void*)val, NULL) == 0)
	return 1;
    }
  return 0;
}

static void
match_last (char const *name)
{
  int rc;
  mu_mailbox_t mbox;
  mu_message_t msg;
  size_t n;
  
  rc = mu_mailbox_create_default (&mbox, name);
  if (rc)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_mailbox_create_default", name, rc);
      set_code (EX_ERROR);
      return;
    }

  rc = mu_mailbox_open (mbox, MU_STREAM_READ);
  if (rc)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_mailbox_open", name, rc);
      set_code (EX_ERROR);
      return;
    }  

  rc = mu_mailbox_messages_count (mbox, &n);
  if (rc)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_mailbox_messages_count", name, rc);
      set_code (EX_ERROR);
      return;
    }

  rc = mu_mailbox_get_message (mbox, n, &msg);
  if (rc)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_mailbox_get_message", name, rc);
      set_code (EX_ERROR);
      return;
    }

  if (match_envelope (msg) || match_header (msg))
    {
      match_count++;
      mu_printf ("%s\n", name);
    }

  mu_mailbox_destroy (&mbox);
}  
  
int
main (int argc, char *argv[])
{
  int i;
  
  mu_register_local_mbox_formats ();
  mu_cli (argc, argv, &cli, capa, NULL, &argc, &argv);
  if (argc < 1)
    {
      mu_error ("not enough arguments");
      exit (EX_USAGE);
    }
  if (!sender)
    {
      MU_ASSERT (mu_list_create (&sender));
      MU_ASSERT (mu_list_append (sender, mu_get_user_email (NULL)));
    }
  mu_list_set_comparator (sender, sender_cmp);
  for (i = 0; i < argc; i++)
    match_last (argv[i]);
  if (match_count == 0)
    set_code (EX_NO_MATCH);
  else if (match_count < argc)
    set_code (EX_SOME_MATCH);
  return code;
}
    
