#!/usr/local/gnu/bin/perl -w
####  _______              __
#### / ___/ /  ___  __ _  / /  ___
####/ /__/ _ \/ _ \/  ' \/ _ \/ _ \
####\___/_//_/\___/_/_/_/_.__/\___/ 
####
####
#### This software is copyright (C) by the Lawrence Berkeley
#### National Laboratory.  Permission is granted to reproduce
#### this software for non-commercial purposes provided that
#### this notice is left intact.
#### 
#### It is acknowledged that the U.S. Government has rights to
#### this software under Contract DE-AC03-765F00098 between
#### the U.S.  Department of Energy and the University of
#### California.
####
#### This software is provided as a professional and academic
#### contribution for joint exchange. Thus it is experimental,
#### is provided ``as is'', with no warranties of any kind
#### whatsoever, no support, no promise of updates, or printed
#### documentation. By using this software, you acknowledge
#### that the Lawrence Berkeley National Laboratory and
#### Regents of the University of California shall have no
#### liability with respect to the infringement of other
#### copyrights by any part of this software.
####

#
# -------------------------------------------------------------------------
# This is the Chombo Fortran PreProcessor (chfpp)
#
# This PERL script will read Fortran code (with Chombo Fortran  extensions)
# It will write C++ prototypes of the SUBROUTINEs and FUNCTIONs encountered to a file named:
#        <input_ftn_filename>_F.H
# where <input_ftn_file_name> is the name of the input Fortran file minus extension
# It also will write Fortran code containing Chombo cpp macros to standard output.
#
# See the "Chombo C++/Fortran Interface User Guide" and
# "Chombo C++/Fortran Interface Specification" for details.
#
# The output C++ header files and Fortran files are made both
# dimension-independent and precision-independent through the use of the
# BoxLib macros:  
#    CH_SPACEDIM
#    Real    (C++ only)
#    REAL_T  (Fortran only. Alternate spellings: Real_T, Real_t, real_t)
#
# The C++ header files are made independent of the calling conventions of the
# Fortran compiler using the macros in the file:
#    $CHOMBO_HOME/include/FORT_PROTO.H
# Caveats:
#   There are two classes of issues to keep in mind when writing Chombo Fortran code:
#    1) some things allowed by the Fortran 77 standard are not supported by ChF;
#    2) some common extensions to Fortran are not supported by ChF.
#   In particular:
#    1a) REAL and DOUBLEPRECISION should not be used.  Use REAL_T instead.
#    1b) Fortran code generated by chfpp may exceed the Fortran standard limit
#        of 18 continuation lines per statement even if the input code does not
#        exceed the limit.  If this happens, it may be possible to use a compiler
#        option to increase the limit.  Otherwise, reduce the number of continuation
#        lines in the original code or break the statement into multiple statements.
#
#
# Implementation notes:
#   All arguments in the C++ prototypes are declared `*const'
#   because all arguments passed to Fortran must be pointers, and the Fortran
#   routines can never modify the pointers.
#
# Method:
#   Lines of the input file are read into a buffer.  When a complete Fortran
#   statement is read (allowing for continuation lines) it is parsed.  
#   If Fortran output is enabled, the statement is output either in its original
#   form or after parsing.  Parsing involves recognizing Fortran procedure
#   declarations and producing special output.  All other Fortran statements are
#   output unchanged.  
#   To parse a Fortran statement, all continued lines are gathered
#   up into a single line, all whitespace and comments are removed and all characters are 
#   converted to lower case.  This makes is easier to find the statement
#   types that must be modified.
#
# Author:  David B. Serafini
#          Applied Numerical Algorithms Group
#          Lawrence Berkeley National Laboratory
# Date:    11 Aug 1999
#
#  28 Jul00 made into perl module by dtgraves
# also:
#  -changed the fortran comment string to the much more sensible c
#  -added file closing and input. 
#  -changed interface to be a subroutine interface
#  -interface is 
#  ChomboFortran::procChomboFortranMacros(input_file,c_output_file, 
#                                         fortran_output_file, debug)
#  
# -------------------------------------------------------------------------

package ChomboFortran;

use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS $VERSION);
use Exporter;
$VERSION = 1.23;
@ISA = qw(Exporter);
@EXPORT = qw(&procChomboFortranMacros);
@EXPORT_OK = qw(&procChomboFortranMacros);
@EXPORT_TAGS= ();

sub ChomboFortran::procChomboFortranMacros
{
($ifile, $ofile, $foutfile, $debugD) = @_;
use File::Basename ;

# Constants

$CHFPP_Version = "1.23" ;

%C_types = ( integer => "CHFp_INT"
            ,real_t  => "CHFp_REAL" ) ;
%Cfunc_types = ( integer => "int"
                ,real_t  => "Real" ) ;

###this comment string is magic. do NOT change it
$F_comment = "!" ;


# Global variables

$MAXDIM = 3 ;    #largest # of dimensions for which code can be built
$MAXDIMm1 = $MAXDIM - 1 ;

%C_symbol_table = () ;  #hash of {arg string} => C declaration string 
%F_symbol_table = () ;  #hash of {arg string} => Fortran77 declaration string
@arg_list = () ;
%name_list = () ;       #hash of {box-or-fab name} => 1

$name = "" ;  #lower case name of Fortran procedure
$NAME = "" ;  #upper case name of Fortran procedure
$Cindent = "" ; 

$outputC = 1 ; # when true, output C++ header file
$outputF = 1 ; # when true, output Fortran source file

$ChF_private = 0 ;  #if true, don't prototype the next procedure
$ChF_multido = 0 ;  #if >0, currently inside a MULTIDO

### -- needed for `D_TERM()' hack. removed in v1.13
###use Config;
###$Cray = $Config{archname} =~ /cray/i ;  #to handle broken Cray cpp

# Begin execution


# one input file is required
( $ifile !~ /^$/ ) or die "Error: one filename is required on the command line\n" ;

## open and initialize output files
open(O,">" . $ofile) 
    or die "Error: cannot open output file " . $ofile . "\n";
open(FOUT,">" . $foutfile) 
    or die "Error: cannot open output file " . $foutfile . "\n";
open(F,"<" . $ifile) 
        or die "Error: cannot open input file " . $ifile . "\n";



# initialize the Fortran output file
$outputF and &start_F_file() ;

# open and initialize C++ output file
if( $outputC == 1 ){
    # extract leading directory and file basename.  Throw away the trailing suffix.
    ($basename,$basepath) = fileparse( $ifile ,'\..*' ) ;
    open(O,">" . $basepath . $ofile) or die "Error: cannot open prototype output file $basepath$ofile" ;
    &start_C_file() or die "Error: failure while initializing prototype output file $basepath$ofile" ;
}


# loop through lines in the input file
$linenum = 0 ;
$statement = "" ;
$obuf = "" ;   
while (defined( $ibuf = <F> )) {
    $linenum++ ;

    # skip comments
    next if $ibuf =~ /^([c*]|[ \t]*!)/i ;
    # skip blank lines
    next if $ibuf =~ /^[ \t]*$/ ; 
    # add cpp directives to the output buffer without any further processing
    if( $ibuf =~ /^[ \t]*\#/ ) {
        $obuf .= $ibuf ;
        next ;
    }

    # If this line is not a continuation line, parse the contents of the
    # statement buffer and overwrite the statement buffer with this line.
    # If the line is a continuation line, append it to the statement buffer.
    # [NOTE: A zero in column 6 is _not_ a continuation marker.]
    # [NOTE: the 2nd part of the regexp allows for "tab" source form.]
    if ( $ibuf =~ /^([ 0-9]{5}[ 0])|(\t[^1-9])/ ) {
        # if the statement buffer is empty but the output buffer is not,
        # just print the output buffer
        if ( $outputF and $obuf and ! $statement ) { print FOUT $obuf ; }

        # if the statement buffer is not empty, process it
        if ( $statement ) { &parse_statement( $statement ,$obuf ) ; }

        # Start a new statement.
        $obuf = $ibuf ;  chomp $ibuf ;        # save the original line
        $statement = &stripcomment( $ibuf ) ; # put the cleaned-up version in the statement buffer
    } else {
        # this line is a continuation of the previous statement
        # append it to the output buffer
        $obuf .= $ibuf ; chomp $ibuf ;

        # strip any trailing comment
        # [NOTE: this will fail if a string constant is continued across
        #        lines and there is a trailing comment.]
        $F1 = &stripcomment( $ibuf ) ;

        # remove the continuation marker (second part allows for tab source form)
        $F1 =~ s/^     .// ;  $F1 =~ s/^\t[^1-9]// ; 

        # append to the statement buffer
        $statement .= $F1 ;
    }
}

# process the last statement in the file
if( $statement ) { &parse_statement( $statement ,$obuf ) ; }

# Finish the C++ output file
$outputC and &finish_C_file() ;

close(O);
close(F);
close(FOUT);
return 1;

################################################################
#
#  Top-level subroutines
#
################################################################


# -------------------------------------------------------------------
# Subroutine: start_F_file
# Inputs: none
# Globals: ifile, CHFPP_Version
# Locals:
# Returns: nothing
# Action:
#  write stuff at the top of the Fortran output file
# -------------------------------------------------------------------

sub start_F_file {
####    print FOUT "$F_comment -*- Mode: Fortran; Modified: \"",scalar(localtime())," by $ENV{USER}\" -*-\n" ;
####    print FOUT "\n" ;
####    print FOUT "$F_comment This file was automatically generated by:\n" ;
####    print FOUT "$F_comment     ",__FILE__," version $CHFPP_Version\n" ;
####    print FOUT "$F_comment from the file: \n" ;
####    print FOUT "$F_comment     $ifile\n" ;
####    print FOUT "\n" ;
    print FOUT "#include \"ChomboFortran.H\"\n" ;
    print FOUT "\n" ;
}

# -------------------------------------------------------------------
# Subroutine: start_C_file
# Inputs: none
# Globals:
#   O  is the C++ output filehandle
#   ifile  is the name of the input file
#   CHFPP_Version  is the version number of this implementation of chfpp
# Locals:
#   tmp  is the basename of the Fortran file converted to upper case
#        for use as a cpp macro variable to protect against multiple #includes
# Returns: nothing
# Action:
#  write stuff at the top of the C++ prototype output file
# -------------------------------------------------------------------

sub start_C_file {
    my $tmp = uc( $basename ) ;
    print O "// -*- Mode: C++; Modified: \"",scalar(localtime())," by $ENV{USER}\" -*-\n" ;
    print O "\n" ;
    print O "\n" ;
    print O "// This file was automatically generated by ",__FILE__," version $CHFPP_Version\n" ;
    print O "// from the file: $ifile\n" ;
    print O "\n" ;
    print O "#ifndef _" ,$tmp ,"_F_H_\n" ;
    print O "#define _" ,$tmp ,"_F_H_\n" ;
    print O "\n" ;
    print O "#include \"FORT_PROTO.H\"\n" ;
    print O "#include \"REAL.H\"\n" ;
    print O "\n" ;
    print O "#ifndef CH_SPACEDIM\n" ;
    print O "#error CH_SPACEDIM macro must be defined.\n" ;
    print O "#endif\n" ;
    print O "\n" ;
    print O "extern \"C\"\n" ;
    print O "{\n" ;    
}

# -------------------------------------------------------------------
# Subroutine numquotes
# Action: count the number of quotes (") in a string
# Inputs: string to search
# Returns: number of quotes in string
# Globals: none
# Locals:
#   n   counts how many quotes
#   p   is the current position in the string
# -------------------------------------------------------------------

sub numquotes { # string
    my $p = -1 ; my $n = 0 ;
    while( ($p = index($_[0] ,"\"" ,$p )) > -1 ){ $n++ ; $p++ }
    return $n ;
}

# -------------------------------------------------------------------
# Subroutine parse_statement
# Inputs:
#  0) a single Fortran statement, possibly with embedded blanks
#  1) a string containing original version of the statement, possibly with newlines
# Returns: 1 if statement was processed, 0 if not
# Globals:
#  outputF, outputC
# Action:
#   if statement starts a procedure, then write to the output file
#      starting a prototype for that procedure (using the $fort_* options),
#      parse the argument list and setup global state to enable looking
#      for types of arguments.
#   if statement ends a procedure, write argument types to output file
#      and clear global state
#   if statement declares variables, add variables to symbol table.
# Assumes:
#   The input statement $_[0] is not empty.
# -------------------------------------------------------------------

sub parse_statement { #statement_buffer ,original_code
    my $retval = 1 ;

    # extract the statment label and save the rest of the statment
    $_[0] =~ m/^([ \t]*[0-9]*[ \t]*)/ ;
    $label = substr($1,0,5) ; my $l = $' ;

    # remove all whitespace and convert to lowercase 
    $l =~ s/[ \t]//g ; $l =~ tr/A-Z/a-z/ ;

    # look for procedure declaration statements and ChF macros

    #[NOTE: must parse chf_private before parsing subroutine|function.]

    if ( $l =~ /^chf_private/ ) {
        # Set the global flag, remove the macro from the original code
        # and print the rest of the code without parsing it.
        #[NOTE: the procedure declaration may be on this line or the next.]
        $ChF_private = 1 ;
        $_[1] =~ s/chf_private[ \t]*//i ;
        $outputF and print  FOUT $_[1] ;

    } elsif ( $l =~ /^subroutine/ ) {
        # possibly a subroutine declaration
        #[NOTE: do the complex matching only if the fixed string matches because
        #       it is faster this way since most lines wont match.]
        if ( ! $ChF_private and ( $l =~ /^subroutine([^(]+)\((.*)\)$/ ) ) {
            $name = $1 ; $NAME = uc( $name ) ;
            &start_C_prototype( "void" ) ;
            &start_F_procedure( "subroutine" ) ;
            &parse_arglist( $2 ) ;
            $outputF and &print_F_arglist() ;
        } else {
            # private subroutine or not a subroutine statement
            $outputF and print  FOUT $_[1] ;
        }

    } elsif ( $l =~ /function/ ) {
        # possibly a function declaration statement
        #[NOTE: do the complex matching only if the fixed string matches because
        #       it is faster this way since most lines wont match.]
        if( ! $ChF_private ) {
            if ( $l =~ /^(real_t|integer)function([^\(]+)\(([^=]*)\)$/ ) {
                $name = $2 ; $NAME = uc( $name ) ;
                &start_C_prototype( $Cfunc_types{$1} ) ;
                &start_F_procedure( $1." function" ) ;
                &parse_arglist( $3 ) ;
                $outputF and &print_F_arglist() ;
            } else {
                # check for function types that aren't allowed and print lines
                if( $l =~ /^function([^\(]+)\(([^=]*)\)$/ ) {
                    die "chfpp: error: datatype of function must be declared in `FUNCTION $1' statement." ;
                }
                if( $l =~ /^(real|integer|doubleprecision|complex|doublecomplex)(\*[0-9]+)?function([^\(]+)\(([^=])\)$/ or $l = /^(character)(.*)function([^\(]+)\([^=]*\)$/ ) {
                    die "chfpp: error: non-private [$1$2] function at line $linenum is not allowed" ;
                }
                $outputF and print  FOUT $_[1] ;
            }
        } else {
            # private function or not a function statement at all
            $outputF and print  FOUT $_[1] ;
        }

    } elsif ( $l =~ /^end$/ ) { #only F77-style END statement is supported, not F90.
        ! $ChF_private and &finish_C_prototype( ) ;
        &finish_F_procedure( ) ;
        $outputF and print  FOUT $_[1],"\n" ;
        $ChF_private = 0 ;

    } elsif ( $l =~ /^(real_t|integer)/ ) {  #only these types are allowed in C++ interfaces
        # possible declaration statement
        ! $ChF_private and &parse_decl( $l ) ;
        $outputF and print  FOUT $_[1] ;

    } elsif ( $l =~ /^chf_box\(([a-z][a-z0-9_]*)\)/ ) { #CHF_BOX outside of arglist
        # declare a local `box' variable
        $' ne "" and print STDERR "chfpp: warning: line ${linenum}: extraneous text after CHF_BOX [$'] will be ignored.\n" ;
        $outputF and &execute_chf_box( $1 ) ;

    } elsif ( $outputF and $l =~ /^chf_multido\((.*)\)/ ) {
        # This generates Fortran code for multiple do loops, depending
        # on dimension; it has no effect on C++ code.
        &execute_chf_multido( $1 ) ;
        $' ne "" and print STDERR "chfpp: warning: line ${linenum}: extraneous text after CHF_MULTIDO [$'] will be ignored.\n" ;
        $ChF_multido++ ;

    } elsif ( $outputF and $l =~ /^chf_enddo$/ ) {
        # make sure a preceeding CHF_MULTIDO has been processed.
        $ChF_multido || die "chfpp: error: line ${linenum}: CHF_ENDDO with no matching CHF_MULTIDO.\n" ;
        # This generates Fortran code; it has no effect on C++ code.
        &execute_chf_enddo ;
        $ChF_multido-- ;

### -- This code was added in v1.10 and removed in 1.13.  Leave it here
###    just in case it can be used later. <dbs>
###
###    } elsif ( $Cray and $outputF and $_[0] =~ /[^a-zA-Z0-9_]D_TERM\(/ ) {
###        # Found a D_TERM() macro. Cray cpp is `unconventional'
###        # in that it doesn't preserve line breaks in macro arguments,
###        # so we add the Fortran statement separator (";") to D_TERM()
###        # arguments that span lines.  The modified lines are printed.
###        # This may read additional lines from <> because the lines
###        # don't look like a single Fortran statement.
###
###        # NOTE: this really should modify the source and then let it
###        #       be parsed again for any other macros.
###
###        $outputF and &execute_dterm( $_[1] ) ;


### -- This was added then removed in v1.4.  Leave it commented in case
###    the code can be reused later <dbs>
###
###    } elsif( $l =~ /chf_ixd\(/ ) {
###        # CHF_IXD( indexvar ,dimension )
###        # This is the CHF_MULTIDO index-delta operator.
###        # It returns 1 if the dimension argument (2nd) is equal to the
###        # dimension represented by the index variable argument (1st)
###        # This generates Fortran code; it has no effect on C++ code.
###        # There may be more than one command in the statement so repeat
###        # parsing until the statement can be output.
###        # make sure a preceeding CHF_MULTIDO has been processed.
###        $ChF_multido or die "chfpp: error: line ${linenum}: CHF_IXD macro used outside of a CHF_MULTIDO.\n" ;
###        if ( $outputF ) {
###            ( $l ,$_[1] ) = &execute_chf_ixd( $l ,$_[1] ) ;
###            $debugD and print "DBG: after exec_chf_ixd [$l]\n" ;
###            # recurse on the result so that any other commands in the statement
###            # will get parsed.
###            $retval = &parse_statement( $l ,$_[1] ) ;
###        }

    } else {
        # ordinary Fortran statement that doesn't require parsing
        $retval = 0 ;
        $outputF and print  FOUT $_[1] ;
    }
    return $retval ;
}

# -------------------------------------------------------------------
# Subroutine: stripcomment
# Inputs: none
# Outputs: $ibuf with trailing comment removed
# Globals: $ibuf
# Locals:
# Actions:
#   look through the string for matching quotes and the first occurrence
#   of "!" that isnt in a string and take it as the comment start
#   and strip the comment.  
# -------------------------------------------------------------------

sub stripcomment {
    my $e = index($ibuf,$F_comment) ;
    if ( $e < 0 ) { return $ibuf ; } #no comment

    my $q = index($ibuf,"'") ;
    if ( $q < 0 or $q > $e ) { return substr($ibuf,0,$e) ; }  #comment before 1st quote

    # comment after open quote, look for close quote
    while( ($q = index($ibuf ,"'" ,$q+1 )) > -1 ) {
        if( $e < $q ) {
            # exclamation point inside string, look for next one
            $e = index($ibuf ,$F_comment ,$q+1) ;
            # done if no more
            if ( $e < 0 ) { return $ibuf ; }
        }
        # look for next open quote
        $q = index($ibuf,"'",$q+1) ;
        # done if open quote after exclamation
        if( $q < 0 or $q > $e ) { return substr($ibuf,0,$e) ; }
    }
    die "Error: chfpp: unmatched quote at line $linenum\n" ;
}


# -------------------------------------------------------------------
# Subroutine: finish_C_file
# Inputs: none
# Returns: nothing
# Globals: none
# Locals: none
# Action:
#   Write stuff at the end of the C++ prototype output file
# -------------------------------------------------------------------

sub finish_C_file {
    print O "\n}\n" ;
    print O "#endif\n" ;
}


################################################################
#
#  Internal subroutines
#
################################################################


# -------------------------------------------------------------------
# Subroutine: start_C_prototype
# Inputs: C_function_type
# Returns: nothing
# Globals: O, ChF_private, Cindent, outputC, C_symbol_table, name, NAME
# Actions:
#  If this function is not private, prints the start of a prototype
#  to the C++ output file.  Initializes the symbol table for this function
#  and sets the Cindent string so arguments on
#  subsequent lines will be indented consistently.
# -------------------------------------------------------------------

sub start_C_prototype { # C_function_type
    if ( $ChF_private ) { return ; }
    if ( $outputC ) {
        print O "\n" ;
        print O "// Prototype for Fortran procedure $NAME ...\n//\n" ;
        print O "#define FORT_$NAME FORTRAN_NAME( $NAME ,$name )\n" ;
        print O "$_[0]\nFORT_$NAME(" ;
    }
    %C_symbol_table = () ; 
    $Cindent = " " x ( length( $NAME ) +6 ) ;
    return ;
}

# -------------------------------------------------------------------
# Subroutine: start_F_procedure
# Inputs: F_procedure_type
# Returns: nothing
# Globals: label, Findent, outputF, F_symbol_table, arg_list, name, NAME
# Actions:
#   Prints the start of a Fortran procedure to std_out.  Initializes the
#   symbol table for this procedure.  Sets the indentation string.
# Assumes:
#   length($label)==5
# -------------------------------------------------------------------

sub start_F_procedure { # F_procedure_type
    $outputF and print  FOUT "$label " . uc( $_[0] ) . " $NAME(" ;
    @arg_list = () ;
    %F_symbol_table = () ; 
    %name_list = () ;
    @comment = () ;
    $Findent = " " x ( length( $_[0] ) + length( $NAME ) +2 ) ;
    return ;
}


# -------------------------------------------------------------------
# Subroutine: parse_arglist
# Inputs: string of comma-separated procedure arguments, possibly
#         containing ChF_ commands
# Returns: nothing
# Modifies: @arg_list 
# Action:
#  For each argument, reads up to the next comma (or end of string),
#  checks for ChF_ command and adds the argument to the arg list
#  or executes the command.  Removes each argument after it is parsed.
#  Repeats until the string is empty.
# Assumes:
#  input has no spaces, input is all lowercase, @arg_list and 
#  %[CF]_symbol_table are empty.
# -------------------------------------------------------------------

sub parse_arglist { # arg_list_string
    # if arglist is empty
    if ( $_[0] ) {
        my $l = $_[0] . "," ;  #append a comma so the while loop will get the last arg
#debug
        $debugD and print "arglist|$l\n";

        my $cmd = "" ;
        # get next command or arg
        while ( $l ) {
            if ( $cmd = &is_chf_command( $l ) ) {
                #NOTE: $cmd is the command name excluding the `chf_' prefix
#debug
                $debugD and print "chf_cmd|$cmd|$l\n";
                $l = &execute_chf_command( $cmd ,$l ) ;
            } else {
                # the argument name is everything up to the next comma
                $l =~ /^([^,]+),/ ;
                $arg_list[$#arg_list+1] = $1 ;
                # We don't need to know the Fortran type of this arg
                $F_symbol_table{$1} = "" ;
                # take this arg off the front of the list
                $l = $' ;
#debug
                $debugD and print "non chf_cmd|$1\n";

            }
        }
    }
}
# -------------------------------------------------------------------
# Subroutine: parse_decl
# Inputs: Fortran_statement
# Returns: nothing
# Globals: F_symbol_table, C_symbol_table
# Action:
#  Determine if the statement has proper syntax for a declaration,
#  extract the variable names and add them to the Fortran and C symbol
#  tables.
# -------------------------------------------------------------------

sub parse_decl { # statement
    $_[0] =~ m/(real_t|integer)/ ;
    my $Ftype = $1 ; my $args = $' . "," ;  #append "," as sentinel
#debug
    $debugD and print "decl|$Ftype|$args\n" ;

    # <type>*<length> form is not allowed
    if( $args =~ /^\*/ ) { return ; }
    # extract variable names from arg list
    while( $args ) {
        $args =~ /^([^,]*),/ ;
        my $var = $1 ; $args = $' ;
        # only scalars variables can be passed without CHF macros,
        # so strip any trailing array declaration
        $var =~ s/\(.*$// ;
        # push this variable into the symbol table
        $F_symbol_table{$var} = "$Ftype $var" ;
        $C_symbol_table{$var} = "$C_types{$Ftype}( $var )" ;
    }    
}

# -------------------------------------------------------------------
# Subroutine: is_chf_command
# Inputs: string 
# Returns: command name (excluding leading `chf_' if the string starts with one
#      of the Chombo Fortran command names, else 0.
# Modifies: nothing
# Action:
#  search the string starting at the beginning for any of the known commands
# Note:
#  this doesn't handle the attribute command (CHF_PRIVATE) or the access commands
#  (CHF_{L,U}BOUND,NCOMP).  Attribute commands are handled at the top level,
#  access commands are handled by cpp macros.
# -------------------------------------------------------------------

sub is_chf_command { # string ==> Boolean
    if( $_[0] =~ /^chf_((box|(const_)?((int|real)(1d)?|f[ir]a1?|v[ir])))\(/ ) {
#debug
        if( $debugD ) {
            print "is_chf_cmd: parse results 1-6: " ;
            print "|" ; defined($1) and print $1 ;
            print "|" ; defined($2) and print $2 ;
            print "|" ; defined($3) and print $3 ;
            print "|" ; defined($4) and print $4 ;
            print "|" ; defined($5) and print $5 ;
            print "|" ; defined($6) and print $6 ;
            print "|\n" ;
        }
        return $1 ;
    } else {
        return 0 ;
    }
}

# -------------------------------------------------------------------
# Subroutine: execute_chf_command
# Inputs: [0] command name without `chf_', [1] entire string starting with `chf_'
# Returns: remainder of string after command
# Modifies: @arg_list, @comment, %F_symbol_table, %C_symbol_table
# Action:
#  Executes the specified command on the arguments in the string
#  and removes the command name and command args from the string.
#  The command will add arguments to the global argument list and
#  possibly modify the symbol tables.
# Assumes:
#  _[0] is the same as the leading substring of _[1] and is a known ChF command.
#  _[1] has a trailing ",".
# -------------------------------------------------------------------

sub execute_chf_command { # command_name, command_string
    my $cmd = $_[0] ;
    # extract the command argument and reset the rest of the string
    #NOTE: the following should be made more precise
    #NOTE: parentheses are not allowed in ChF command arguments
#debug
    my ( $prefix, $list, $rest ) = &get_list( $_[1] ,"(" ,")" ) ;
    $debugD and print "exec_cmd: _1=$_[1]\n" ;
    $debugD and print "exec_cmd: prefix,list,rest = $prefix|$list|$rest|\n" ;
    $list =~ /\(([^,]*)(,.*)?\)/ ;    #strip delimiters
    $arg = $1 ;
    if( $2 ) {
        $arg2 = $2 ; $arg2 =~ s/^,// ;    #arg2 is all the remaining macro arguments after the comma
    } else {
        $arg2 = "" ;
    }
    $arg_list[$#arg_list+1] = $arg ;
    $_[1] = $rest ; $_[1] =~ s/^,// ;  #skip over the "," if any
#debug
    $debugD and print "exec_cmd: arg1,2=$arg|$arg2\n" ;
    $debugD and print "exec_cmd: remainder=$_[1]\n" ;

    # save the chf command name as the comment
    $comment[$#arg_list] = "ChF_" . $cmd . "()";

    # determine if this is a `const' command
    my $const = "" ;
    if ( $cmd =~ /^const_/ ) { $const = "const " ; $cmd = $' ; }
    my $CONST = uc( chop($const) and "_$const" )  ; 

    # determine what command this is and execute it
    if ( $cmd =~ /^int(1d)?$/ ) {
        # `CHF_INT' or `CHF_CONST_INT' or CHF_INT1D or CHF_CONST_INT1D
#debug
        $debugD and print "exec_chf: int*: cmd=$cmd, arg=$arg\n";
        if ( $1 ) {
            $arg_list[$#arg_list+1] = "n${arg}comp" ;
            $F_symbol_table{$arg} = "integer $arg(0:$arg_list[$#arg_list]-1)" ;
            $C_symbol_table{$arg} = "CHFp${CONST}_INT1D( $arg ,$arg_list[$#arg_list] )" ;
            $F_symbol_table{$arg_list[$#arg_list]} = "integer $arg_list[$#arg_list]" ;
            $C_symbol_table{$arg_list[$#arg_list]} = " " ;
        } else {
            $F_symbol_table{$arg} = "integer $arg" ;
            $C_symbol_table{$arg} = "CHFp${CONST}_INT( $arg )" ;
        }
    } elsif ( $cmd =~ /^real(1d)?$/ ) {
        # `CHF_REAL' or `CHF_CONST_REAL' or `CHF_REAL1D' or `CHF_CONST_REAL1D'
        if ( $1 ) {
            $arg_list[$#arg_list+1] = "n${arg}comp" ;
            $F_symbol_table{$arg} = "REAL_T $arg(0:$arg_list[$#arg_list]-1)" ;
            $C_symbol_table{$arg} = "CHFp${CONST}_REAL1D( $arg ,$arg_list[$#arg_list] )" ;
            $F_symbol_table{$arg_list[$#arg_list]} = "integer $arg_list[$#arg_list]" ;
            $C_symbol_table{$arg_list[$#arg_list]} = " " ;
        } else {
            $F_symbol_table{$arg} = "REAL_T $arg" ;
            $C_symbol_table{$arg} = "CHFp${CONST}_REAL( $arg )" ;
        }

    } elsif ( $cmd =~ /^box$/ ) {
        # `CHF_BOX'
        # the box variable name isn't used, only the names of the
        # created vars, so take the box name off the argument list
        $#arg_list-- ;
        # Box has two D_DECL macros
        for $lohi ( "lo","hi" ) {
            $arg_list[$#arg_list+1] = "CHF_DDECL_BEGIN" ;
##            my $comma = "" ;
            for $i ( 0..$MAXDIMm1 ) {
                $arg_list[$#arg_list] .= "\[". "i" . $arg . $lohi .$i. "\]";
##                $comma = "," ;
            }
            $arg_list[$#arg_list] .= "CHF_DDECL_END" ; 

            $F_symbol_table{$arg_list[$#arg_list]} = "integer " . $arg_list[$#arg_list] ;
        }
        # only one argument is needed in the C++ prototype, so set the second to blank
        # so it wont be printed out 
        $C_symbol_table{$arg_list[$#arg_list-1]} = "CHFp_BOX( $arg )" ;
        $C_symbol_table{$arg_list[$#arg_list]} = " " ;
        # add this arg to the list of boxes and fab names
        $name_list{$arg} = 1 ;

    } elsif ( $cmd =~ /^f([ir])a(1?)$/ ) {
        # `CHF_FIA'  or `CHF_FRA'  or `CHF_CONST_FIA'  or `CHF_CONST_FRA' or
        # `CHF_FIA1' or `CHF_FRA1' or `CHF_CONST_FIA1' or `CHF_CONST_FRA1'
        # Build the C++ prototype macro name from the Fortran macro name
        my $CMD = uc( $cmd ) ;
        $C_symbol_table{$arg_list[$#arg_list]} = "CHFp${CONST}_$CMD( $arg )" ;

        # Store the type of the array argument, then add the extra args
        if ( $1 eq "i" ) {
            $F_symbol_table{$arg_list[$#arg_list]} = "integer $arg(" ;
        } else {
            $F_symbol_table{$arg_list[$#arg_list]} = "REAL_T $arg(" ;
        }
        # Create additional arguments for the array bounds and the number of components
        # and add them to the array declaration
        for $lohi ( "lo","hi" ) {
            $arg_list[$#arg_list+1] = "CHF_DDECL_BEGIN" ;
##            my $comma = "" ;

            for $i ( 0..$MAXDIMm1 ) {
                $arg_list[$#arg_list] .= "\[" . "i" . $arg . $lohi .$i ."\]";
##                $comma = "," ;
            }
            $arg_list[$#arg_list] .= "CHF_DDECL_END" ;

            #[NOTE: The Fortran declaration has the form `integer D_DECL(a,b,c)' while
            #       the C++ declaration has the form `D_DECL(int a,int b,int c)'.]
            $F_symbol_table{$arg_list[$#arg_list]} = "integer " . $arg_list[$#arg_list] ;

            # The C++ prototype doesn't need the extra args so set them to blank
            $C_symbol_table{$arg_list[$#arg_list]} = " " ;
        }
        # add bounds to the Fortran array declaration
        #[NOTE: the order of things in the D_DECL is different from the declarations of
        #       the bounds variables themselves.]
        $F_symbol_table{$arg} .= "CHF_DDECL_BEGIN" ;
##        $comma = "" ;
        for $i ( 0..$MAXDIMm1 ) {
            $F_symbol_table{$arg} .= "\[" . "i" . $arg . "lo" . $i . ":i" .$arg . "hi" . $i. "\]";
##            $comma = "," ;
        }
        $F_symbol_table{$arg} .= "CHF_DDECL_END" ;
        # add number_of_components variable to arg list and array declaration
        if( ! $2 ) {
            $arg_list[$#arg_list+1] = "n" . $arg . "comp" ;
            $F_symbol_table{$arg_list[$#arg_list]} = "integer " . $arg_list[$#arg_list] ;
            $F_symbol_table{$arg} .= ",0:$arg_list[$#arg_list]-1" ;
            $C_symbol_table{$arg_list[$#arg_list]} = " " ;
        }
        $F_symbol_table{$arg} .= ")" ;
        # add this arg to the list of boxes and fab names
        $name_list{$arg} = 1 ;

    } elsif( $cmd =~ /^v([ir])$/ ) {
        # `CHF_VI' or `CHF_VR' or `CHF_CONST_VI' or `CHF_CONST_VR'
        # Build the C++ prototype macro name from the Fortran macro name
        my $CMD = uc( $cmd ) ;
        $C_symbol_table{$arg_list[$#arg_list]} = "CHFp${CONST}_$CMD( $arg )" ;

        # Store the type of the array argument, then add the extra length arg
        if ( $1 eq "i" ) {
            $F_symbol_table{$arg_list[$#arg_list]} = "integer $arg(" ;
        } else {
            $F_symbol_table{$arg_list[$#arg_list]} = "REAL_T $arg(" ;
        }
        $arg_list[$#arg_list+1] = "n" . $arg . "comp" ;
        $F_symbol_table{$arg_list[$#arg_list]} = "integer " . $arg_list[$#arg_list] ;
        $F_symbol_table{$arg} .= "0:$arg_list[$#arg_list]-1" ;
        $C_symbol_table{$arg_list[$#arg_list]} = " " ;

        $F_symbol_table{$arg} .= ")" ;
        # add this arg to the list of boxes and fab names
        $name_list{$arg} = 1 ;


    } else {
        warn "Unknown ChF command: $cmd\n" ;
    }
    return $_[1] ;
}


# -------------------------------------------------------------------
# Subroutine: execute_chf_box
# Inputs: name of box
# Returns: nothing
# Globals: MAXDIMm1,name_list
# Actions:
#  Prints code to declare box variables with dimension-independence
#  and adds its argument to the list of box/array names
# Assumes:
#  Fortran code output ($outputF) is enabled.
# -------------------------------------------------------------------

sub execute_chf_box { # name of variable to declare
    # syntax: CHF_BOX( name )
    # Generate declarations of integer variables for low and high corners
    # of the box, with dimension independence
    $debugD and print "chf_box: arg is $_[0]\n" ;
    for $lohi ( "hi","lo" ) {
        print  FOUT "      integer CHF_DDECL_BEGIN" ;
##        my $comma = "" ;
        for $i ( 0..$MAXDIMm1 ) {
            print  FOUT "\[" ,"i",$_[0],$lohi,$i."\]" ;
##            $comma = "," ;
        }
        print  FOUT "CHF_DDECL_END \n" ; 
    }
    # add this arg to the list of boxes and fab names
    $name_list{$_[0]} = 1 ;
}


# -------------------------------------------------------------------
# Subroutine: execute_chf_multido
# Inputs: arglist to chf_multido macro
# Returns: nothing
# Globals: MAXDIM, name_list, 
# Actions:
#  Prints code for DO loops with dimension-independent #ifdef's
# Assumes:
#  Fortran code output ($outputF) is enabled.
# -------------------------------------------------------------------

sub execute_chf_multido { # arglist
    # syntax: CHF_MULTIDO( box-or-array-arg ,indexvar1,...,indexvar$MAXDIM )
    # Check for the box-or-array-arg in the symbol table.
    # Generate DO loops with appropriate #ifdef's to get dimension-independence.
    # First, find the arguments of the MULTIDO and initialize the lookup table that the
    # CHF_IX and IXD macros will need.
    $_[0] =~ /^([^,]*),(.*)$/ ;  #split the arglist into first and rest
    my $arg = $1 ;
    my @mdo_indexvarname = split /,/ ,$2 ;
    my %mdo_indexvar = () ;
    # check that enough indexvars were given
    $debugD and print "DBG: multido #indexvars ",scalar(@mdo_indexvarname),@mdo_indexvarname,"\n";
    scalar(@mdo_indexvarname) >= $MAXDIM || die "chfpp: error: line ${linenum}: insufficient arguments to CHF_MULTIDO\n" ;
    # check that the box or fab arg is in the symbol table
    defined $name_list{$arg} || die "chfpp: error: line ${linenum}: undeclared Box or Array [$arg] used in CHF_MULTIDO.\n" ;
    # save the index variable names for each dimension and generate code
    for $i  ( reverse 0..$MAXDIMm1 ) {
        $mdo_indexvar{$mdo_indexvarname[$i]} = $i ;
        print  FOUT "#if (CH_SPACEDIM >= ",$i+1,")\n" ;
        print  FOUT "      do $mdo_indexvarname[$i] = CHF_LBOUND( $arg ,$i ),CHF_UBOUND( $arg ,$i )\n" ;
        print  FOUT "#endif\n" ;
    }
    if( $debugD ) {
        print "DBG: multido index vars: " ;
        foreach $i (%mdo_indexvar) { print "$i " ; }
        print "\n" ;
    }
}


# -------------------------------------------------------------------
# Subroutine: execute_chf_enddo
# Inputs: none
# Returns: nothing
# Globals: MAXDIM
# Actions:
#  Prints code for ENDDO with dimension-independence
# Assumes:
#  A matching CHF_MULTIDO is present in the code before this macro
#  Fortran code output ($outputF) is enabled.
# -------------------------------------------------------------------

sub execute_chf_enddo { #
    # syntax: CHF_ENDDO
    for $i  ( 1..$MAXDIM ) {
        print  FOUT "#if (CH_SPACEDIM >= $i)\n" ;
        print  FOUT "      enddo\n" ;
        print  FOUT "#endif\n" ;
    }
}


### -- NOTE: this routine is not used currently.  Keep it in because
###          it might be useful in the future <dbs>
#### -------------------------------------------------------------------
#### Subroutine: execute_dterm
#### Inputs: input_source_lines
#### Returns: nothing
#### Globals: MAXDIM
#### Actions:
####  Prints the input source lines, inserting ";" at the end of each
####  argument to the D_TERM() macro that is followed by a line break.
#### Assumes:
####  Fortran code output ($outputF) is enabled.
####  Running on a system that doesn't have a conventional `cpp'
#### -------------------------------------------------------------------
###
###sub execute_dterm { #input_source_lines
###    # Find the D_TERM commands in the source lines
###    #NOTE: it must be UPPERCASE because cpp wont match it otherwise
###    my $src = $_[0] ,$i=-1 ;
###    chomp $src ;
###    while( ($i = index($src ,"D_TERM(" ,$i)) > -1 ) {
###        # Either the whole macro is in this string, in which case there are
###        # no embedded line breaks, or not.
###        my $macro_args = substr($src ,$i+6) ;
###        my ($p,$l,$s) = get_list($macro_args ,"(",")") ;
###        $debugD and print "exec_dterm: p,l,s = |$p|$l|$s|\n" ;
###        if( $p !~ /\s*/ ) {
###            print "InternalError: execute_dterm missed the start of the arglist.\n" ;
###            print "               text before arglist is [$p]\n" ;
###            exit( -1 ) ;
###        }
###        if( substr($l ,-1) ne ")" ) {
###            # the list wasnt closed, so there must be another line;
###            # add the Fortran90 statement separator and try again
###            $debugD and print "exec_dterm: list not closed.\nibuf = |$ibuf|\n" ;
###            
###            $src .= ";\n" . $ibuf ; chomp $src ;
###            $ibuf = <F> ;
###            $debugD and print "exec_dterm: new source string is |$src|\n" ;
###        } else {
###            # the arg list is all on this line, so skip it and
###            # look for another occurrence of D_TERM()
###            $i += 6 + length( $l ) ;
###        }
###    }
###    # finished processing D_TERM() macros, so print the whole string
###    print $src,"\n" ;
###}


### -- NOTE: this routine is not used currently.  Keep it in because
###          it might be useful in the future <dbs>
#### -------------------------------------------------------------------
#### Subroutine: execute_chf_ixd
#### Inputs: statement ,original_code
#### Returns: new_statement ,new_original_code
#### Globals: none
#### Actions:
####  find the chf_ixd macro in the statement, expand it appropriately,
####   and replace the macro with its expansion in both the statement and
####   the original code, and return both.
####  To expand the macro, determine which dimension the index variable
####  corresponds to and generate the appropriate reference to the array
####  of "index delta" values.
#### Assumes:
####  A `match' operation "chf_ixd(" has just been performed.
####  Fortran code output ($outputF) is enabled.
####  A CHF_MULTIDO has been parsed without a closing CHF_ENDDO.
#### -------------------------------------------------------------------
###
###sub execute_chf_ixd { # statement, code => statement, code
###    # extract command arglist from string after the command
###    # [NOTE: the $prefix result should be empty.]
###    # [NOTE: get_list() will work even though $' doesnt start with an open-paren
###    #        (it was gobbled up by the pattern match).]
###    my ( $prefix, $list, $rest ) = &get_list( $' ,"(" ,")" ) ; 
###    my $cmd = quotemeta $& . $list ;
###    # remove the trailing close paren [note: there is no open paren]
###    # and extract the macro arguments
###    $list =~ s/\)$// ;
###    my @args = split /,/ ,$list ;
###
###    # check syntax
###    # ( indexvar1 ,... ,indexvarMAXDIM ,direction [,delta] )
###    # NOTE: direction and delta could be parenthesized expressions 
###    #       but indexvar* must all be variable names
###    scalar(@args) < $MAXDIM+1 || die "chfpp: error: line ${linenum}: incorrect number of args to CHF_IDX."
###                             ," Must be ",$MAXDIM+1,", not " ,scalar(@args) ,".\n" ;
###    for ( my $i=0 ; $i < $MAXDIM ; $i++ ) {
###        exists $mdo_indexvar{$args[$i]} || die "chfpp: error: line ${linenum}: syntax error in CHF_IDX.  "
###                                         ,"`$args[$i]' is not a CHF_MULTIDO index variable.\n" ;
###    }
###
###    # patch the last 2 args back together in case of parenthesized expressions
###    for ( my $i=$MAXDIM ; $i < scalar(@args) ; $i++ ) {
###    }
###
###    # generate code to reference the array of delta values
###    
###    my $expansion = "CHFIXDELTA($mdo_indexvar{$args[0]},$args[1])" ;
###
###    # Substitute expansion into both argument variables:
###    # First modify the parsed statement because its easy. 
###    $_[0] =~ s/$cmd/$expansion/ ;
###
###    # Substituting into the original code is a little trickier because there may be
###    # whitespaces and continuation lines..  We use a brute-force approach:
###    # remove continuations from the code and replace them with a marker (\e);
###    # after every char in the chf_ixd command string add a regexp that matches 
###    #  whitespace and the marker (explicitly merge quoted metachars back together
###    #  before append the regexp)
###    # substitute the macro expansion for the whole regexp (destroying continuations
###    #  in the middle of the command, if any)
###    # replace markers that weren't destroyed
###
###    # put each char in $cmd into a separate array element
###    my @cmdarray = split //,$cmd ;
###    # replace continuations in original source code with the \e marker
###    $_[1] =~ s/\n     [^0]/\e/g ;
###    # add the regexp after each character (splicing quoted meta-chars back together first)
###    for ( my $i=0 ; $i<@cmdarray-2 ; $i++ ) { #note: -2 because we dont need to match after the
###                                           #      next-to-last element, which is a metachar 
###        if ( $cmdarray[$i] eq "\\" ) { splice(@cmdarray ,$i,2 ,$cmdarray[$i].$cmdarray[$i+1] );}
###        $cmdarray[$i] .= "[ \\t\\e]*" ; #\e+[ \\t]*#(\n(     [^0]|\t&))+[ \t]*
###    }
###    $cmd = join("",@cmdarray) ;
###    # replace the original code for the command with its expansion and replace markers
###    $_[1] =~ s/$cmd/$expansion/i ;
###    $_[1] =~ s/\e/\n     &/g ;
###
###    # done
###    return ( $_[0] ,$_[1] ) ;
###}


# -------------------------------------------------------------------
# Subroutine: print_F_arglist
# Inputs: none
# Returns: nothing
# Globals: arg_list, F_symbol_table
# Actions:
#  Print the argument list, one argument per line.  Then print the
#  variable declarations, in reverse order.
# Notes:
#  The arglist declarations are printed in reverse order so that arguments
#  that are used as bounds in array declarations are declared before the
#  array that uses them is declared.  This is required by the Fortran
#  standard.
# -------------------------------------------------------------------

sub print_F_arglist { #
    my $comma = " " ; my $indent = "" ; 
    for( my $i=0 ; $i <= $#arg_list ; $i++ ) {
        print  FOUT "$indent$comma$arg_list[$i]\n" ;
        $comma = "," ; $indent = "     &" . $Findent ;
    }
    print FOUT "$indent)\n" ;
    print FOUT "      implicit none\n";
    print FOUT "\n" ;
#    print FOUT "$F_comment This array is used by the ChomboFortran macros.\n" ;
#    print FOUT "$F_comment It may generate an \"unused variable\" compiler warning.  Ignore it.\n" ;
    print FOUT "      integer CHF_IJK(0:2,0:2)\n" ;
    print FOUT "      data    CHF_IJK/ 1,0,0 ,0,1,0 ,0,0,1 /\n" ;
    print FOUT "\n" ;
    for( my $i=$#arg_list ; $i >= 0 ; $i-- ) {
        if( $F_symbol_table{$arg_list[$i]} ) {
            print  FOUT "      $F_symbol_table{$arg_list[$i]}";
#            print  FOUT "      $F_symbol_table{$arg_list[$i]}" and
#                $comment[$i] and print  FOUT "     $F_comment $comment[$i]" ;
            print  FOUT "\n" ;
        }
    }
}

# -------------------------------------------------------------------
# Subroutine: print_C_arglist
# Inputs: none
# Returns: nothing
# Globals: arg_list, C_symbol_table
# Actions:
#  Print the argument list, one argument per line.  All arguments except
#  the first are indented.  If the arglist is empty, print "void".
# -------------------------------------------------------------------

sub print_C_arglist { #
    if ( $#arg_list < 0 ) {
        print O " void\n" ;
    } else {
        my $comma = " " ; my $indent = "" ; 
        for( my $i=0 ; $i <= $#arg_list ; $i++ ) {
            $debugD and print "arg $i = $arg_list[$i], C_sym = $C_symbol_table{$arg_list[$i]}.\n" ;
            $C_symbol_table{$arg_list[$i]} or die "chfpp: error: no datatype defined for argument $arg_list[$i]\n" ;
            $C_symbol_table{$arg_list[$i]} =~ /^ $/ or print O "$indent$comma$C_symbol_table{$arg_list[$i]}\n" ;
            $comma = "," ; $indent = $Cindent ;
        }
    }
}

# -------------------------------------------------------------------
# Subroutines: finish_C_prototype, finish_F_procedure
# Inputs: none?
# Actions:
#  
# Notes:
#  We cant finish the C prototype until the whole Fortran procedure
#  has been read because we need the data types of the arguments that
#  arent declared with ChF macros.
#  Theres nothing to do in the Fortran output when the procedure
#  finishes since the top-level parser prints out the END statement.
# -------------------------------------------------------------------
sub finish_C_prototype {
    $outputC and &print_C_arglist() ; 
    $outputC and print O $Cindent,") ;\n" ;
}
sub finish_F_procedure {
    $ChF_multido and die "chfpp: error: line ${linenum}: CHF_MULTIDO without corresponding CHF_ENDDO.\n"
#debug
#    if( ! $ChF_private ) {
#    my $i ;
#    for( $i=0 ; $i <= $#arg_list ; $i++ ) {
#        print "arg $i|$arg_list[$i]|$F_symbol_table{$arg_list[$i]}\n" ;
#    }}
}


# -------------------------------------------------------------------
# Subroutine: get_list
# Inputs: string, open_delimiter, close_delimiter
# Returns: prefix_substring, delimited_substring, rest_of_string
# Globals: none
# Actions:
#  extract the first list (as defined by the delimiters) in the input
#  string and return everything before the list, the list and the rest
#  of the input after the close_delimiter,
#  allowing for embedded sub-lists
# Notes:
#  This needs to allow for escaped open and close delimiters, either
#  by direct escape (ie <escape_char><delim>) or as part of a quoted
#  string (ie <open_quote>...<delim>...<close_quote>).  Handling the
#  direct escape requires handling escaping the escape.  Similarly,
#  handling the quoted string requires handling escaped quotes.
#  Possible interface would have the escaped delim represented by a
#  reg exp (which should be specified to not match an escaped escape),
#  and the quote chars represented by a regexp that matches the whole
#  string (which should be specified to skip escaped quotes).
# -------------------------------------------------------------------

sub get_list { #string ,open_delim ,close_delim
    my $cnt = 0 ; my $len = length( $_[0] ) ; my $start = $len ;
#debug
    $debugD and print "start: input,length $_[0]|$len\n" ;
    my $i = 0 ;
    for( $i = 0 ; $i < $len ; $i++ ) {
        my $c = substr( $_[0] ,$i ,1 ) ;
        if( $c eq $_[1] ) {
            # set start to mark the first open_delim
            if( $cnt++ == 0 ) { $start = $i ; }
        } elsif ( $c eq $_[2] ) {
            # done when cnt gets back to 0.
            #NOTE: too many close delims if $cnt<0
            if( --$cnt <= 0 ){ last ; }
        }
    }
#debug
    $debugD and print "done: start,end = $start | $i\n" ;
    if( $start > $i ) { $start = 0 }
    return ( substr( $_[0] ,0 ,$start )             #prefix
            ,substr( $_[0] ,$start ,$i-$start+1 )   #list
            ,$i+1 <= length($_[0]) ? substr( $_[0] ,$i+1 ) : "" ); #rest or null_string
}
    

# -------------------------------------------------------------------
# Take a string of variable declarations and returns a 
# list of the variable names.  Has to skip over all 
# parenthesized lists.  This is brute-force, but
# hopefully isn't used often
# -------------------------------------------------------------------

sub split_syms { #string => list_of_names
    my $l = $_[0] ;
    # as long as there is still a pair of parens in the string,
    # remove the whole parenthesized expression.  This removes
    # nexted parens from the inside out.
    while( $l =~ /\([^()]*\)/ ) {
        $l =~ s/\([^()]*\)//g ;
    }
    # the only thing left should be comma-separated symbol names
    return split /,/ ,$l ;
}


# -------------------------------------------------------------------
# -------------------------------------------------------------------

sub warn_star {
    print STDERR "chfpp: Warning: file $basename.f, line $linenum: ignoring [$1] notation in declaration.\n" ;
    print STDERR $obuf ;
}

# NOTE 1: 
#  removing `!.*' is not completely correct, since the `!' could be
#  in a string, but if there's a string in the statement it isn't
#  what we're looking for anyway.

}
###i have no idea why this is here.
###the perl cookbook book told me to put it there.
###really.
1;
