/*
** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
**  
** For further information about this software, contact:
** 
*/

#ifndef CH_ARENA
#define CH_ARENA

//
// $Id: Arena.H,v 1.1.1.1 2000/09/05 23:55:55 graves Exp $
//

#include <cstddef>


//
//@Man:
//@Memo: A Virtual Base Class for Dynamic Memory Management
/*@Doc:

  This is a virtual base class for objects that manage their own dynamic
  memory allocation.  Since it is a virtual base class, you have to derive
  something from it to use it.
*/

class Arena
{
public:
    /*@ManDoc: Allocate a dynamic memory arena of size sz.
               A pointer to this memory should be returned.
    */
    virtual void* alloc (size_t sz) = 0;

    /*@ManDoc: A pure virtual function for deleting the arena pointed
               to by pt.
    */
    virtual void free (void* pt) = 0;

    /*@ManDoc: Given a minimum required arena size of sz bytes, this returns
               the next largest arena size that will hold an integral number
               of objects of the largest of the types void*, long,
               double and function pointer.
    */
    static size_t align (size_t sz);
    typedef void (*FP) ();
protected:
    //
    // Types used by align().
    //


    union Word
    {
        void*  p;
        double d;
        long   l;
        FP     f;
    };
};

//
// Inlines.
//

inline
size_t
Arena::align (size_t s)
{
    size_t x = s + sizeof(Word) - 1;
    x -= x%sizeof(Word);
    return x;
}

//
//@Man:
//@Memo: A Concrete Class for Dynamic Memory Management
/*@Doc:

  This is the simplest dynamic memory management class derived from Arena.

  Makes calls to ::operator new() and ::operator delete().
*/

class BArena
    :
    public Arena
{
public:

    /*@ManDoc: Allocates a dynamic memory arena of size sz.  Returns a
               pointer to this memory.
    */
    virtual void* alloc (size_t sz);
    //
    //@ManDoc: Deletes the arena pointed to by pt.
    //
    virtual void free (void* pt);
};

#include <cstddef>

#include <set>
#include <vector>
using std::set;


//
//@Man:
//@Memo: A Concrete Class for Dynamic Memory Management
/*@Doc:

  This is a coalescing memory manager.  It allocates (possibly) large
  chunks of heap space and apportions it out as requested.  It merges
  together neighboring chunks on each free().
*/

class CArena
    :
    public Arena
{
public:

    /*@ManDoc: Construct a coalescing memory manager.  `hunk\_size' is the
               minimum size of hunks of memory to allocate from the heap.
               If hunk\_size == 0 we use DefaultHunkSize as specified below.
    */
    CArena (size_t hunk_size = 0);
    //
    //@ManDoc: The destructor.
    //
    virtual ~CArena ();
    //
    //@ManDoc: Allocate some memory.
    //
    virtual void* alloc (size_t nbytes);

    /*@ManDoc: Free up allocated memory.  Merge neighboring free memory chunks
               into largest possible chunk.
    */
    virtual void free (void* ap);

    /*@ManDoc: Mirror the C calloc() function.  Returns zero'd memory of
               size nmemb*size.  This is here so that we can implement
               malloc(3) and family.  User's shouldn't use this function.
    */
    void* calloc (size_t nmemb, size_t size);

    /*@ManDoc: Mirror the C realloc() function.  This is here so that we can
               implement malloc(3) and family.  User's shouldn't use this
               function.
    */
    void* realloc (void* ptr, size_t size);
    //
    //@ManDoc: The default memory hunk size to grab from the heap.
    //
    enum { DefaultHunkSize = 1024*1024 };

protected:
    //
    // The nodes in our free list and block list.
    //
    class Node
    {
    public:
        //
        // The default constructor.
        //
        Node ()
            :
            m_block(0), m_size(0) {}
        //
        // Another constructor.
        //
        Node (void* block, size_t size)
            :
            m_block(block), m_size(size) {}
        //
        // The copy constructor.
        //
        Node (const Node& rhs)
            :
            m_block(rhs.m_block), m_size(rhs.m_size) {}
        //
        // The copy assignment constructor.
        //
        Node& operator= (const Node& rhs)
        {
            m_block = rhs.m_block;
            m_size  = rhs.m_size;
            return *this;
        }
        //
        // The "less-than" operator.
        //
        bool operator< (const Node& rhs) const
        {
            return m_block < rhs.m_block;
        }
        //
        // The equality operator. 
        //
        bool operator== (const Node& rhs) const
        {
            return m_block == rhs.m_block;
        }
        //
        // The block address.
        //
        void* block () const { return m_block; }
        //
        // Set block address.
        //
        void block (void* blk) { m_block = blk; }
        //
        // The size of the memory block.
        //
        size_t size () const { return m_size; }
        //
        // Set size.
        //
        void size (size_t sz) { m_size = sz; }

    private:
        //
        // The block of memory we reference.
        //
        void* m_block;
        //
        // The size of the block we represent.
        //
        size_t m_size;
    };
    //
    // The type of our freelist and blocklist.
    // We use a set sorted from lo to hi memory addresses.
    //
    typedef set < Node > NL;
    //
    // The list of blocks allocated via ::operator new().
    //
    std::vector<void*> m_alloc;
    //
    // The free list of allocated but not currently used blocks.
    // Maintained in lo to hi memory sorted order.
    //
    NL m_freelist;
    //
    // The list of busy blocks.
    // A block is either on the freelist or on the blocklist, but not on both.
    //
    NL m_busylist;
    //
    // The minimal size of hunks to request via ::operator new().
    //
    size_t m_hunk;

private:
    //
    // Disallowed.
    //
    CArena (const CArena& rhs);
    CArena& operator= (const CArena& rhs);
};

//
// The Arena used by BaseFab code.
//
extern Arena* The_FAB_Arena;


#endif /*CH_ARENA*/
