/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// BaseFab.H
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.

//CH_COPYRIGHT_NOTICE

#ifndef CH_BASEFAB_H
#define CH_BASEFAB_H

//
// $Id: BaseFab.H,v 1.1.1.1 2000/09/05 23:55:55 graves Exp $
//

#include <cstdlib>
#include <assert.h>
#include "Box.H"
#include "Arena.H"
#include "Interval.H"


//
//@Man:
//@Memo: A Fortran Array-like Container
/*@Doc:

  BaseFab emulates the Fortran array concept.  
  Useful operations can be performed upon
  BaseFab's in C++, and they provide a convenient interface to
  Fortran when it is necessary to retreat into that language.
        
  `BaseFab' is a template class.  Through use of the
  template, a `BaseFab' may be based upon any class.  So far at least,
  most applications have been based upon simple types like `integer's,
  `real's, or `doubleprecision's.  Most applications do not use BaseFab's 
  directly, but utilize specialized classes derived from BaseFab.
        
  It will be easier to use a `BaseFab' if you understand the following
  concepts.  `BaseFab' objects depend on the dimensionality of space
  (indirectly through the DOMAIN `Box' member).  It is
  typical to define the macro `CH_SPACEDIM' to be 1, 2, or 3 to indicate
  the dimension of space.  See the discussion of class `Box' for more
  information.  A `BaseFab' contains a `Box' DOMAIN, which indicates the
  integer indexing space over which the array is defined.  A `BaseFab'
  also has NVAR components.  By components, we mean that for each
  point in the rectangular indexing space, there are NVAR values
  associated with that point.  A Fortran array corresponding to a
  `BaseFab' would have (CH_SPACEDIM+1) dimensions.
        
  By design, the array layout in a `BaseFab' mirrors that of a
  Fortran array.  The first index (x direction for example) varies
  most rapidly, the next index (y direction), if any, varies next
  fastest. The component index varies last, after all the spatial
  indices.
        
  It is sometimes convenient to be able to treat a sub-array within an
  existing `BaseFab' as a `BaseFab' in its own right.  This is often
  referred to as 'aliasing' the `BaseFab'.  Note that when aliasing is 
  used, the BaseFab's domain will not, in general, be the same as the 
  parent BaseFab's domain, nor will the number of components.
  BaseFab is a dimension dependent class, so CH_SPACEDIM must be 
  defined as either 1, 2, or 3 when compiling.

  This is NOT a polymorphic class.

  It does NOT provide a copy constructor or assignment operator.

  T MUST have a default constructor and an assignment operator.
  */

template <class T>
class BaseFab
{
public:

/// {\bf constructors, destructor and defines}

  /*@ManDoc: Constructs an invalid `BaseFab'.  The domain is invalid, the
    number of components is zero, and no actual array memory is
    allocated.  An invalid `BaseFab' must be resize()d
    (see `BaseFab::resize') before use.
    */
  BaseFab ();

///
/**
  Constructs a BaseFab with desired domain and number of components.
*/
  BaseFab (const Box& bx,
		    int        n);

///
/**
  The destructor deletes the array memory.
*/
  virtual ~BaseFab ();

  /*@ManDoc: This function resizes a `BaseFab' so it covers the `Box' B
               with N components.  The default action
               is that under resize()ing, the memory allocated for the
               `BaseFab' only grows and never shrinks.  This function is
               particularly useful when a `BaseFab' is used as a temporary
               space which must be a different size whenever it is used.
               Resize()ing a temp will often be faster than re-allocating a
               `BaseFab' because memory allocation can often be avoided.
    */
  void resize (const Box& b,
	       int        N = 1);

  /* @ManDoc: Make BaseFab with desired domain and number of components.
		 existing data is lost.  Data is in uninialized state.
		 */
  virtual void define(const Box& box, int comps) { resize(box, comps);}

  /*@ManDoc: The function returns the `BaseFab' to the invalid state.  (See
               comments for constructors above.)  The memory is freed.
    */
  void clear ();


/// {\bf accessors}

///
  /**
  Returns the number of components.
  */
  int nComp () const;

///
  /**
  Returns the domain (box) where the array is defined.
  */
  const Box& box () const;

  /*@ManDoc: Returns a pointer to an array of SpaceDim integers
               giving the length of the domain in each direction.
    */
  const int* size () const;

  /*@ManDoc: Returns the lower corner of the domain.  See
               class `Box' for analogue.
    */
  const IntVect& smallEnd () const;
 
  /*@ManDoc:  Returns the upper corner of the domain.  See
                class `Box' for analogue.
    */
  const IntVect& bigEnd () const;

///
/**
   Returns an Interval for the entire range on components.
*/
  Interval interval() const
    { return Interval(0, nvar-1); }

  /*@ManDoc: Returns a modifiable lvalue reference to the Nth component value
               defined at position p in the domain.  This operator may be
               inefficient if the C++ compiler is unable to optimize the
               C++ code.
    */
  T& operator() (const IntVect& p,
		 int            N);

  T& operator() (const IntVect& p);

  /*@ManDoc: Returns a conatant reference to the Nth component value
    defined at position p in the domain.  This operator may be
    inefficient if the C++ compiler is unable to optimize the
    C++ code.
    */
  const T& operator() (const IntVect& p,
		       int            N) const;

  const T& operator() (const IntVect& p) const;

  /*@ManDoc: This function puts numcomp component values, starting at
               component N, from position pos in the domain into array data,
               that must be allocated by the user.
    */
  void getVal (T*             data,
	       const IntVect& pos,
	       int            N,
	       int            numcomp) const;

  /*@ManDoc: This function puts all component values, starting at
               component 0, from position pos in the domain into array data,
               that must be allocated by the user.
    */
  void getVal (T*             data,
	       const IntVect& pos) const;

///{\bf Fortran interface functions}

  /*@ManDoc: Returns the lower corner of the domain.  Instead of
               returning them in the form of IntVects, as in smallEnd
               and bigEnd, it returns the values as a pointer to an
               array of constant integers.  This is useful when
               interfacing to Fortran subroutines.  It should not be
               used in any other context!!!

    */
  const int* loVect () const;
 
  /*@ManDoc: Returns the upper corner of the domain.  Instead of
               returning them in the form of IntVects, as in smallEnd
               and bigEnd, it returns the values as a pointer to an
               array of constant integers.  This is useful when
               interfacing to Fortran subroutines.  It should not be
               used in any other context!!!

    */
  const int* hiVect () const;
 
  /*@ManDoc: Returns a pointer to an integer that contains the 
			   number of components in the BaseFab. This is useful when
               interfacing to Fortran subroutines.  It should not be
               used in any other context!!!

    */
  const int* nCompPtr () const;
 
  /*@ManDoc: Returns a pointer to an object of type T that is the
               value of the Nth component associated with the cell at
               the low end of the domain.  This is commonly used to
               get a pointer to data in the array which is then handed
               off to a Fortran subroutine. It should not be used in
               any other context!!!  Remember that data is stored in
               Fortran array order, with the component index coming
               last.  In other words, `dataPtr' returns a pointer to
               all the Nth components.  

*/
  T* dataPtr (int N = 0);

  /*@ManDoc: Returns a constant pointer to an object of type T that is
               the value of the Nth component associated with the cell
               at the low end of the domain.  This is commonly used to
               get a pointer to data in the array which is then handed
               off to a Fortran subroutine.  It should not be used in
               any other context!!! Remember that data is stored in
               Fortran array order, with the component index coming
               last.  In other words, `dataPtr' returns a pointer to
               all the Nth components.  

*/
  const T* dataPtr (int N = 0) const;

///{\bf comparison functions}

  /*@ManDoc: Returns true if the domain of fab is totally contained within
               the domain of this `BaseFab'.
    */
  bool contains (const BaseFab<T>& fab) const;
 
  /*@ManDoc: Returns true if bx is totally contained
               within the domain of this `BaseFab'.
    */
  bool contains (const Box& bx) const;

///{\bf data modification functions}

  /*@ManDoc: The setVal functions set subregions in the `BaseFab' to a
               constant value.  This most general form specifies the sub-box,
               the starting component number, and the number of components
               to be set.
    */
  void setVal (T          x,
	       const Box& bx,
	       int        nstart,
	       int        ncomp);

///
/**
   Modifies this BaseFab so that all values of component N in the
   specified Box bx are set to the given value x.  

*/
  void setVal (T          x,
	       const Box& bx,
	       int        N);

///
/**
   Modifies this BaseFab so that all values of component N are set to
   the given value x.

*/
  void setVal (T   x,
	       int N);

///
/**
   Modifies this BaseFab so that all values of all components are set to
   the given value x.

*/
  void setVal (T x);


  /*@ManDoc: Modifies this BaseFab by coping the contents of the
               argument BaseFab into it.  This, the most general form
               of copy, specifies the contents of any sub-box srcbox
               in `BaseFab' src may be copied into a (possibly
               different) destbox in the destination `BaseFab'.  Note
               that although the srcbox and the destbox may be
               disjoint, they must be the same size and shape.  If the
               sizes differ, the copy is undefined and a runtime error
               results.  This copy function is the only one of the
               copy functions to allow a copy between differing
               boxes. The user also specifies how many components are
               copied, starting at component srccomp in src and stored
               starting at component destcomp.  The results are
               UNDEFINED if the src and dest BaseFabs are the same and
               the srcbox and destbox overlap.

*/
  BaseFab<T>& copy (const BaseFab<T>& src,
		    const Box&        srcbox,
		    int               srccomp,
		    const Box&        destbox,
		    int               destcomp,
		    int               numcomp);

  /*@ManDoc: Modifies this BaseFab by coping the contents of the
             argument BaseFab into it.  A copy within the intersecting
             region of the domains of the two BaseFabs is performed.
             The user specifies how many components are copied,
             starting at component srccomp in src and stored starting
             at component destcomp.

*/
  BaseFab<T>& copy (const BaseFab<T>& src,
		    int               srccomp,
		    int               destcomp,
		    int               numcomp = 1);

  /*@ManDoc: Modifies this BaseFab by coping the contents of the
             argument BaseFab into it.  A copy within the intersecting
             region of the domains of the two BaseFabs and the
             specified Box destbox is performed.  All components are
             copied.

*/
  BaseFab<T>& copy (const BaseFab<T>& src,
		    const Box&        destbox);

  /*@ManDoc: Modifies this BaseFab by coping the contents of the
             argument BaseFab into it.  A copy within the intersecting
             region of the domains of the two BaseFabs is performed.
             All components are copied.

*/
  BaseFab<T>& copy (const BaseFab<T>& src);

///
  /**
    Copy from a subsection of one box into another.  Assumes
    the boxes are both in the same index space, and that box R is completely
    contained in both the source and destination boxes.
    */
  void copy(const Box& R, const Interval& Cdest, const BaseFab<T>& src, 
	    const Interval& Csrc);


///    {\bf domain modification functions}

///
  /**
     Modifies the domain of this BaseFab by shifting.  Equivalent to
     fab.shift(0,iv[0]).shift(1,iv[1]) ....  There is no effect upon
     the array memory.

*/
  BaseFab<T>& shift (const IntVect& v);

///
  /**
     Modifies the domain of this BaseFab by shifting it n_cells
     indexing positions in coordinate direction idir.  Directions are
     zero-based.  It is an error if not 0 <= idir < SpaceDim.There is
     no effect upon the array memory.

*/
  BaseFab<T>& shift (int idir,
		     int n_cell);

///
  /**
     Modifies the domain of this BaseFab by shifting by "half" indices,
     thereby converting the Box from type CELL to NODE or vice-versa.
     fab.shiftHalf(0,1) shifts the domain to the right by 1/2 cells.
     fab.shiftHalf(1,-3) shifts the domain in the -j direction by 3/2 cells.
     NOTE: If num\_halfs is EVEN the shift is num\_halfs/2 full zones
     and hence will not change the type.  This is: fab.shifthalf(4) ==
     fab.shift(2).  Directions are zero-based.  It is an error if not 0
     <= dir < SpaceDim.   There is no
     effect upon the array memory. 

*/
  BaseFab<T>& shiftHalf (int dir,
			 int num_halfs);

///
  /**
    Modifies the domain of this BaseFab by shifting by half indices.
     Equivalent to fab.shiftHalf(0,iv[0]).shiftHalf(1,iv[1]) ...
     There is no effect upon the array memory.

  */
  BaseFab<T>& shiftHalf (const IntVect& num_halfs);

///{\bf linearization functions}

///
  /**
     Returns the size, in number of bytes, of a flat linear
     representation of the data in this object in the area defined
     by the input Box R and the component Interval comps.  The size
     does not include the size of R and comps.
     */
  int size(const Box& b, const Interval& comps) const;

///
  /**
     Write a linear representation of the internal data.  Assumes
     that sufficient memory for the buffer has already been allocated by the
     caller.
    */
  void linearOut(void* buf, const Box& R, const Interval& comps) const;

  void linearIn(void* buf, const Box& R, const Interval& comps);

  static bool preAllocatable(){ return true;}

protected:
  //
  // Allocates memory for the `BaseFab<T>'.
  //
  void define ();
  //
  // Deallocates memory for the `BaseFab<T>'.
  //
  void undefine ();
  //
  // The function called by BaseFab copy operations.
  //
  virtual void performCopy (const BaseFab<T>& src,
		    const Box&        srcbox,
		    int               srccomp,
		    const Box&        destbox,
		    int               destcomp,
		    int               numcomp);

  //
  // This function is called by the `BaseFab' setVal operations.
  //
  void performSetVal (T          x,
		      const Box& bx,
		      int        nstart,
		      int        numcomp);


private:
  //
  // These functions are made private to prevent use of the default 
  // functions provided by the C++ compiler.
  //

  BaseFab<T>& operator= (const BaseFab<T>&);
  BaseFab (const BaseFab<T>&);

protected:

  Box  domain;   // My index space.
  int  nvar;     // Number components.
  long numpts;   // Cached number of points in FAB.
  long truesize; // nvar*numpts that was allocated on heap.
  T*   dptr;     // The data pointer.
};

#include "BaseFabImplem.H"

#endif /*CH_BASEFAB_H*/

