// -*- Mode: C++; c-file-style: "gnu"; Modified: "Thu 30 Dec 1999 16:11:44 by dave"; -*- 
// file: BoxIterator.H

/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/

//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// DTGraves

#ifndef _BOX_ITERATOR_H_
#define _BOX_ITERATOR_H_

#include <cstdlib>

#include "Box.H"
#include "REAL.H"
#include "SPACE.H"
#include "Tuple.H"
#include "IntVect.H"

///iterates through the IntVects of a Box
/**
   BoxIterator iterates through the IntVects of a box.  The actual
   sqeuence of IntVects is implementation-specific.  
   Typical usage:\\

\begin{verbatim}
   Box b; 
   ...
   BoxIterator bit (b); 
   for (bit.begin(); bit.ok(); ++bit)  
   {  
     IntVect iv = bit(); 
     (do operations involving iv) 
   }
\end{verbatim}
*/
class BoxIterator
{
public:

  ///{\bf Constructors, destructors, defines}

  ///
  /**
     Default constructor.  This constructs an invalid iterator.
     The user must call define before using.
  */    
  BoxIterator ()
  {
    icur = 0;
    for(int idir = 0; idir < SpaceDim; idir++)
      {
        nx[idir] = 0;
        len[idir] = 0;
        iloc[idir] = 0;
      }
  }

  ///
  /**
     Constructs a BoxIterator and associates it with a Box.\\
     {\bf Arguments:}\\
     bx (not modified) the Box to iterate over.\\
     {\bf Returns:} none.\\
     {\bf This:}\\
     This object is modified.
  */
  BoxIterator(const Box& bx)
  {
    define(bx);
  }

  void setBox(const Box& bx)
  {
    define(bx);
  }
  ///
  /**
     Associates a Box with this BoxIterator.\\
     {\bf Arguments:}\\
     bx (not modified) the Box to iterate over. \\
     {\bf Returns:} none.\\
     {\bf This:}\\
     This object is modified.
  */
  void define (const Box& bx)
  {
    icur = 0;
    box_ptr = bx;
    current= bx.smallEnd();
    offset = current;
    int idir = 0;
    for(idir = 0; idir < SpaceDim; idir++)
      {
        nx[idir] = bx.size(idir);
        iloc[idir] = 0;
      }
    len[0] = 1;
    for(idir = 1; idir < SpaceDim; idir++)
      {
        len[idir] = len[idir-1]*nx[idir-1];
      }
  }

  ///
  /**
     Copy constructor.\\
     {\bf Arguments:}\\
     tx (not modified) the BoxIterator to copy.\\
     {\bf Returns:} none.\\
     {\bf This:}\\
     This object is modified.
  */
  BoxIterator (const BoxIterator& tx)
  {
    icur = tx.icur;
    box_ptr = tx.box_ptr;
    current=  tx.current;
    offset = tx.offset;
    for(int idir = 0; idir < SpaceDim; idir++)
      {
        len[idir] = tx.len[idir];
        nx[idir] = tx.nx[idir];
        iloc[idir] = tx.iloc[idir];
      }
  }
    
  ///
  /**
     Destructor.
  */
  ~BoxIterator ()
  {}

  ///{\bf Modification functions}

  ///
  /**
     Sets this BoxIterator to the first IntVect in its Box.  The
     definition of the "first" IntVect is
     implementation-dependent.\\
     {\bf Arguments:} none.\\
     {\bf Returns:} none.\\
     {\bf This:}\\
     This object is modified.
  */
  void begin()
  {
    icur = 0;
    current=  box_ptr.smallEnd();
    for(int idir = 0; idir < SpaceDim; idir++)
      {
        iloc[idir] = 0;
      }
  }

  void reset()
  {
    begin();
  }

  ///
  /**
     Modifies this BoxIterator to set it to the next location in its
     Box.  The definition of the "next location" of a Box is
     implementation-dependent.\\ 
     {\bf Arguments:} none.  \\
     {\bf Returns:} none. \\
     {\bf This:} \\This object is modified.  

*/
  void operator++()
  {
    next();
  }

  void next()
  {
    icur++;
    //compute where in grid we are, then add offset
    //mea culpa, i cannot figure out any easy way
    //to make this dimension-independent
#if (CH_SPACEDIM == 1)
    iloc[0] = icur/len[0];
    IntVect ivdiff(iloc[0]);
#elif (CH_SPACEDIM == 2)
    iloc[1] = icur/len[1];
    iloc[0] = (icur-iloc[1]*len[1])/len[0];
    IntVect ivdiff(iloc[0],iloc[1]);
#elif (CH_SPACEDIM == 3)
    iloc[2] = icur/len[2];
    iloc[1] = (icur-iloc[2]*len[2])/len[1];
    iloc[0] = (icur-iloc[1]*len[1]-iloc[2]*len[2])/len[0];
    IntVect ivdiff(iloc[0],iloc[1],iloc[2]);
#else
#error CH_SPACEDIM must be either 1,2 or 3.
#endif
    ivdiff += offset;
    current = ivdiff;
  }



  ///{\bf Access functions}
   
  ///
  /**
     Returns the value of the InVect for the current location of this BoxIterator.\\
     {\bf Arguments:} none.\\
     {\bf Returns:} \\
     the value of the intvect for the current location.\\
     {\bf This:}\\
     This object is not modified.
  */
  const IntVect& operator() () const
  {
    assert(current <= box_ptr.bigEnd());
    assert(current >= box_ptr.smallEnd());
    return current;
  }

  ///
  /**
     Returns true if this BoxIterator's location is within its Box.\\
     {\bf Arguments:} none.\\
     {\bf Returns:} \\
     true if this BoxIterator's current location is in its Box.\\
     {\bf This:}\\
     This object is not modified.
  */
  bool ok()
  {
    return (current  <= box_ptr.bigEnd());
  }

protected:

  Box box_ptr;
  Tuple<int, SpaceDim> nx;
  Tuple<int, SpaceDim> iloc;
  Tuple<int, SpaceDim> len;
  IntVect current;
  IntVect offset;
  int icur;
};

#endif
