/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/


// BoxLayout.H
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// ============

#ifndef BOXLAYOUT_H
#define BOXLAYOUT_H



#include "Box.H"
#include "Vector.H"
class DataIterator;
class LayoutIterator;
#include "RefCountedPtr.H"
#include "DataIndex.H"
#include "SPMD.H"


// This header file defines the BoxLayout oject.

///A not-necessarily disjoint collective of boxes
/**
  A BoxLayout is a collection of Box objects that are assigned
  to a process number.  Each box is associated with only one
  process.  processes are numbered from 0 to n-1 (for an n process
  parallel job)

  BoxLayout can be either open or closed.  null construction or deepcopy
  creates an open BoxLayout.  Boxes may be added to an open BoxLayout. 
  non-const operations may be performed on the boxes existing in an
  open BoxLayout.  closed BoxLayouts are sorted.  Many uses of BoxLayouts
  require a closed BoxLayout.  A closed BoxLayout cannot be modified.

  Ref-counting
  ------------
  BoxLayout is an explicitly ref-counted object.

  Assignment and copy are compiler generated.  They increment the refcount
  on the contained data members.  They perform shallow, ref-counted
  operations.

  Refcounting is a process whereby multiple instaniations make
  use of a single implementation of that object and keep a tally
  of how many instantiations are sharing.  Thus

  \begin{verbatim}
  BoxLayout b1(boxes, procIDs);

            b1  ----> refcount = 1
                ---->  m_boxes
		---->  m_processors

  BoxLayout b2(b1)

            b1  ----> refcount = 2  <---- b2
                ---->  m_boxes      <----
		---->  m_processors <----

  BoxLayout b3;
  b3 = b2;

            b1  ----> refcount = 3  <---- b2
                ---->  m_boxes      <----
		---->  m_processors <----
                        ^^^
                        |||
                         b3
  \end{verbatim}
*/
class BoxLayout
{
public:
  ///
  BoxLayout();

  ///
  /** Construct from a Vector of Boxes and  a Vector of
      processor assignments.  On exit, the BoxLayout will be closed.
  */
  BoxLayout(const Vector<Box>& a_boxes,
            const Vector<int>& a_procIDs);

  ///
  /**
    Ref-counted destruction.  Once the last reference to the 
    implementation of this class is destroyed, the data members
    are cleaned up
  */
  virtual
  ~BoxLayout();

  /// refcounted assignment
  /**
   */
  BoxLayout& operator=(const BoxLayout& a_rhs);

  ///
  /** Define this BoxLayout from a Vector of Boxes and a 
      Vector of processor assignments.  Any pre-existing layout will
      be lost (ref-count dropped by one).  The processor assignment Vector
      must be the same length
      as the Box Vector.  On exit, the BoxLayout will be closed.  
  */
  virtual void
  define(const Vector<Box>& a_boxes,
         const Vector<int>& a_procIDs);


  ///
  /** const accessor operator.  see also {\tt get(const DataIndex&)}
   */
  const Box&
  operator[](const LayoutIndex& it) const;

  ///
  Box&
  operator[](const LayoutIndex& it);

  ///
  /**  Refcounted pointer check.  True if these two objects
    share the same implementation.
    */
  inline bool
  operator==(const BoxLayout& rhs) const ;

  
  // not a user function
  bool check(const LayoutIndex& index) const
    { return index.m_layoutIntPtr == m_layout;}


  ///
  /** add a box to this BoxLayout. Takes the processor
    assignment. The input 'box' is copied.  BoxLayout must be 'open'
    for this operation, otherwise a runtime error will occur.
      */
  DataIndex
  addBox(const Box& box, int procID);

  ///
  /** Mark this BoxLayout as complete and unchangeable.
   */
  virtual void
  close();

  ///
  /** Actual deep copy operation.  New object created with copied
      data.  This object disassociates itself with original implementation
      safely.  This object now is considered 'open' and can be non-const
      modified.  There is no assurance that the order in which this BoxLayout
      is indexed corresponds to the indexing of a_source.
      
       BoxLayout b1(boxes, procIDs);
\begin{verbatim}
            b1  ----> refcount = 1
                ---->  m_boxes
		---->  m_processors

       BoxLayout b2(b1)

            b1  ----> refcount = 2  <---- b2
                ---->  m_boxes      <----
		---->  m_processors <----

       BoxLayout b3;
       b3.deepCopy(b2);
            b1  ----> refcount = 2  <---- b2  b3 ----> refcount = 1 
                ---->  m_boxes      <----        ---->  m_boxes
		---->  m_processors <----        ---->  m_processors
\end{verbatim}
  */ 
  virtual void
  deepCopy(const BoxLayout& a_source);

  ///
  /** As a consquence of the C++ compiler being free to choose which
      version of operator[] when the object is technically non-const, we very
      often get 'BoxLayout closed' errors.  This is a non-overloaded get method.
  */
  const Box&
  get(const LayoutIndex& it) const;


  /** return true if close() has been called.  closed BoxLayout is always sorted.
   */
  bool
  isClosed() const;


  ///
  /** return true if sort() has been called
   */
  bool
  isSorted() const;

  ///
  /* parallel iterator.  Returns DataIndex objects that that only correspond to boxes
     on the local processor */
  DataIterator
  dataIterator() const;

  ///
  /** If BoxLayout is closed, then LayoutIterator will return the Box's in sorted
    order.  Processes through ALL the boxes in a BoxLayout */
  LayoutIterator
  layoutIterator() const;
  
  /// invokes cout<<*this;  pretty-print dump of BoxLayout
  void
  print() const;

  ///
  /** Returns the processor to which this box has been assigned.
      Not a user function, at least, not a new user function.  It can
      be used safely at anytime, closed or open.  A person needing this
      level of knowledge of the processor assignment should have non-trivial
      needs, like writing your own load balancer or such.  Most user level parallel
      coding needs can be addressed using the DataIterator class.
  */
  unsigned int
  procID(const LayoutIndex& a_index) const ;

  ///
  /* returns the number of boxes assigned to a given procID
   */
  int numBoxes(const int procID) const;

  ///
  /** Assign a Box in the BoxLayout to a processor.
      Requires the BoxLayout to be open.
  */
  void
  setProcID(const LayoutIndex& a_index, unsigned int a_procID);

  ///
  inline unsigned int
  size() const ;


  //const or non-const operation ?.....I can think of usages either way...bvs
  ///
  /**
    Sort the boxes.  
    */
  void
  sort();

  ///
  /**  'output' must be open, 'input' must be closed, 'refinement' must be
    a positive non-zero integer.  'output' and 'input' do not share an
    implementation.  'output' is first deepcopy'ed from 'input', 
    then coarsen(refinement) is called on each box of 'output'.  'output'
    returns from this function closed.  LayoutIterators from 'input' and 'output'
    can perform interchangeably. */
  friend void coarsen(BoxLayout& output, const BoxLayout& input, int refinement);

  ///
  /** see 'coarsen(BoxLayout& output, const BoxLayout& input, int refinement) '
    and substitute the word "refine" for "coarsen" */
  friend void refine(BoxLayout& output, const BoxLayout& input, int refinement);

  struct Entry
  {
    Entry(): m_procID(::procID()){;}
    Entry(const Box& a_box)
      :box(a_box), m_procID(::procID()){}
    Entry(const Box& a_box, const unsigned int a_index)
      :box(a_box), index(a_index), m_procID(::procID()){}
    bool operator < (const Entry& rhs) const
      { return box < rhs.box;}
    
    Box box;
    unsigned int index;
    unsigned int m_procID;// not used in serial code.
  };
  
  // not a user function.  used in IO routine.
  unsigned int index(const LayoutIndex& index) const;

protected:
	
  void setIndexVector();
  friend class LayoutIterator;
  friend class DataIterator;

  RefCountedPtr<std::vector<Entry> >        m_boxes;
  RefCountedPtr<std::vector<unsigned int> > m_index;
  RefCountedPtr<int>                        m_layout;
  RefCountedPtr<bool>                       m_closed;
  RefCountedPtr<DataIterator>               m_dataIterator;

private:

};


//========================================================


// inlined functions
// =================


// constructors
// ============



/*
inline
BoxLayout::BoxLayout(int a_boxes)
  :m_boxes(new std::vector<Entry>(a_boxes)),
   m_index(new std::vector<unsigned int>(a_boxes)),
   m_layout(new int),
   m_closed(new bool(false))
{
  std::vector<Entry>& boxes = *m_boxes;
  std::vector<unsigned int>& index = *m_index;
  for(unsigned int i=0; i<boxes.size(); ++i)
    {
      boxes[i].index = i;
      index[i] = i;
    }
}
*/

// operators
// =========

inline const Box&
BoxLayout::operator[](const LayoutIndex& index) const
{
  assert(check(index));// make sure this LayoutIndex came from my own iterator
  return m_boxes->operator[](m_index->operator[](index.m_index)).box;
}

inline Box&
BoxLayout::operator[](const LayoutIndex& index)
{
  if(*m_closed)
    {
      MayDay::Error("attempt to modify closed BoxLayout");
    }
  assert(check(index)); // make sure this LayoutIndex came from my own iterator
  return m_boxes->operator[](m_index->operator[](index.m_index)).box;
}

inline bool
BoxLayout::operator==(const BoxLayout& rhs) const
{
  return m_boxes == rhs.m_boxes;
}


// member functions
// ================


inline const Box&
BoxLayout::get(const LayoutIndex& index) const
{
  assert(check(index)); // make sure this LayoutIndex came from my own iterator
  return m_boxes->operator[](m_index->operator[](index.m_index)).box;
}

inline unsigned int
BoxLayout::index(const LayoutIndex& a_index) const
{
  return m_index->operator[](a_index.m_index);
}

inline bool
BoxLayout::isClosed() const
{
  return *m_closed;
}

inline unsigned int
BoxLayout::procID(const LayoutIndex& a_index) const
{
  assert(check(a_index));
  return m_boxes->operator[](m_index->operator[](a_index.m_index)).m_procID;
}

inline void
BoxLayout::setProcID(const LayoutIndex& a_index, unsigned int a_procID)
{
  assert(check(a_index));
  m_boxes->operator[](m_index->operator[](a_index.m_index)).m_procID = a_procID;
}

inline unsigned int
BoxLayout::size() const 
{
  return m_boxes->size();
}


// global functions
// ================

std::ostream& operator<<(std::ostream& os, const BoxLayout& a_layout);




#endif // DATAPLAN_H

