//BoxLayout.cpp
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//

#include <iostream>
#include <algorithm>
#include "BoxLayout.H"
#include "DataIterator.H"
#include "LayoutIterator.H"

using std::ostream;

//need at least one non-inlined function, otherwise
//some compilers don't build a class description in the
//object file.

BoxLayout::~BoxLayout()
{}

BoxLayout::BoxLayout()
  :m_boxes(new std::vector<Entry>()),
   m_index(new std::vector<unsigned int>()),
   
   m_layout(new int),
   m_closed(new bool(false)),
   m_dataIterator(NULL)
{}

BoxLayout& BoxLayout::operator=(const BoxLayout& a_rhs)
{
  if(this == &a_rhs) return *this;
  m_boxes = a_rhs.m_boxes;
  m_index = a_rhs.m_index;
  m_layout = a_rhs.m_layout;
  m_closed = a_rhs.m_closed;
  m_dataIterator = a_rhs.m_dataIterator;
  return *this;
}


void BoxLayout::sort()
{
  if(!*m_closed)
    {
      std::sort(m_boxes->begin(), m_boxes->end());
      setIndexVector();
    }
}

void BoxLayout::close()
{
  if(!*m_closed){
    sort();
    *m_closed = true;
	m_dataIterator = new DataIterator(*this, m_layout);
  }
}

void BoxLayout::setIndexVector()
{
  std::vector<Entry>& boxes = *m_boxes;
  std::vector<unsigned int>& index = *m_index;
  index.resize(boxes.size());
  for(unsigned int i=0; i<boxes.size(); ++i)
    {
      index[boxes[i].index] = i;
    }
}

// Constructors and such
// =====================

DataIterator BoxLayout::dataIterator() const
{
  assert(*m_closed);
  //DataIterator rtn(*this, m_layout);
  return *m_dataIterator;
  //return rtn;
}

LayoutIterator BoxLayout::layoutIterator() const
{
  return LayoutIterator(*this, m_layout);
}


BoxLayout::BoxLayout(const Vector<Box>& a_boxes, const Vector<int>& assignments)
  :m_boxes(new std::vector<Entry>()),
   m_index(new std::vector<unsigned int>()),
   m_layout(new int),
   m_closed(new bool(false))
{
  define(a_boxes, assignments);
}

void
BoxLayout::define(const Vector<Box>& a_boxes, const Vector<int>& a_procIDs)
{
  if(*m_closed)
    {
      MayDay::Error("attempt to define(..) a closed BoxLayout");
    }
  const int num_boxes = a_boxes.size();
  const int num_procs = a_procIDs.size();
  if( num_procs > 0 && num_boxes != num_procs )
    {
      MayDay::Error("BoxLayout::define(): vector of processor assignments is different length from vector of boxes");
    }
  m_boxes->resize(num_boxes);
  for (unsigned int i = 0; i < num_boxes; ++i)
    {
      m_boxes->operator[](i) = a_boxes[i];
      m_boxes->operator[](i).index = i;
      if( num_procs > 0 ) m_boxes->operator[](i).m_procID = a_procIDs[i];
    }
  close();
}


// Other member functions
// ======================

DataIndex
BoxLayout::addBox(const Box& box, int procID)
{
  if(*m_closed)
    {
      MayDay::Error("attempt to addBox to closed BoxLayout");
    }
  if(m_layout.isNonUnique())
    {
      m_layout = new int;
    }
  std::vector<Entry>& boxes = *m_boxes;
  unsigned int i = boxes.size();
  boxes.push_back(box);
  boxes[i].index = i;
  boxes[i].m_procID = procID;
  (*m_index).push_back(i);
  return DataIndex(i, m_layout);
}


void
BoxLayout::deepCopy(const BoxLayout& a_source)
{
  m_boxes =  new std::vector<Entry>(*(a_source.m_boxes));
  m_index =  new std::vector<unsigned int>(*(a_source.m_index));
  m_layout = a_source.m_layout;
  *m_closed = false;
}


/*
void
BoxLayout::resize(unsigned int n, const Box& box) 
{
  if(*m_closed)
    {
      MayDay::Error("attempt to resize closed BoxLayout");
    }
  if(n == m_boxes->size()) return;
  if(m_layout.isNonUnique())
    {
      m_layout = new int;
    }
  unsigned int size = m_boxes->size();
  m_boxes->resize(n, box);
  m_index->resize(n);
  if(n > size)
    {
      for(unsigned int i=size; i<n; ++i)
        m_boxes->operator[](i).index = i;
    }
  setIndexVector();
}

*/



// Global functions
// ================

// For now, we can just have the one coarsen funtion.  If a DisjointBoxLayout
// enters this function, is coarsened, and then doesn't remain disjoint, it
// will be caught here at the call to close().  Debugging should not be

void
coarsen(BoxLayout& a_output, const BoxLayout& a_input, int a_refinement)
{
   if(!a_input.isClosed())
    {
      MayDay::Error("input to coarsen must be called with closed BoxLayout");
    }
  if(a_output.isClosed())
    {
      MayDay::Error("output of coarsen must be called on open BoxLayout");
    }
  a_output.deepCopy(a_input);


  for(LayoutIterator it(a_input.layoutIterator()); it.ok(); ++it)
    {
      a_output[it()].coarsen(a_refinement);
    }
  a_output.close();
}

// we have an easier time with refine, since we know that refinement will
// not change the state of a sort, but, we will play it safe for now
// until this function shows up in the profiler.

void refine(BoxLayout& a_output, const BoxLayout& a_input, int a_refinement)
{
  if(!a_input.isClosed())
    {
      MayDay::Error("input to refine must be called with closed BoxLayout");
    }
  if(a_output.isClosed())
    {
      MayDay::Error("output of refine must be called on open BoxLayout");
    }
  a_output.deepCopy(a_input);
  
  for(LayoutIterator it(a_input.layoutIterator()); it.ok(); ++it)
    {
      a_output[it()].refine(a_refinement);
    }
  a_output.close();
}

ostream& operator<<(ostream& os, const BoxLayout& a_layout)
{
  int i=0;
  for(LayoutIterator it(a_layout.layoutIterator()); it.ok(); ++it)
    {
      os << a_layout.get(it())<<"["<<a_layout.procID(it())<<"]";
      ++i;
      if(i==4){ os <<"\n"; i=0;}
      else {os <<" # ";}
    }

  os <<"\n";
  return os;
}


void BoxLayout::print() const { std::cout << *this;}
	
int BoxLayout::numBoxes(const int procID) const
{
  int num = 0;
  for(int i=0; i<m_boxes->size(); ++i)
    {
      if(m_boxes->operator[](i).m_procID == procID) ++num;
    }
  return num;
}


