/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// BoxLayoutData.H
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//

#ifndef BOXLAYOUTDATA_H
#define BOXLAYOUTDATA_H

#include "LayoutData.H"
#include "Interval.H"
#include "FArrayBox.H"

/// Factory object to data members of a BoxLayoutData container
template <class T> class DataFactory
{
public:
  /// factory function.  creates a new 'T' object
  /** creates a new 'T' object and returns a pointer to it.  Responsiblitly
	  for calling operator 'delete' on this pointer is passed to the user. */
  virtual T* create(const Box& box, int ncomps) const;
  
};


template<class T> class LevelData;


/// Data on a BoxLayout
/**

  A collection of Box-oriented objects.  The arrangement
  of Boxes is given by the underlying BoxLayout object.
  BoxLayoutData attempt to prevent users from maniupulating
  the template class T to become out of synch with the
  boxes in the BoxLayout.  Caveat emptor.
  
  All access to the data in a BoxLayoutData is forced to be
  data-parallel, by use of the DataIterator.

  class T must provide the following methods:
\begin{itemize}
    \item {\tt T()}
    \item {\tt T(const Box& box, int comps)}
    \item {\tt define(const Box& box, int comps)}
    \item {\tt void copy(const Box& R, const Interval& Cd, const T& source, const Interval Cs)}
       Cs and Cd must have same length, but can be displaced
  
    \item {\tt static bool preAllocatable()} returns true if the size(...) function is strictly
                                a function of Box and Interval, and does not depend
                                on the current state of the T object.
    \item {\tt int size(const Box& R, const Interval& comps)}
    \item {\tt void linearOut(void* buf, const Box& R, const Interval& comps)}
    \item {\tt void linearIn(void* buf, const Box& R, const Interval& comps)}
  \end{itemize}

  In sequential codes, many of these functions will not be called, but in the
  general parallel case, they all will be.
 
*/

template<class T> 
class BoxLayoutData : public LayoutData<T>
{
public:
  ///
  BoxLayoutData():m_comps(0) {m_isdefined = false;}
 
  virtual ~BoxLayoutData(){;}
  ///
  BoxLayoutData(const BoxLayout& boxes, int comps, 
				const DataFactory<T>& factory = DataFactory<T>());
  
  ///
  virtual void define(const BoxLayout& boxes, int comps, 
					  const DataFactory<T>& factory = DataFactory<T>());
  
  ///
  virtual void define(const BoxLayoutData<T>& da,
					  const DataFactory<T>& factory = DataFactory<T>());
  
  ///
  /** Define this BoxLayoutData to be the subset of the
	  data in da defined by the Interval comps.
  */
  virtual void define(const BoxLayoutData<T>& da, const Interval& comps,
					  const DataFactory<T>& factory = DataFactory<T>());

  /// overridden and throws an error.  
  virtual void define(const BoxLayout& boxes);
  
  ///
  int nComp() const { return m_comps;}

  ///
  Interval interval() const
  {
    Interval outint(0, m_comps-1);
    return(outint);
  }


    ///
  /* User writes a function with the signature:
     \begin{verbatim}
     void setFunction(const Box& box, int comps, T& t){ your code here;}
     \end{verbatime}
     They can then hand this off to LayoutData::setVal.  This class
     then cycles through all the T objects and invokes this function.  Function
     must not be inline. (I'm still trying to figure out a nice way to send
     in non-static member functions).
     */
  void setVal(void (*setFunction)(const Box& box, int comps, T& t));


  ///
  virtual bool isDefined() const;
protected:

  int             m_comps;
  bool            m_isdefined;

  friend class LevelData<T>;
  void setVector(const BoxLayoutData<T>& da, 
		 const Interval& srcComps,
		 const Interval& destComps);
  void allocateGhostVector(const DataFactory<T>& factory, 
						   const IntVect& ghost = IntVect::TheZeroVector());
};

///   
/** not actually L-p norms, since it doesn't take into
  account the dx of the system.  A user can take that
  into account or not.
  

  returns: (sum[ | A[i][j,interval] |^p )^(1/p)
              p>=2, for all i in A, for valid regions j in A[i]

           global max( A[i][j, interval] )  p==0
  
  to turn into an L-p norm, one needs to multiply by dx^(SpaceDim/p)

  some people don't like that this norm is not normalized based on
  number of points in A.  Normalization is your problem.
  */
Real norm(const BoxLayoutData<FArrayBox>& A, 
	  const Interval& interval,
	  const int& p = 2);


//======================================================================
template < >
BaseFab<int>* DataFactory<BaseFab<int> >::create(const Box& box, int ncomps) const;

template < >
FArrayBox* DataFactory<FArrayBox>::create(const Box& box, int ncomps) const;

template <class T>
T* DataFactory<T>::create(const Box& box, int ncomps) const
{
  return new T(box, ncomps);
}


template<class T>
inline bool BoxLayoutData<T>::isDefined() const
{
  return m_isdefined;
}

template <class T>
inline void BoxLayoutData<T>::setVector(const BoxLayoutData<T>& da, 
					const Interval& srcComps,
					const Interval& destComps)
{
  if(&da != this)
    {
      for(DataIterator it(dataIterator()); it.ok(); ++it)
	{
	  m_vector[m_boxLayout.index(it())]->copy(box(it()) , destComps, da[it()], srcComps);
	}
    }
}

template<class T>
inline void BoxLayoutData<T>::define(const BoxLayoutData<T>& da, const Interval& comps,
									 const DataFactory<T>& factory)
{
  if(this == &da){
    MayDay::Error("BoxLayoutData<T>::define(const LayoutData<T>& da,.....) called with 'this'");
  }
  assert(comps.size()>0);
  assert(comps.end()<=m_comps);
  assert(comps.begin()>=0);
  m_boxLayout = da.boxLayout();
	
  m_comps = comps.size();

  Interval dest(0, m_comps-1);
  allocateGhostVector(factory);
  setVector(da, comps, dest);

}

template<class T>
inline void BoxLayoutData<T>::define(const BoxLayout& boxes, int comps,
									 const DataFactory<T>& factory)
{
  assert(boxes.isClosed());
  m_boxLayout = boxes;
  m_comps = comps;
  m_isdefined = true;
  allocateGhostVector(factory);
}

template<class T>
inline void BoxLayoutData<T>::define(const BoxLayout& boxes)
{
  MayDay::Error("BoxLayoutData<T>::define(const BoxLayout& boxes)...needs comps");
}

template<class T>
inline BoxLayoutData<T>::BoxLayoutData(const BoxLayout& boxes, int comps,
									   const DataFactory<T>& factory)
  :m_comps(comps)
{
  assert(boxes.isClosed());
  m_boxLayout = boxes;
  m_isdefined = true;
  allocateGhostVector(factory);
}

template<class T>
inline void BoxLayoutData<T>::define(const BoxLayoutData<T>& da,
									 const DataFactory<T>& factory)
{
  if(this != &da){
    m_isdefined = da.m_isdefined;
    m_boxLayout = da.boxLayout();
    m_comps    = da.nComp();
    Interval srcAnddest(0, m_comps-1);
    allocateGhostVector(factory);
    setVector(da, srcAnddest, srcAnddest);
  }
}


template<class T>
inline void BoxLayoutData<T>::allocateGhostVector(const DataFactory<T>& factory, const IntVect& ghost)
{
  for(unsigned int i=0; i<m_vector.size(); ++i)
    {
      delete m_vector[i];
      m_vector[i] = NULL;
    }
		
  m_vector.resize(m_boxLayout.size(), NULL);

  for(DataIterator it(dataIterator()); it.ok(); ++it)
    {
      unsigned int index = m_boxLayout.index(it());
      Box abox = box(it());
      abox.grow(ghost);
      m_vector[index] = factory.create(abox, m_comps);
      if(m_vector[index] == NULL){
	     MayDay::Error("OutOfMemory in boxlayoutdata::allocate");
      }
    }
}

template<class T>
inline void BoxLayoutData<T>::setVal(void (*a_setVal)(const Box& box, int comps, T& t))
{
  for(DataIterator it(dataIterator()); it.ok(); ++it)
    {
      unsigned int index = m_boxLayout.index(it());
      a_setVal(box(it()), m_comps, *(m_vector[index]));
    }
}

  

//======================================================================

#endif // BOXLAYOUTDATA_H
