/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// Copier.H
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
#ifndef COPIER_H
#define COPIER_H

#include "DisjointBoxLayout.H"
#include <vector>

class MotionItem;

class CopyIterator;


/// A strange but true thing to make copying from one boxlayoutdata to another fast
/**
  class to handle the of data from a LevelData to a BoxLayoutData
  defined over the same index space.  The idea beind this object
  is twofold:
  A)  there is a significant amount of information that can be
  computed once and reused from one copy to the next for
  a LevelData and BoxLayoutData that have unchanging BoxLayouts
  underneath them.  In these cases, LevelData::copyTo methods
  has an optional argument.  That way, in cases where the operation
  is only performed a single time, we do not waste our time building
  optimizing data, etc.
  B)  when we interface with KeLP, this class maps quite well
  with the existing KeLP API.

  pains have been taken to not have this class be templated.  Thus
  it does not actually perform the copying itself (which would require
  knowledge of the template class type).  It provides an iterator
  interface to MotionItems that specify the required work.  LevelData
  provides the copyTo methods.
*/

class Copier
{
public:

  ///null constructor, copy constructor and operator= can be compiler defined.
  Copier(){;}
  ///
  Copier(const DisjointBoxLayout& a_level, const BoxLayout& a_dest);

  /// Copier to copy data into the valid and invalid regions of a_dest from a_level
  Copier(const DisjointBoxLayout& a_level, 
	 const BoxLayout& a_dest, 
	 const IntVect& a_destGhost);

  ///
  virtual ~Copier();

  ///
  virtual void define(const DisjointBoxLayout& a_level, const BoxLayout& a_dest);

  ///
  virtual void define(const DisjointBoxLayout& a_level, 
		      const BoxLayout& a_dest,
		      const IntVect& a_destGhost);
  ///
  virtual void clear();

  /// check that this Copier object corresponds to these two BoxLayouts.
  bool check(const DisjointBoxLayout& from, const BoxLayout& to) const;

  int print() const;

protected:

  friend class CopyIterator;

  std::vector<MotionItem*> m_localMotionPlan;
  std::vector<MotionItem*> m_fromMotionPlan;
  std::vector<MotionItem*> m_toMotionPlan;


private:

  // keep a refcounted reference around for debugging purposes, we can
  // decide afterwards if we want to eliminate it.
  DisjointBoxLayout m_originPlan;
  BoxLayout  m_dest;

};

std::ostream& operator<<(std::ostream& os, const Copier& copier);

//===========================================================================
// end of public interface for Copier.  
//===========================================================================


//  These classes are public because I can't find a nice
//  way to make a class a friend of all the instantiations
//  of a template class.  These classes are not part of
//  the public interface for the Array API.  
//
//  Later, if MotionItem shows up in the profiler, we
//  can start using a pool allocation scheme and placement new

class MotionItem{
public:
  DataIndex fromIndex, toIndex;
  Box region;
  int procID;
  MotionItem(const DataIndex& a_from, 
	     const DataIndex& a_to,
	     const Box&       a_region);
};

inline MotionItem::MotionItem(const DataIndex& a_from, 
			      const DataIndex& a_to,
			      const Box&       a_region)
  :fromIndex(a_from), toIndex(a_to), region(a_region), procID(-1)
{;}



class CopyIterator
{
public:
  enum local_from_to {LOCAL, FROM, TO};

  inline CopyIterator(const Copier& a_copier, local_from_to);

  inline const MotionItem& operator()() const;

  inline void operator++();

  inline bool ok() const;

  inline void reset();
private:
  const std::vector<MotionItem*>* m_motionplanPtr;
  unsigned int  m_current;
};

//====== inlined functions====================================

inline CopyIterator::CopyIterator(const Copier& a_copier, local_from_to type)
  :m_current(0)
{
  switch(type){
  case LOCAL:
    m_motionplanPtr = &(a_copier.m_localMotionPlan);
    break;
  case FROM:
    m_motionplanPtr = &(a_copier.m_fromMotionPlan);
    break;
  case TO:
    m_motionplanPtr = &(a_copier.m_toMotionPlan);
    break;
  default:
    MayDay::Error("illegal local_from_to option for CopyIterator");
  }
}

inline const MotionItem& CopyIterator::operator()() const
{
  assert(m_current < m_motionplanPtr->size());
  return *(m_motionplanPtr->operator[](m_current));
}

inline void CopyIterator::operator++() {++m_current;}

inline bool CopyIterator::ok() const
{
  return m_current < m_motionplanPtr->size();
}

inline void CopyIterator::reset()
{
  m_current = 0;
}




#endif 
